package com.gogirl.interfaces.order.serve;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.gogirl.application.common.xcx.GogirlTokenService;
import com.gogirl.application.order.serve.OrderManageService;
import com.gogirl.application.order.serve.OrderRecordService;
import com.gogirl.application.order.serve.OrderServeService;
import com.gogirl.application.order.serve.ScheduleManageService;
import com.gogirl.assembler.OrderManageDTOAssembler;
import com.gogirl.domain.common.xcx.GogirlToken;
import com.gogirl.domain.order.serve.OrderComment;
import com.gogirl.domain.order.serve.OrderManage;
import com.gogirl.domain.order.serve.OrderServe;
import com.gogirl.domain.order.serve.OrderServeDescribe;
import com.gogirl.domain.product.purchase.PurchaseSku;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.common.util.StringUtils;
import com.gogirl.infrastructure.mapper.order.serve.OrderCommentMapper;
import com.gogirl.infrastructure.mapper.order.serve.OrderServeDescribeMapper;
import com.gogirl.infrastructure.mapper.product.purchase.PurchaseSkuMapper;
import com.gogirl.infrastructure.util.SessionUtils;
import com.gogirl.infrastructure.util.lock.OrderIdLock;
import com.gogirl.shared.order.ListOrderQuery;
import com.gogirl.shared.order.OrderManageDTO;
import com.gogirl.shared.order.OrderPhotoResp;
import com.gogirl.shared.order.RecordOrderDataRequest;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.web.bind.annotation.*;

import java.security.InvalidParameterException;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Optional;

@Api(tags = "预约订单接口")
@RestController
@AllArgsConstructor
@Slf4j
public class OrderManageController {

    /**
     * domain service
     */
    private final GogirlTokenService gogirlTokenService;

    /**
     * domain service
     */
    private final OrderManageService orderManageService;

    private final OrderServeService orderServeService;


    /**
     * repository
     */
    private final OrderServeDescribeMapper orderServeDescribeMapper;
    private final PurchaseSkuMapper purchaseSkuMapper;

    private final OrderCommentMapper orderCommentMapper;
    private final OrderManageDTOAssembler orderManageDTOAssembler;
    private final ScheduleManageService scheduleManageService;
    private final OrderRecordService orderRecordService;


    @ApiOperation(value = "查询我的客照")
    @GetMapping(value = "/customer/ordermanage/queryMyOrderPhoto")
    public JsonResult<Page<OrderPhotoResp>> queryMyOrderPhoto(@RequestHeader String token,
                                                              @RequestParam Integer pageNum,
                                                              @RequestParam Integer pageSize) {
        Page<OrderPhotoResp> orderPhotoRespPage = orderRecordService.queryMyOrderPhoto(pageNum, pageSize);
        return JsonResult.success(orderPhotoRespPage);
    }

    @ApiOperation(value = "查询我的订单")
    @GetMapping("/customer/ordermanage/queryMyOrderForPage")
    public JsonResult<IPage<OrderManageDTO>> queryMyOrderForPage(@RequestHeader String token,
                                                                 @RequestParam Integer pageNum,
                                                                 @RequestParam Integer pageSize) {

        Integer currentCustomerId = Optional.ofNullable(gogirlTokenService.getByToken(token).getCustomerId())
                .orElseThrow(() -> new RRException(500, "token不存在"));

        ListOrderQuery qry = new ListOrderQuery();
        qry.setCustomerId(currentCustomerId);
        qry.setPageNum(pageNum);
        qry.setPageSize(pageSize);

        //不显示未核销订单
        qry.setShowUnCheck(false);

        Page<OrderManageDTO> orderManageDTOPage = (Page<OrderManageDTO>) orderManageService.queryListOrder(qry).convert(orderManageDTOAssembler);
        return JsonResult.success(orderManageDTOPage);
    }

    @ApiOperation(value = "查询订单")
    @GetMapping(value = "/customer/ordermanage/queryOrderForDetail")
    public JsonResult<OrderManageDTO> customerQueryOrderForDetail(@RequestParam Integer id) {
        OrderManage orderManage = orderManageService.queryOrder(id);
        return JsonResult.success(orderManageDTOAssembler.apply(orderManage));
    }


    @ApiOperation(value = "查询订单")
    @GetMapping(value = "/customer/ordermanage/queryOrderForComment")
    public JsonResult<OrderManage> queryOrderForComment(@RequestParam Integer id) {
        try {
            //外层兜底处理查询历史订单
            OrderManage orderManage = orderManageService.queryOrderForComment(id);
            return JsonResult.success(orderManage);
        } catch (Exception e) {
            throw new RRException("您已经评价过该订单");
        }
    }


    @ApiOperation(value = "查询所有可选的描述")
    @GetMapping("/technician/ordermanage/listOrderServeDescribe")
    public JsonResult<List<OrderServeDescribe>> listOrderServeDescribe(@RequestParam Integer type) {

        List<OrderServeDescribe> orderServeDescribeList = orderServeDescribeMapper.selectByType(type);
        return JsonResult.success(orderServeDescribeList);
    }

    @ApiOperation("根据skuName查询sku")
    @PostMapping("/technician/ordermanage/selectBySkuName")
    public JsonResult<PurchaseSku> selectBySkuName(String skuName) {
        PurchaseSku purchaseSku = purchaseSkuMapper.selectBySkuName(skuName);
        return JsonResult.success(purchaseSku);
    }

    @ApiOperation("查询款式/色号和款式描述")
    @GetMapping("/technician/ordermanage/getOrderServeDetail")
    public JsonResult<OrderServe> getOrderServeDetail(@RequestParam("orderServeId") Integer orderServeId) {
        OrderServe orderServe = orderServeService.getOrderServeDetail(orderServeId);
        return JsonResult.success(orderServe);
    }

    @ApiOperation("美甲师录入补全订单服务的款式色号、款式描述和客照")
    @PostMapping("/technician/ordermanage/recordOrderServeData")
    public JsonResult<Void> recordOrderServeData(@RequestBody OrderServe param) {
        orderManageService.recordOrderServeData(param);
        return JsonResult.success();
    }

    @ApiOperation(value = "美甲师录入订单资料")
    @PostMapping("/technician/ordermanage/recordOrderData")
    public JsonResult<Void> recordOrderData(@RequestHeader String token,
                                            @RequestBody RecordOrderDataRequest recordOrderDataRequest) {
        orderManageService.recordOrderData(recordOrderDataRequest);
        return JsonResult.success();
    }

    @ApiOperation(value = "查询会员消费记录")
    @GetMapping("/technician/ordermanage/queryCustomerOrder")
    public JsonResult<Map<String, Object>> queryCustomerOrder(@RequestHeader String token,
                                                              @RequestParam Integer customerId,
                                                              @RequestParam Integer pageNum,
                                                              @RequestParam Integer pageSize) {
        Map<String, Object> result = orderManageService.queryCustomerOrder(token, customerId, pageNum, pageSize);
        return JsonResult.success(result);
    }

    @ApiOperation(value = "根据订单id查询订单详情，原接口")
    @GetMapping("/technician/ordermanage/queryOrderForDetail")
    public JsonResult<OrderManageDTO> queryOrderForDetail(@RequestParam Integer id) {
        OrderManage orderManage = orderManageService.queryOrder(id);
        return JsonResult.success(orderManageDTOAssembler.apply(orderManage));
    }

    @ApiOperation(value = "店员查询店铺订单列表")
    @GetMapping("/technician/ordermanage/queryListOrder")
    public JsonResult<Page<OrderManageDTO>> queryListOrder(@RequestHeader String token,
                                                           @RequestParam Integer status,
                                                           @RequestParam Integer pageNum,
                                                           @RequestParam Integer pageSize,
                                                           @RequestParam String customerPhone) {

        GogirlToken gogirlToken = SessionUtils.getTechnicianToken();
        ListOrderQuery qry = ListOrderQuery.builder()
                .pageSize(pageSize).pageNum(pageNum)
                //店铺id
                .departmentId(gogirlToken.getDepartmentId())
                //是否显示未核算订单
                .showUnCheck(true)
                //会员手机号
                .customerPhone(customerPhone)
                //订单状态
                .status(status)
                .build();

        Page<OrderManage> page = (Page<OrderManage>) orderManageService.queryListOrder(qry);
        Page<OrderManageDTO> orderManageDTOPage = (Page<OrderManageDTO>) page.convert(orderManageDTOAssembler);
        return JsonResult.success(orderManageDTOPage);
    }

    @ApiOperation(value = "美甲师接单前确认检查")
    @GetMapping("/technician/ordermanage/checkBeforeTakeOrders")
    public JsonResult<String> checkBeforeTakeOrders(@RequestParam Integer scheduleId) {

        Integer technicianId = SessionUtils.getTechnicianId();
        scheduleManageService.checkBeforeTakeOrders(technicianId, scheduleId);
        return JsonResult.success();
    }

    @ApiOperation("核算订单")
    @GetMapping("/technician/ordermanage/checkOrder")
    public JsonResult<Void> checkOrder(@RequestParam Integer orderId) {
        OrderIdLock orderIdLock = OrderIdLock.getInstance();
        try {
            orderIdLock.lock(orderId);
            orderManageService.checkOrder(orderId);
        } finally {
            orderIdLock.unlock(orderId);
        }
        return JsonResult.success();
    }

    @ApiOperation("更新订单")
    @PostMapping("/technician/ordermanage/updateOrderManage")
    public JsonResult<Void> updateOrderManage(@RequestHeader String token,
                                              @RequestBody OrderManage param) {
        OrderIdLock orderIdLock = OrderIdLock.getInstance();
        if (param.getId() == null) {
            throw new InvalidParameterException();
        }
        try {
            orderIdLock.lock(param.getId());
            orderManageService.updateOrderManage(param);
        } finally {
            orderIdLock.unlock(param.getId());
        }
        return JsonResult.success();
    }


    @ApiOperation("查看评论")
    @GetMapping("/technician/ordermanage/getMyOrderComment")
    public JsonResult<IPage<OrderComment>> getMyOrderComment(@RequestParam Integer pageNum,
                                                             @RequestParam Integer pageSize,
                                                             @RequestHeader String token,
                                                             @DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss")
                                                             @RequestParam(required = false) LocalDateTime startTime,
                                                             @DateTimeFormat(pattern = "yyyy-MM-dd HH:mm:ss")
                                                             @RequestParam(required = false) LocalDateTime endTime) {
        Integer technicianId = SessionUtils.getTechnicianId();
        Page<OrderComment> page = new Page<>(pageNum, pageSize);
        IPage<OrderComment> orderCommentIPage = orderCommentMapper.queryMyCommentForPage(page, technicianId, startTime, endTime);
        orderCommentIPage.getRecords().forEach(orderComment -> {
            if (StringUtils.isEmpty(orderComment.getRemark())) {
                orderComment.setRemark("");
            }
        });
        return JsonResult.success(orderCommentIPage);
    }


    @ApiOperation("重算业绩")
    @GetMapping("/technician/ordermanage/no_achievementReCalc/{orderId}")
    public JsonResult<Void> achievementReCalc(@PathVariable Integer orderId) {
//        marketService.achievementReCalc(orderId);
        return JsonResult.success();
    }

    @ApiOperation("重算订单金额")
    @GetMapping("/technician/ordermanage/no_calcOrderAmount/{orderId}")
    public JsonResult<Void> calcOrderAmount(@PathVariable Integer orderId) {
//        marketService.calcOrderAmount(orderId);
        return JsonResult.success();
    }
}
