package com.gogirl.infrastructure.schedule;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.gogirl.application.order.mall.MallOrderService;
import com.gogirl.application.order.serve.ScheduleManageService;
import com.gogirl.application.product.serve.FeaturesMappingService;
import com.gogirl.application.product.serve.ProduceSalesService;
import com.gogirl.application.store.store.StoreManageService;
import com.gogirl.application.store.store.StoreTechnicianService;
import com.gogirl.application.user.customer.CustomerService;
import com.gogirl.domain.order.mall.MallOrder;
import com.gogirl.domain.order.serve.ScheduleManage;
import com.gogirl.domain.order.serve.ScheduleServe;
import com.gogirl.domain.product.serve.BaseFeatures;
import com.gogirl.domain.product.serve.FeaturesMapping;
import com.gogirl.domain.product.serve.ProduceSales;
import com.gogirl.domain.store.store.StoreManage;
import com.gogirl.domain.store.store.StoreTechnician;
import com.gogirl.domain.user.customer.Customer;
import com.gogirl.domain.user.customer.CustomerBalanceRecord;
import com.gogirl.infrastructure.common.util.ListUtil;
import com.gogirl.infrastructure.mapper.market.coupon.CouponCustomerRelevanceMapper;
import com.gogirl.infrastructure.mapper.order.mall.MallOrderMapper;
import com.gogirl.infrastructure.mapper.order.serve.OrderServeMapper;
import com.gogirl.infrastructure.mapper.order.serve.ScheduleManageMapper;
import com.gogirl.infrastructure.mapper.order.serve.ScheduleServeMapper;
import com.gogirl.infrastructure.mapper.product.serve.BaseFeaturesMapper;
import com.gogirl.infrastructure.mapper.product.serve.FeaturesMappingMapper;
import com.gogirl.infrastructure.mapper.product.serve.ProduceSalesMapper;
import com.gogirl.infrastructure.mapper.user.customer.CustomerBalanceRecordMapper;
import com.gogirl.infrastructure.subscribe.SubscribeService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

//@EnableScheduling
@AllArgsConstructor
@Slf4j
@Component
public class Schedule {

    /**
     * domain service
     */
    private final StoreTechnicianService storeTechnicianService;
    private final StoreManageService storeManageService;
    private final ScheduleManageService scheduleManageService;

    /**
     * repository
     */
    private final CouponCustomerRelevanceMapper couponCustomerRelevanceDao;
    private final CustomerBalanceRecordMapper balanceRecordDao;
    private final ScheduleManageMapper scheduleManageMapper;
    private final MallOrderMapper mallOrderMapper;


    private final SubscribeService subscribeService;
    private final CustomerService customerService;
    private final MallOrderService mallOrderService;

    private final ProduceSalesMapper produceSalesMapper;
    private final ScheduleServeMapper scheduleServeMapper;
    private final BaseFeaturesMapper baseFeaturesMapper;
    private final FeaturesMappingMapper featuresMappingMapper;
    private final ProduceSalesService produceSalesService;
    private final FeaturesMappingService featuresMappingService;

    private final OrderServeMapper orderServeMapper;

    /**
     * 每天定时任务,判断优惠券是否过期
     */
    @Scheduled(cron = "0 0 3 * * ?")
    public void setCouponExpireJob() {
        log.debug("每天定时任务,判断优惠券是否过期开始");
        couponCustomerRelevanceDao.setCouponExpire(new Date());
        log.debug("每天定时任务,判断优惠券是否过期结束");
    }

    /**
     * 每天定时设置推荐人:当天服务的技师为推荐人
     */
    @Scheduled(cron = "0 50 23 * * *")
    public void setChargeReferees() {
        log.debug("每天晚上11:50定时设置当天服务的技师为推荐人** 任务开始");

        SimpleDateFormat yyyyMMdd = new SimpleDateFormat("yyyy-MM-dd");

        String day = yyyyMMdd.format(new Date());

        log.debug("设置推荐人:当天服务的技师为推荐人");
        Map<String, String> mapTechnicianManage = new HashMap<>();
        List<StoreTechnician> listTechnicianManage = storeTechnicianService.listTechnicianForPage(new StoreTechnician());

        //查出所有的技师,找到技师id和user_id
        for (StoreTechnician item : listTechnicianManage) {
            mapTechnicianManage.put(String.valueOf(item.getTechnicianId()), String.valueOf(item.getUserId()));
        }

        //找到当天的订单的推荐人
        List<CustomerBalanceRecord> list = balanceRecordDao.getXcxChargeRecord(day);

        //遍历设置推荐人
        for (CustomerBalanceRecord customerBalanceRecord : list) {

            //推荐人列表
            Map<String, String> map = new HashMap<>();
            StringBuilder referee_ids = new StringBuilder();

            CustomerBalanceRecord cbrt = balanceRecordDao.selectByPrimaryKey(customerBalanceRecord.getId());

            if (customerBalanceRecord.getRefereeId() != null) {
                //每次都修改推荐人,且每晚都会设置推荐人
                String[] arr = customerBalanceRecord.getRefereeId().split(",");
                for (String s : arr) {
                    if (!map.containsKey(s)) {
                        map.put(s, s);
                        referee_ids.append(",").append(mapTechnicianManage.get(s));
                    }
                }
                if (referee_ids.toString().startsWith(",")) {
                    referee_ids = new StringBuilder(referee_ids.substring(1));
                }
                customerBalanceRecord.setRefereeId(referee_ids.toString());
            }

            if (cbrt != null && cbrt.getDepartmentId() != null) {
                customerBalanceRecord.setDepartmentId(cbrt.getDepartmentId());
            }
            if (customerBalanceRecord.getId() != null && (customerBalanceRecord.getRefereeId() != null || customerBalanceRecord.getDepartmentId() != null)) {
                balanceRecordDao.updateByPrimaryKeySelective(customerBalanceRecord);
            }
        }
        log.debug("每天晚上11:50定时设置当天服务的技师为推荐人** 任务结束");
    }

    /**
     * 提前一小时通知客户到店服务
     */
    @Scheduled(cron = "0 1/2 * * * *")
    public void notifyCustomerToShopService() {
        log.debug("提前一个小时通知客户到店服务** 任务开始");

        List<ScheduleManage> scheduleManageList = scheduleManageMapper.selectArriveTimeBeforeOneHour();

        scheduleManageList.forEach(scheduleManage -> {

            //如果没有发送过
            if (scheduleManage.getIsSend() == null || !scheduleManage.getIsSend().equals(1)) {
                Customer customer = customerService.getById(scheduleManage.getScheduledUser());
                StoreManage storeManage = storeManageService.getById(scheduleManage.getDepartmentId());

                List<ScheduleServe> scheduleServeList = scheduleManageMapper.selectByScheduleId(scheduleManage.getId());
                subscribeService.sendScheduleMsg(customer.getOpenid1(), scheduleServeList.get(0).getServeName(), scheduleManage.getArriveTime(), storeManage.getName(), storeManage.getAddress());

                scheduleManage.setIsSend(1);
                scheduleManageMapper.updateById(scheduleManage);
            }
        });

        log.debug("提前一个小时通知客户到店服务** 任务结束");
    }

    /**
     * 每两分钟定时检查有无过期的预约
     */
    @Scheduled(cron = "0 0/2 * * * *")
    public void cancelPassSchedule() {

        log.debug("定时检查有无过期的预约** 任务开始");
        Date passDate = new Date(new Date().getTime() - new Long("900000"));

        List<ScheduleManage> scheduleManageList = scheduleManageMapper.selectList(new LambdaQueryWrapper<ScheduleManage>()
                .lt(ScheduleManage::getArriveTime, passDate)
                .eq(ScheduleManage::getStatus, 1));

        //批量更新为 失约预约
        if (ListUtil.isNotEmpty(scheduleManageList)) {


            scheduleManageList.forEach(scheduleManage -> {

                List<ScheduleServe> scheduleServeList = scheduleManageMapper.selectByScheduleId(scheduleManage.getId());

                if (!scheduleManageList.stream().map(ScheduleManage::getStatus)
                        .map(status -> status.equals(1))
                        .collect(Collectors.toList())
                        .contains(false)) {

                    scheduleManage.setStatus(3);
                    scheduleManage.setLastUpdateTime(new Date());
                    Customer customer = customerService.getById(scheduleManage.getScheduledUser());
                    StoreManage storeManage = storeManageService.getById(scheduleManage.getDepartmentId());
                    if (customer != null) {
                        subscribeService.sendOverScheduleMsg(customer.getOpenid1(), scheduleServeList.get(0).getServeName(), scheduleManage.getArriveTime(), storeManage.getName(), storeManage.getAddress());
                    }
                }
            });
            scheduleManageService.updateBatchById(scheduleManageList);
        }

        log.debug("定时检查有无过期的预约** 任务结束");
    }

    /**
     * 每两分钟定时检查有无过期的商城订单
     */
    @Scheduled(cron = "0 0/2 * * * *")
    public void cancelPassMallOrder() {

        log.debug("定时检查有无过期的商城订单** 任务开始");
        Date passDate = new Date(new Date().getTime() - new Long("1800000"));

        List<MallOrder> mallOrderList = mallOrderMapper.selectList(new LambdaQueryWrapper<MallOrder>()
                .lt(MallOrder::getCreateTime, passDate)
                //状态待付款
                .eq(MallOrder::getStatus, 1));

        //批量更新为 失约预约
        if (ListUtil.isNotEmpty(mallOrderList)) {
            mallOrderList.forEach(mallOrder -> {
                //交易关闭
                mallOrder.setStatus(5);
                //超时关闭
                mallOrder.setCloseReason(1);
            });
        }

        if (ListUtil.isNotEmpty(mallOrderList)) {
            mallOrderService.updateBatchById(mallOrderList);
        }
        log.debug("定时检查有无过期的商城订单** 任务结束");
    }

    /**
     * 同步商品销量
     */
    @Scheduled(cron = "0 0/5 * * * *")
    public void syncProduceSales() {
        log.debug("同步商品销量** 任务开始");
        BaseFeatures baseFeatures = baseFeaturesMapper.selectOne(new LambdaQueryWrapper<BaseFeatures>().eq(BaseFeatures::getName, "热销款"));
        if (baseFeatures != null) {
            List<ProduceSales> produceSalesList = produceSalesMapper.countAll();

            featuresMappingMapper.delete(new LambdaQueryWrapper<FeaturesMapping>()
                    .eq(FeaturesMapping::getFeaturesId, baseFeatures.getId())
                    .eq(FeaturesMapping::getDataType, 2));

            featuresMappingService.saveBatch(produceSalesList.stream()
                    .map(produceSales -> {
                        FeaturesMapping featuresMapping = new FeaturesMapping();
                        featuresMapping.setDataType(2);
                        featuresMapping.setFeaturesId(baseFeatures.getId());
                        featuresMapping.setDataId(produceSales.getProduceId());
                        return featuresMapping;
                    })
                    .collect(Collectors.toList()));

            produceSalesService.remove(new LambdaQueryWrapper<>());
            produceSalesService.saveBatch(produceSalesList);

        }
        log.debug("同步商品销量** 任务结束");
    }

    /**
     * 每分钟同步订单实际时间
     */
    @Scheduled(cron = "0 0/1 * * * *")
    public void syncOrderTimes() {
        orderServeMapper.syncOrderServeTimes();
    }

}
