package com.gogirl.interfaces.market.coupon;

import com.gogirl.application.market.coupon.CouponService;
import com.gogirl.domain.market.coupon.Coupon;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.common.util.SessionUtils;
import com.gogirl.infrastructure.common.util.lock.CustomerIdLock;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

@RestController
@Api(tags = {"优惠券配置接口"})
@AllArgsConstructor
@Slf4j
public class CouponController {

    private final CouponService couponService;

    @ApiOperation(value = "根据配置id查询优惠券配置详情")
    @GetMapping("/customer/selectCouponByPrimaryKey")
    public JsonResult<Coupon> selectCouponByPrimaryKey(@RequestParam Integer id) {
        Coupon coupon = couponService.getById(id);
        return JsonResult.success(coupon);
    }


    @ApiOperation(value = "查询是否可领新人优惠券,是新人返回券信息,不是新人返回null;")
    @GetMapping("/customer/checkNewCustomer")
    public JsonResult<Coupon> checkNewCustomer() {
        Coupon coupon = couponService.checkNewCustomer(SessionUtils.getCustomerId());
        return JsonResult.success(coupon);
    }

    @ApiOperation(value = "发放新人优惠券")
    @PostMapping("/customer/sendNewCustomerCouponXcx")
    public JsonResult<Coupon> sendNewCustomerCouponXcx() {
        Integer customerId = SessionUtils.getCustomerId();
        CustomerIdLock lock = CustomerIdLock.getInstance();
        try {
            lock.lock(customerId);
            Coupon coupon = couponService.sendNewCustomerCouponXcx(customerId);
            return JsonResult.success(coupon);
        } finally {
            lock.unlock(customerId);
        }
    }

    @ApiOperation(value = "小程序发券")
    @PostMapping("/customer/sendCouponXcx")
    public JsonResult<Coupon> sendCouponXcx(@RequestParam Integer couponId) {
        Coupon coupon = couponService.sendCoupon(couponId, SessionUtils.getCustomerId());
        return JsonResult.success(coupon);
    }


    @ApiOperation(value = "美甲师根据订单号查询可用外部券")
    @GetMapping("/technician/getOrderExternalCoupon")
    public JsonResult<List<Coupon>> getOrderExternalCoupon(@RequestParam Integer orderId) {
        List<Coupon> couponList = couponService.getOrderExternalCoupon(orderId);
        return JsonResult.success(couponList);
    }
}
