package com.gogirl.application.store.store.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.gogirl.application.store.store.CareerCmdService;
import com.gogirl.application.store.store.WorksImagesService;
import com.gogirl.domain.product.purchase.PurchaseSku;
import com.gogirl.domain.product.serve.BaseProduce;
import com.gogirl.domain.store.career.Career;
import com.gogirl.domain.store.career.Works;
import com.gogirl.domain.store.career.WorksImages;
import com.gogirl.domain.store.training.TrainingCourse;
import com.gogirl.domain.store.training.TrainingStartClass;
import com.gogirl.domain.store.training.TrainingStartClassLogTechnician;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.common.util.ListUtil;
import com.gogirl.infrastructure.mapper.product.purchase.PurchaseSkuMapper;
import com.gogirl.infrastructure.mapper.product.serve.BaseProduceMapper;
import com.gogirl.infrastructure.mapper.store.career.CareerMapper;
import com.gogirl.infrastructure.mapper.store.career.WorksImagesMapper;
import com.gogirl.infrastructure.mapper.store.career.WorksMapper;
import com.gogirl.infrastructure.mapper.store.training.TrainingCourseMapper;
import com.gogirl.infrastructure.mapper.store.training.TrainingStartClassLogTechnicianMapper;
import com.gogirl.infrastructure.mapper.store.training.TrainingStartClassMapper;
import com.gogirl.infrastructure.util.SessionUtils;
import com.gogirl.shared.store.*;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

@Service
@Slf4j
@AllArgsConstructor
@Transactional
public class CareerCmdServiceImpl implements CareerCmdService {


    private final TrainingStartClassLogTechnicianMapper trainingStartClassLogTechnicianMapper;
    private final WorksMapper worksMapper;
    private final WorksImagesService worksImagesService;

    private final PurchaseSkuMapper purchaseSkuMapper;
    private final BaseProduceMapper baseProduceMapper;

    @Override
    public void submitWorks(SubmitWorksCommand cmd) {

        Works submittedWork = worksMapper.selectOne(new LambdaQueryWrapper<Works>()
                .eq(Works::getTechnicianId, cmd.getTechnicianId())
                .eq(Works::getProduceId, cmd.getProduceId())
        );
        if (submittedWork != null) {
            throw new RRException("您已经上传过美甲作品");
        }

        Works works = new Works();
        works.setCreateTime(System.currentTimeMillis());
        works.setRemarks(cmd.getRemarks());
        works.setTechnicianId(cmd.getTechnicianId());
        works.setProduceId(cmd.getProduceId());

        BaseProduce baseProduce = baseProduceMapper.selectById(works.getProduceId());
        works.setProduceName(baseProduce.getName());
        worksMapper.insert(works);

        List<WorksImages> worksImagesList = cmd.getImageUrlList()
                .stream()
                .map(imageUrl -> {
                    WorksImages worksImages = new WorksImages();
                    worksImages.setImageUrl(imageUrl);
                    worksImages.setWorksId(works.getId());
                    return worksImages;
                })
                .collect(Collectors.toList());

        worksImagesService.saveBatch(worksImagesList);
    }

    @Override
    public void signUpTraining(SignUpTrainingCommand cmd) {

        TrainingStartClassLogTechnician trainingStartClassLogTechnician = trainingStartClassLogTechnicianMapper
                .selectOne(new LambdaQueryWrapper<TrainingStartClassLogTechnician>()
                        .eq(TrainingStartClassLogTechnician::getTrainingStartClassId, cmd.getTrainingStartClassId())
                        .eq(TrainingStartClassLogTechnician::getTechnicianId, cmd.getTechnicianId()));

        if (trainingStartClassLogTechnician != null) {
            throw new RRException(500, "您已经报名过该课程");
        }


        trainingStartClassLogTechnician = TrainingStartClassLogTechnician
                .builder()
                .trainingStartClassId(cmd.getTrainingStartClassId())
                .technicianId(cmd.getTechnicianId())
                .createDate(new Date())
                .build();

        trainingStartClassLogTechnicianMapper.insert(trainingStartClassLogTechnician);
    }

    @Override
    @Transactional(timeout = 10000)
    public void submitFeedBack(SubmitFeedbackCommand cmd) {

        TrainingStartClassLogTechnician trainingStartClassLogTechnician = trainingStartClassLogTechnicianMapper.selectById(cmd.getTrainingStartClassLogTechnicianId());

        trainingStartClassLogTechnician.setFeedback(cmd.getFeedBackJson());
        trainingStartClassLogTechnician.setCreateDate(new Date());
        trainingStartClassLogTechnicianMapper.updateById(trainingStartClassLogTechnician);

    }

    @Override
    public IPage<Works> queryWorks(WorksQuery qry) {

        IPage<Works> page = new Page<>(qry.getPageNum(), qry.getPageSize());

        Calendar start = Calendar.getInstance();
        start.set(Calendar.YEAR, qry.getYear());
        start.set(Calendar.MONTH, qry.getMonth() - 1);
        start.set(Calendar.DAY_OF_MONTH, start.getActualMinimum(Calendar.DAY_OF_MONTH));

        Calendar end = Calendar.getInstance();
        end.set(Calendar.YEAR, qry.getYear());
        end.set(Calendar.MONTH, qry.getMonth() - 1);
        end.set(Calendar.DAY_OF_MONTH, end.getActualMaximum(Calendar.DAY_OF_MONTH));

        long startTime = start.getTimeInMillis();
        long entTime = end.getTimeInMillis();

        page = worksMapper.selectPage(page, new LambdaQueryWrapper<Works>()
                .gt(Works::getCreateTime, startTime)
                .lt(Works::getCreateTime, entTime)
                .eq(Works::getTechnicianId, SessionUtils.getTechnicianId())
                .orderByDesc(Works::getCreateTime));

        if (ListUtil.isEmpty(page.getRecords())) {
            return page;
        }

        List<Long> ids = page.getRecords().stream().map(Works::getId).collect(Collectors.toList());
        List<WorksImages> worksImages = worksImagesMapper.selectList(new LambdaQueryWrapper<WorksImages>().in(WorksImages::getWorksId, ids));
        Map<Long, List<WorksImages>> map = worksImages.stream().collect(Collectors.groupingBy(WorksImages::getWorksId));
        page.getRecords().forEach(works -> {
            List<WorksImages> worksImagesList = map.get(works.getId());
            if (ListUtil.isNotEmpty(worksImagesList)) {
                works.setWorksImagesList(new HashSet<>(worksImagesList));
            }
        });
        return page;
    }

    private final WorksImagesMapper worksImagesMapper;
    private final TrainingStartClassMapper trainingStartClassMapper;
    private final TrainingCourseMapper trainingCourseMapper;

    @Override
    public IPage<TrainingStartClass> queryPageClass(TrainingStartClassQuery qry) {
        IPage<TrainingStartClass> page = new Page<>(qry.getPageNum(), qry.getPageSize());
        page = trainingStartClassMapper.selectPage(page, new LambdaQueryWrapper<TrainingStartClass>().eq(TrainingStartClass::getStatus, 1));
        if (ListUtil.isEmpty(page.getRecords())) {
            return page;
        }

        //join
        List<Integer> courseIds = page.getRecords().stream().map(TrainingStartClass::getCourseId).collect(Collectors.toList());
        List<TrainingCourse> trainingCourseList = trainingCourseMapper.selectBatchIds(courseIds);
        Map<Integer, List<TrainingCourse>> map = trainingCourseList.stream().collect(Collectors.groupingBy(TrainingCourse::getId));
        page.getRecords().forEach(trainingStartClass -> {
            if (ListUtil.isNotEmpty(map.get(trainingStartClass.getCourseId()))) {
                trainingStartClass.setTrainingCourse(map.get(trainingStartClass.getCourseId()).stream().findAny().orElse(null));
            }
        });
        return page;
    }

    private final CareerMapper careerMapper;

    @Override
    public Career career(Integer technicianId) {
        Career career = careerMapper.selectOne(new LambdaQueryWrapper<Career>().eq(Career::getTechnicianId, technicianId));
        if (career == null) {
            career = new Career();
            career.setTechnicianId(technicianId);
        }
        return career;
    }

    @Override
    public Page<BaseProduce> queryWorkTpl(Integer technicianId, Integer pageNum, Integer pageSize, String searchKeyWord) {
        Page<BaseProduce> baseProducePage = new Page<>(pageNum, pageSize);

        baseProducePage = careerMapper.queryWorkTpl(baseProducePage, technicianId, searchKeyWord);

        List<PurchaseSku> purchaseSkuList = purchaseSkuMapper.listByProduceIds(baseProducePage.getRecords().stream().map(BaseProduce::getId).collect(Collectors.toList()));

        Map<Integer, List<PurchaseSku>> map = purchaseSkuList.stream().collect(Collectors.groupingBy(PurchaseSku::getProduceId));

        baseProducePage.getRecords().forEach(baseProduce -> {
            List<PurchaseSku> list = map.get(baseProduce.getId());
            if (ListUtil.isNotEmpty(list)) {
                baseProduce.setPurchaseSkuList(list);
            }
        });

        return baseProducePage;
    }
}
