package com.gogirl.interfaces.xcx;

import com.gogirl.application.order.serve.OrderManageService;
import com.gogirl.application.order.serve.ScheduleManageService;
import com.gogirl.application.store.store.StoreManageService;
import com.gogirl.application.store.store.StoreTechnicianService;
import com.gogirl.application.user.customer.CustomerService;
import com.gogirl.application.xcx.GogirlTokenService;
import com.gogirl.application.xcx.QRCodeService;
import com.gogirl.application.xcx.WechatService;
import com.gogirl.domain.store.store.StoreManage;
import com.gogirl.domain.store.store.StoreTechnician;
import com.gogirl.domain.user.Customer;
import com.gogirl.domain.xcx.GogirlToken;
import com.gogirl.infrastructure.common.base.JsonResult;
import com.gogirl.infrastructure.common.exception.RRException;
import com.gogirl.infrastructure.config.GogirlProperties;
import com.gogirl.infrastructure.mapper.user.CustomerBalanceMapper;
import com.gogirl.infrastructure.mapper.user.CustomerDetailMapper;
import com.gogirl.infrastructure.mapper.user.CustomerMapper;
import com.gogirl.infrastructure.mapper.xcx.GogirlTokenMapper;
import com.gogirl.infrastructure.util.ImageUtil;
import com.gogirl.infrastructure.util.SessionUtils;
import com.gogirl.shared.user.query.qry.CustomerQuery;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.security.InvalidAlgorithmParameterException;
import java.util.List;

@RestController
@AllArgsConstructor
@Slf4j
@Api(tags = "version2.0: 小程和用户相关接口")
public class XcxController {

    private final WechatService wechatService;
    private final GogirlTokenService gogirlTokenService;
    private final GogirlProperties gogirlProperties;

    private final GogirlTokenMapper gogirlTokenMapper;
    private final CustomerMapper customerMapper;


    private final QRCodeService qrCodeService;
    private final CustomerBalanceMapper customerBalanceMapper;
    private final CustomerDetailMapper customerDetailMapper;
    private final ScheduleManageService scheduleManageService;
    private final StoreTechnicianService storeTechnicianService;
    private final OrderManageService orderManageService;

    /**
     * 将emoji表情替换成空串
     *
     * @param source
     * @return 过滤后的字符串
     */
    private static String filterEmoji(String source) {
        if (source != null && source.length() > 0) {
            return source.replaceAll("[\ud800\udc00-\udbff\udfff\ud800-\udfff]", "");
        } else {
            return source;
        }
    }

    @ApiOperation("用户授权接街道地址信息")
    @GetMapping("/customer/xcx/bindAddressInfo")
    public JsonResult<String> bindAddressInfo(@RequestParam Integer customerId,
                                              @RequestParam String province,
                                              @RequestParam String city,
                                              @RequestParam String area,
                                              @RequestParam String street) {
        customerService.bindAddressInfo(customerId, province, city, area, street);
        return JsonResult.success();
    }

    @ApiOperation(("查询会员"))
    @PostMapping("/customer/xcx/query")
    public JsonResult<Customer> query(@RequestBody CustomerQuery qry) {
        Customer customer = customerMapper.selectById(qry.getCustomerId());
        return JsonResult.success(customer);
    }

    private final CustomerService customerService;

    @ApiOperation(value = "客户根据code获取token")
    @GetMapping("/customer/xcx/login")
    public JsonResult<String> login(@RequestParam String code) {
        String token = customerService.login(code);
        return JsonResult.success(token);
    }

    @ApiOperation(value = "客户授权后绑定个人信息")
    @RequestMapping(method = {RequestMethod.GET, RequestMethod.POST}, value = "/customer/xcx/authorized1")
    public JsonResult<Customer> authorized1(
            @RequestParam(name = "token", required = false) String token,
            @RequestParam(name = "encryptedData", required = false) String encryptedData,
            @RequestParam(name = "iv", required = false) String iv) throws InvalidAlgorithmParameterException {

        log.info("调用authorized，授权用户信息到程序");
        Customer customer = customerService.authorized1(token, encryptedData, iv);
        return JsonResult.success(customer);
    }

    @ApiOperation(value = "客户授权手机号码")
    @RequestMapping(method = {RequestMethod.POST}, value = "/customer/xcx/authorizedPhone")
    public JsonResult<Customer> authorizedPhone(@RequestHeader String token,
                                                @RequestParam String encryptedData,
                                                @RequestParam String iv) throws InvalidAlgorithmParameterException {
        log.info("手机号码授权,用户信息:");
        Customer customer = customerService.authorizedPhone(token, encryptedData, iv);
        return JsonResult.success();
    }

    @GetMapping("/customer/xcx/getUserInfo")
    @ApiOperation(value = "客户根据token获取用户信息")
    public JsonResult<Customer> getUserInfo(@RequestParam String token) throws Exception {
        log.info("根据token:" + token + ",查询用户信息.");
        Customer customer = customerService.getUserInfo(token);
        return JsonResult.success(customer);
    }

    @ApiOperation(value = "图片上传")
    @PostMapping("/customer/xcx/upload")
    public JsonResult<String> upload(MultipartFile file) throws Exception {
        log.info("图片上传");
        if (file == null) {
            throw new RRException("图片为空");
        }
        String imgUrl = ImageUtil.saveImage(gogirlProperties.getPicturePath(), file);
        return JsonResult.success(imgUrl);
    }

    @ApiOperation(value = "美甲师根据code获取token")
    @GetMapping(value = "/technician/xcx/login_t")
    public JsonResult<String> technicianLogin(@RequestParam String code) {
        log.info("调用login，授权用户信息到程序 code:{} 获取用户信息 ", code);
        String token = storeTechnicianService.login(code);
        return JsonResult.success(token);
    }

    @ApiOperation(value = "美甲师根据token获取美甲师信息")
    @GetMapping("/technician/xcx/getUserInfo_t")
    public JsonResult<StoreTechnician> getTechnician(String token) {
        log.info("根据token:" + token + ",查询用户信息.");
        return JsonResult.success(SessionUtils.getStoreTechnician());
    }

    private final StoreManageService storeManageService;

    @ApiOperation(value = "获取美甲师绑定的店铺")
    @GetMapping("/technician/xcx/getTechnicianShopList")
    public JsonResult<List<StoreManage>> getTechnicianShopList() {
        List<StoreManage> storeManageList = storeManageService.getTechnicianShopList();
        return JsonResult.success(storeManageList);
    }

    @ApiOperation(value = "美甲师选择多门店的账号")
    @PostMapping("/technician/xcx/choseStore")
    public JsonResult<Void> choseStore(@RequestParam(name = "token") String token,
                                       @RequestParam(name = "technicianId") Integer technicianId,
                                       @RequestParam Integer departmentId) {
        log.info("美甲师选择多门店的账号");
        GogirlToken gogirlToken = gogirlTokenService.getByToken(token);
        gogirlToken.setDepartmentId(departmentId);
        gogirlTokenService.updateById(gogirlToken);
        return JsonResult.success();
    }

    @ApiOperation(value = "图片上传")
    @PostMapping("/technician/xcx/upload")
    public JsonResult<String> techUpload(MultipartFile file) throws Exception {
        log.info("美甲师选择多门店的账号");
        if (file == null) {
            return new JsonResult<>(false, "file为空", null);
        }
        String imgUrl = ImageUtil.saveImage(gogirlProperties.getPicturePath(), file);
        return JsonResult.success(imgUrl);
    }
}
