package com.bailuntec.job;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bailuntec.domain.entity.DcBaseCompanyAccount;
import com.bailuntec.domain.entity.JobAccountLog;
import com.bailuntec.domain.entity.JobAmazonAdLog;
import com.bailuntec.domain.entity.JobPointLog;
import com.bailuntec.domain.example.DcBaseCompanyAccountExample;
import com.bailuntec.domain.example.JobAccountLogExample;
import com.bailuntec.domain.example.JobAmazonAdLogExample;
import com.bailuntec.domain.pojo.AmazonAdAuth;
import com.bailuntec.domain.pojo.AmazonAdSuccessResult;
import com.bailuntec.mapper.DcBaseCompanyAccountMapper;
import com.bailuntec.mapper.JobAccountLogMapper;
import com.bailuntec.mapper.JobAmazonAdLogMapper;
import com.bailuntec.support.PointJob;
import com.bailuntec.utils.ExceptionUtil;
import com.bailuntec.utils.OkHttpUtil;
import com.bailuntec.utils.PropertiesUtil;
import com.bailuntec.utils.SessionUtil;
import com.dangdang.ddframe.job.api.ShardingContext;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.util.List;

@Slf4j
public class AmazonAdGenerateReportIdJob extends PointJob {
    private final int platformId = 15;
    private static final PropertiesUtil propertiesUtil = PropertiesUtil.getInstance("const");
    private final OkHttpClient client = OkHttpUtil.getInstance();
    @Override
    public void executeJob(ShardingContext shardingContext, JobPointLog jobPointLog) {
        int totalPage = getCount();
        try {
            JobAccountLogMapper jobAccountLogMapper = SessionUtil.getSession().getMapper(JobAccountLogMapper.class);
            int pageSize = totalPage % shardingContext.getShardingTotalCount() == 0 ? totalPage / shardingContext.getShardingTotalCount() : totalPage / shardingContext.getShardingTotalCount() + 1;
            List<JobAccountLog> listByPage = jobAccountLogMapper.getListByPageAmazon(platformId,pageSize * shardingContext.getShardingItem(), pageSize);
            if (listByPage != null && listByPage.size() > 0) {
                for (JobAccountLog jobAccountLog : listByPage) {
                    if (jobAccountLog.getId() == null) { //Id为null在任务表无记录
                        BeanUtils.copyProperties(jobAccountLog, jobPointLog);
                        jobAccountLog.setId(null);
                    }
                    generateReportId(jobAccountLog);
                    if (jobAccountLog.getId() == null) { //null在任务表无记录
                        jobAccountLogMapper.insertSelective(jobAccountLog);
                    } else {
                        jobAccountLog.setBjModified(LocalDateTime.now());
                        jobAccountLogMapper.updateByExampleSelective(jobAccountLog, JobAccountLogExample.newAndCreateCriteria().andAccountIdEqualTo(jobAccountLog.getAccountId()).andJobNameEqualTo(jobAccountLog.getJobName()).example());
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.warn("Amaozom定时同步广告接口错误", e);
        } finally {
            SessionUtil.closeSession();
        }
    }

    /**
     * 如果请求报告时间比当天12点还早36小时,
     * 才允许生成报告
     * 避免报告不完整
     * 比如 7月26号12:00才允许生成7月25号的报告
     * @param jobAccountLog
     */
    private void generateReportId(JobAccountLog jobAccountLog) {
        LocalDateTime startZeroTime = LocalDateTime.of(jobAccountLog.getStartTime().toLocalDate(), LocalTime.MIN);
        LocalDateTime now = LocalDateTime.now();
        LocalDateTime localDateTime = LocalDateTime.of(now.getYear(), now.getMonth(), now.getDayOfMonth(), 12,0);
        if (Duration.between(startZeroTime, localDateTime).toHours() >= 36) {
            DcBaseCompanyAccount dcBaseCompanyAccount = getToken(jobAccountLog);
            if (StringUtils.isNoneBlank(dcBaseCompanyAccount.getAmazonAdAuthJson())) {
                /**
                 * 先查DB里有没有jobAccountLog.getStartTime那天的ReportId
                 */
                JobAmazonAdLogMapper jobAmazonAdLogMapper = SessionUtil.getSession().getMapper(JobAmazonAdLogMapper.class);
                JobAmazonAdLog jobAmazonAdLog = jobAmazonAdLogMapper.selectOneByExample(JobAmazonAdLogExample.newAndCreateCriteria().andCompanyIdEqualTo(jobAccountLog.getCompanyId()).andAccountIdEqualTo(jobAccountLog.getAccountId()).andReportDateEqualTo(jobAccountLog.getStartTime().toLocalDate()).example());
                /**
                 * jobAccountLog.getStartTime那天的ReportId不存在
                 * 发请求, 存在就更新时间
                 */
                if (jobAmazonAdLog == null || (jobAmazonAdLog != null && StringUtils.isBlank(jobAmazonAdLog.getReportId()))) {
                    /**
                     * 授权信息存在才去调接口获取ReportId
                     */
                    AmazonAdAuth amazonAdAuth = JSON.parseObject(dcBaseCompanyAccount.getAmazonAdAuthJson(), AmazonAdAuth.class);
                    //1. 先获取ReportId
                    postReportId(jobAccountLog, amazonAdAuth, dcBaseCompanyAccount.getSiteEn(), jobAmazonAdLogMapper);
                } else {
                    jobAccountLogUpdateTime(jobAccountLog);
                }
            }
        }
    }

    private void postReportId(JobAccountLog jobAccountLog, AmazonAdAuth amazonAdAuth, String siteEn, JobAmazonAdLogMapper jobAmazonAdLogMapper) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("reportDate", jobAccountLog.getStartTime().format(DateTimeFormatter.ofPattern("yyyyMMdd")));
        jsonObject.put("metrics", propertiesUtil.getPropertyAsString("METRICS"));
        MediaType mediaType = MediaType.parse("application/json");
        Response response = null;
        try{
            RequestBody body = RequestBody.create(mediaType, jsonObject.toJSONString());
            Request request = new Request.Builder()
                    .url(switchSiteUrl(siteEn, null))
                    .post(body)
                    .addHeader("Authorization",  amazonAdAuth.getAccessToken())
                    .addHeader("Amazon-Advertising-API-ClientId", amazonAdAuth.getClientId())
                    .addHeader("Amazon-Advertising-API-Scope", amazonAdAuth.getProfileId())
                    .addHeader("Content-Type", "application/json")
                    .build();
            response = client.newCall(request).execute();
            String string = response.body().string();
            if (response.isSuccessful()) {
                AmazonAdSuccessResult amazonAdSuccessResult = JSON.parseObject(string, AmazonAdSuccessResult.class);
                JobAmazonAdLog jobAmazonAdLog = new JobAmazonAdLog(null, jobAccountLog.getAccountId(),amazonAdSuccessResult.getReportId(),jobAccountLog.getStartTime().toLocalDate(),false, null, LocalDateTime.now(), siteEn, jobAccountLog.getCompanyId());
                int updateInt = jobAmazonAdLogMapper.updateByExampleSelective(jobAmazonAdLog, JobAmazonAdLogExample.newAndCreateCriteria().andAccountIdEqualTo(jobAccountLog.getAccountId()).andReportDateEqualTo(jobAccountLog.getStartTime().toLocalDate()).example());
                if (updateInt == 0) {
                    jobAmazonAdLogMapper.insertSelective(jobAmazonAdLog);
                }
               jobAccountLogUpdateTime(jobAccountLog);
            }
        }catch (Exception e){
            jobAccountLog.setMessage("调用获取AmazonADReportId错误:" + ExceptionUtil.transform(e));
        } finally {
            if (response != null) {
                response.close();
            }
        }
    }

    private void jobAccountLogUpdateTime(JobAccountLog jobAccountLog) {
        /*
         * 今天只能拿到昨天的广告费, 不然不完整
         * 如果开始时间比今天0点晚, 就设为22点(或者你随便减几个小时都可以)
         */
        LocalDateTime zeroTime = LocalDateTime.of(LocalDate.now(), LocalTime.MIN);
        LocalDateTime overStartTime = jobAccountLog.getEndTime().isAfter(zeroTime) ? zeroTime.minusHours(8) : jobAccountLog.getEndTime();
        jobAccountLog.setStartTime(overStartTime);
        jobAccountLog.setEndTime(jobAccountLog.getStartTime().plusDays(jobAccountLog.getIntervalTime()).isAfter(LocalDateTime.now()) ? LocalDateTime.now() : jobAccountLog.getEndTime().plusDays(jobAccountLog.getIntervalTime()));
        jobAccountLog.setMessage("执行成功");
    }

    /**
     * 如果param为null,
     * 就POST _URL
     * 否则GET _URL
     * https://advertising.amazon.com/API/docs/v2/reference/product_ads
     * @param siteEn
     * @param param
     * @return
     */
    protected static String switchSiteUrl(String siteEn, String param) {
        switch (siteEn.toUpperCase()) {
            case "UK":
            case "FR":
            case "IT":
            case "ES":
            case "DE":
                return getURL("EU_URL", param);
            case "AU":
            case "JP":
            case "IN":
                return getURL("FE_URL", param);
            case "US":
            case "CA":
            case "MX":
            default:
                return getURL("NA_URL", param);
        }

    }

    protected static String getURL(String siteUrl, String param) {
        if (StringUtils.isBlank(param)) {
            return  propertiesUtil.getPropertyAsString(siteUrl) + propertiesUtil.getPropertyAsString("POST_PRODUCT_ADS_REPORT_ID");
        }
        return propertiesUtil.getPropertyAsString(siteUrl) + propertiesUtil.getPropertyAsString("GET_PRODUCT_ADS_REPORT_DOWNLOAD_PREFIX")+ param + propertiesUtil.getPropertyAsString("GET_PRODUCT_ADS_REPORT_DOWNLOAD_SUFFIX");
    }

    private DcBaseCompanyAccount getToken(JobAccountLog jobAccountLog) {
        DcBaseCompanyAccountMapper mapper = SessionUtil.getSession().getMapper(DcBaseCompanyAccountMapper.class);
        return  mapper.selectOneByExample(DcBaseCompanyAccountExample.newAndCreateCriteria().andCompanyIdEqualTo(jobAccountLog.getCompanyId()).andAccountIdEqualTo(jobAccountLog.getAccountId()).example());
    }

    private int getCount()  {
        int i = 0;
        try {
            DcBaseCompanyAccountMapper accountMapper = SessionUtil.getSession().getMapper(DcBaseCompanyAccountMapper.class);
            i = (int)accountMapper.countByExample(DcBaseCompanyAccountExample.newAndCreateCriteria().andPlatformIdEqualTo(platformId).andAmazonAdAuthJsonNotEqualTo("").example());
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            SessionUtil.closeSession();
        }
        return i;
    }
}
