package com.bailuntec.balancesheet.service;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bailuntec.balancesheet.dto.CwSystemApiResultDTO;
import com.bailuntec.balancesheet.dto.PlatformAndSiteDTO;
import com.bailuntec.balancesheet.entity.DcBalanceSheet;
import com.bailuntec.balancesheet.mapper.*;
import com.bailuntec.balancesheet.util.DateTimeUtil;
import com.bailuntec.balancesheet.util.HttpUtil;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.context.junit4.SpringRunner;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;

@RunWith(SpringRunner.class)
@SpringBootTest
public class BalanceSheetServiceTest {

    @Autowired
    private BalanceSheetService balanceSheetService;

    @Autowired
    private DcBaseSkuMapper dcBaseSkuMapper;

    @Autowired
    private DcCompanyMainBodyMapper dcCompanyMainBodyMapper;
    @Autowired
    private DcBankAccountBalanceMapper dcBankAccountBalanceMapper;
    @Autowired
    private DcWithdrawOnWayMapper dcWithdrawOnWayMapper;
    @Autowired
    private DcShortTermBorrowMapper dcShortTermBorrowMapper;
    @Autowired
    private DcBalanceSheetMapper dcBalanceSheetMapper;
    @Autowired
    private DcInventoryBalanceMapper dcInventoryBalanceMapper;
    @Autowired
    private DcPlatformBalanceMapper dcPlatformBalanceMapper;
    @Autowired
    private DcPlatformAndSiteMapper dcPlatformAndSiteMapper;
    @Autowired
    private DcSupplierTransactionMapper dcSupplierTransactionMapper;
    @Autowired
    private DcExchangeRateMapper dcExchangeRateMapper;
    @Value("${url.getPlatformAndSite}")
    private String getPlatformAndSiteUrl;

    @Autowired
    private DcBaseCostMapper dcBaseCostMapper;
    @Autowired
    private DcOtherCostMapper dcOtherCostMapper;
    @Autowired
    private DcFixedAssetsDetailMapper dcFixedAssetsDetailMapper;
    @Autowired
    private DcSemiDailyStockMapper dcSemiDailyStockMapper;

    @Test
    public void updatePlatformBanlance() {
        dcPlatformBalanceMapper.updatePayPalBalance(DateTimeUtil.getCurrentLastDay());
    }

    @Test
    public void getExchangeRates() {
        balanceSheetService.getExchangeRate(DateTimeUtil.getCurrentLastDay());
    }

    @Test
    public void doScheduledTask() {
        balanceSheetService.doScheduledTask0000();
    }

    @Test
    public void getBankAccountBalance() {
        Date lastDay = DateTimeUtil.getCurrentLastDay();
        // 银行账户余额
        balanceSheetService.getBankAccountBalance(lastDay);
    }

    @Test
    public void getPlatformAndSites() {
        try {
            CwSystemApiResultDTO<JSONArray> responseDTO = JSONObject.parseObject(HttpUtil.httpGet(getPlatformAndSiteUrl), CwSystemApiResultDTO.class);
            if (responseDTO.getSuccess() && responseDTO.getData().size() > 0) {
                List<PlatformAndSiteDTO> platformAndSiteDTOList = JSONObject.parseArray(responseDTO.getData().toJSONString(), PlatformAndSiteDTO.class);
                dcPlatformAndSiteMapper.insertPlatformAndSites(platformAndSiteDTOList);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    @Test
    public void generateBalanceSheet() {
        Date startDate = DateTimeUtil.stringToDate("2019-01-28", DateTimeUtil.DATE_FORMAT);
        Date endDate = DateTimeUtil.stringToDate("2019-02-26", DateTimeUtil.DATE_FORMAT);
        while (startDate.compareTo(endDate) == -1) {
            balanceSheetService.generateBalanceSheet(0, "百伦供应链", startDate);
            startDate = DateTimeUtil.addDays(startDate, 1); // 日期增加一天
        }
    }

    @Test
    public void updatePlatformBalance() {
        Date startDate = DateTimeUtil.stringToDate("2019-03-15", DateTimeUtil.DATE_FORMAT);
        Date endDate = DateTimeUtil.stringToDate("2019-03-16", DateTimeUtil.DATE_FORMAT);
        while (startDate.compareTo(endDate) == -1) {
            // 银行账号
            BigDecimal bankAccountBalanceSum = dcBankAccountBalanceMapper.getBalanceSum(startDate);
            if (bankAccountBalanceSum == null) bankAccountBalanceSum = BigDecimal.ZERO;
            // 提现在途金额汇总
            BigDecimal withdrawOnWaySum = dcWithdrawOnWayMapper.getWithdrawOnWaySum(startDate);
            if (withdrawOnWaySum == null) withdrawOnWaySum = BigDecimal.ZERO;
            // 第三方平台余额
            BigDecimal platformBalanceSum = dcPlatformBalanceMapper.getPlatformBalanceSum(startDate);
            if (platformBalanceSum == null) platformBalanceSum = BigDecimal.ZERO;
            DcBalanceSheet dcBalanceSheet = new DcBalanceSheet();
            dcBalanceSheet.setMonetaryFund(bankAccountBalanceSum.add(withdrawOnWaySum).add(platformBalanceSum));
            dcBalanceSheet.setCompanyValue(0);
            dcBalanceSheet.setStatisticalTime(startDate);
            dcBalanceSheetMapper.updateByCompanyAndDaySelective(dcBalanceSheet);
            startDate = DateTimeUtil.addDays(startDate, 1); // 日期增加一天
        }
    }

    @Test
    public void getSupplierTransaction() {
        Date date = DateTimeUtil.stringToDate("2019-04-24", DateTimeUtil.DATE_FORMAT);
        balanceSheetService.getSupplierTransaction(date);
        balanceSheetService.updateSuppliserTransactionOfBalanceSheet(date);
    }

    @Test
    public void getSupplierTransactionBatch() {
        Date startDate = DateTimeUtil.stringToDate("2019-04-25", DateTimeUtil.DATE_FORMAT);
        Date endDate = DateTimeUtil.stringToDate("2019-05-08", DateTimeUtil.DATE_FORMAT);
        while (startDate.compareTo(endDate) == -1) {
            balanceSheetService.getSupplierTransaction(startDate);
//            balanceSheetService.getLogisticsSupplierTransaction(startDate);
            balanceSheetService.updateSuppliserTransactionOfBalanceSheet(startDate);
            startDate = DateTimeUtil.addDays(startDate, 1); // 日期增加一天
        }
    }

    @Test
    public void getLogisticsSupplierTransaction() {
        Date date = DateTimeUtil.stringToDate("2019-03-29", DateTimeUtil.DATE_FORMAT);
        balanceSheetService.getLogisticsSupplierTransaction(date);
        balanceSheetService.updateSuppliserTransactionOfBalanceSheet(date);
    }

    @Test
    public void getShortTermBorrow() {
        Date lastDay = DateTimeUtil.getCurrentLastDay();
        balanceSheetService.getShortBorrowBalance(lastDay);
        BigDecimal shortBorrowBalanceSum = dcShortTermBorrowMapper.getShortTermBorrowBalanceSum(lastDay);
        if (shortBorrowBalanceSum == null) shortBorrowBalanceSum = BigDecimal.ZERO;
        dcBalanceSheetMapper.updateShortTermBorrow(shortBorrowBalanceSum, lastDay);
    }

    @Test
    public void getInventoryBalance() {
        Date startDate = DateTimeUtil.stringToDate("2019-05-01", DateTimeUtil.DATE_FORMAT);
        Date endDate = DateTimeUtil.stringToDate("2019-07-19", DateTimeUtil.DATE_FORMAT);
        while (startDate.compareTo(endDate) == -1) {
//            balanceSheetService.getInventoryBalance(startDate);
            balanceSheetService.updateInventoryBalance(startDate);
            startDate = DateTimeUtil.addDays(startDate, 1); // 日期增加一天
        }
    }

    @Test
    public void getSemiInventoryBalance() {
        /*Date startDate = DateTimeUtil.stringToDate("2014-04-25", DateTimeUtil.DATE_FORMAT);
        Date endDate = DateTimeUtil.stringToDate("2018-08-06", DateTimeUtil.DATE_FORMAT);
        balanceSheetService.getSemiInventoryBalance(startDate,endDate);*/
        Date startDate = DateTimeUtil.stringToDate("2020-09-28", DateTimeUtil.DATE_FORMAT);
        Date endDate = DateTimeUtil.stringToDate("2020-09-29", DateTimeUtil.DATE_FORMAT);
        assert endDate != null;
        assert startDate != null;
        while (startDate.compareTo(endDate) < 0) {
            balanceSheetService.getSemiInventoryBalance(startDate, DateTimeUtil.addDays(startDate, 1));
            startDate = DateTimeUtil.addDays(startDate, 1); // 日期增加一天
        }
    }

    @Test
    public void getSemiTransferAmount() {

        Date startDate = DateTimeUtil.stringToDate("2020-09-25", DateTimeUtil.DATE_FORMAT);
        Date endDate = DateTimeUtil.stringToDate("2020-09-29", DateTimeUtil.DATE_FORMAT);
        assert endDate != null;
        assert startDate != null;
        while (startDate.compareTo(endDate) < 0) {
            balanceSheetService.getSemiTransferAmount(startDate, DateTimeUtil.addDays(startDate, 1));
            // 日期增加一天
            startDate = DateTimeUtil.addDays(startDate, 1);
        }
    }

    @Test
    public void getOtherCost() {
        Date today = DateTimeUtil.getCurrentDay();
        Date lastDay = DateTimeUtil.getCurrentLastDay();
        Date twoDaysAgo = DateTimeUtil.addDays(lastDay, -1);
        // 其他应收款、其他应付款
        balanceSheetService.getOtherCost(twoDaysAgo, lastDay, today);
        // 其他应收款、其他应付款
        BigDecimal otherAccountsReceivable = dcOtherCostMapper.getOtherAccountsReceivableSum(lastDay); // 其他应收款
        if (otherAccountsReceivable == null) otherAccountsReceivable = BigDecimal.ZERO;
        BigDecimal otherAccountsPayable = dcOtherCostMapper.getOtherAccountsPayableSum(lastDay); // 其他应付款
        if (otherAccountsPayable == null) otherAccountsPayable = BigDecimal.ZERO;
        // 更新资产负债表统计字段
        DcBalanceSheet dcBalanceSheet = new DcBalanceSheet();
        dcBalanceSheet.setCompanyValue(0);
        dcBalanceSheet.setStatisticalTime(lastDay);
        dcBalanceSheet.setOtherAccountsReceivable(otherAccountsReceivable); // 其他应收款
        dcBalanceSheet.setOtherAccountsPayable(otherAccountsPayable); // 其他应付款
        dcBalanceSheetMapper.updateByCompanyAndDaySelective(dcBalanceSheet);
    }

    @Test
    public void updateOther() {
        Date date = DateTimeUtil.stringToDate("2020-09-29", DateTimeUtil.DATE_FORMAT);
        Date dateSubOne = DateTimeUtil.addDays(date, -1);
        dcOtherCostMapper.insertOneDayOtherReceive(date, dateSubOne);
        dcOtherCostMapper.insertOneDayOtherPay(date, dateSubOne);
        // 其他应收款、其他应付款
        BigDecimal otherAccountsReceivable = dcOtherCostMapper.getOtherAccountsReceivableSum(date); // 其他应收款
        if (otherAccountsReceivable == null) otherAccountsReceivable = BigDecimal.ZERO;
        BigDecimal otherAccountsPayable = dcOtherCostMapper.getOtherAccountsPayableSum(date); // 其他应付款
        if (otherAccountsPayable == null) otherAccountsPayable = BigDecimal.ZERO;
        // 更新资产负债表统计字段
        DcBalanceSheet dcBalanceSheet = new DcBalanceSheet();
        dcBalanceSheet.setCompanyValue(0);
        dcBalanceSheet.setStatisticalTime(date);
        dcBalanceSheet.setOtherAccountsReceivable(otherAccountsReceivable); // 其他应收款
        dcBalanceSheet.setOtherAccountsPayable(otherAccountsPayable); // 其他应付款
        dcBalanceSheetMapper.updateByCompanyAndDaySelective(dcBalanceSheet);
    }

    @Test
    public void getFixedAssets() {
        Date startDate = DateTimeUtil.stringToDate("2019-03-01", DateTimeUtil.DATE_FORMAT);
        Date endDate = DateTimeUtil.stringToDate("2019-03-07", DateTimeUtil.DATE_FORMAT);
        while (startDate.compareTo(endDate) == -1) {
            // 固定资产
            balanceSheetService.getFixedAssets(DateTimeUtil.addDays(startDate, -1), startDate, DateTimeUtil.addDays(startDate, 1));
            BigDecimal fixedAssetsSum = dcFixedAssetsDetailMapper.getFixedAssetsSum(startDate);
            if (fixedAssetsSum == null) fixedAssetsSum = BigDecimal.ZERO;
            // 更新资产负债表统计字段
            DcBalanceSheet dcBalanceSheet = new DcBalanceSheet();
            dcBalanceSheet.setCompanyValue(0);
            dcBalanceSheet.setStatisticalTime(startDate);
            dcBalanceSheet.setFixedAssets(fixedAssetsSum);
            dcBalanceSheetMapper.updateByCompanyAndDaySelective(dcBalanceSheet);
            startDate = DateTimeUtil.addDays(startDate, 1); // 日期增加一天
        }
    }

    @Test
    public void getSellerAccount() {
        balanceSheetService.getSellerAccount();
    }

    @Test
    public void updateFund() {
        Date startDate = DateTimeUtil.stringToDate("2019-05-01", DateTimeUtil.DATE_FORMAT);
        Date endDate = DateTimeUtil.stringToDate("2019-06-11", DateTimeUtil.DATE_FORMAT);
        while (startDate.compareTo(endDate) == -1) {
            DcBalanceSheet dcBalanceSheet = new DcBalanceSheet();
            // 银行账户余额汇总
            BigDecimal bankAccountBalanceSum = dcBankAccountBalanceMapper.getBalanceSum(startDate);
            if (bankAccountBalanceSum == null) bankAccountBalanceSum = BigDecimal.ZERO;
            dcBalanceSheet.setBankAccountBalance(bankAccountBalanceSum);
            // 提现在途金额汇总
            BigDecimal withdrawOnWaySum = dcWithdrawOnWayMapper.getWithdrawOnWaySum(startDate);
            if (withdrawOnWaySum == null) withdrawOnWaySum = BigDecimal.ZERO;
            dcBalanceSheet.setWithdrawAmount(withdrawOnWaySum);
            // 第三方平台余额
            BigDecimal platformBalanceSum = dcPlatformBalanceMapper.getPlatformBalanceSum(startDate);
            if (platformBalanceSum == null) platformBalanceSum = BigDecimal.ZERO;
            dcBalanceSheet.setPlatformBalance(platformBalanceSum);
            // 货币资金
            dcBalanceSheet.setMonetaryFund(bankAccountBalanceSum.add(withdrawOnWaySum).add(platformBalanceSum)); // 货币资金 = 银行余额 + 提现在途 + 第三方平台余额

            dcBalanceSheet.setCompanyValue(0);
            dcBalanceSheet.setStatisticalTime(startDate);

            // 更新资产负债表统计字段
            dcBalanceSheetMapper.updateByCompanyAndDaySelective(dcBalanceSheet);
            startDate = DateTimeUtil.addDays(startDate, 1); // 日期增加一天
        }
    }

}
