package com.bailuntec.job;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bailuntec.domain.entity.*;
import com.bailuntec.domain.example.DcBaseStockConfigFbaExample;
import com.bailuntec.domain.example.DcBaseStockConfigFbaLoseExample;
import com.bailuntec.domain.example.DcBaseStockExample;
import com.bailuntec.domain.example.DcBaseStockFbaExample;
import com.bailuntec.domain.pojo.FbaStockData;
import com.bailuntec.domain.pojo.FbaStockInfo;
import com.bailuntec.mapper.DcBaseStockConfigFbaLoseMapper;
import com.bailuntec.mapper.DcBaseStockConfigFbaMapper;
import com.bailuntec.mapper.DcBaseStockFbaMapper;
import com.bailuntec.mapper.DcBaseStockMapper;
import com.bailuntec.support.PointJob;
import com.bailuntec.utils.OkHttpUtil;
import com.bailuntec.utils.PropertiesUtil;
import com.bailuntec.utils.SessionUtil;
import com.dangdang.ddframe.job.api.ShardingContext;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;

@Slf4j
public class FbaStockJob extends PointJob {
    private PropertiesUtil propertiesUtil = PropertiesUtil.getInstance("const");

    /**
     * 1. 这里是将FBA库存分页查出
     * 放到一个哈希表
     * 然后不做处理直接保存到dc_base_stock_fba留底
     * 2. 要把共享FBA库存找出来,
     * 按优先级配置只保留一个
     * 按bailun_sku + warehouse维度 插入dc_base_stock
     *
     * @param shardingContext
     * @param jobPointLog
     */
    @Override
    public void executeJob(ShardingContext shardingContext, JobPointLog jobPointLog) {
        Integer totalPage = null;
        HashMap<String, DcBaseStockFba> hashmap = new HashMap<>();
        do {
            log.info("总共{}页", totalPage);
            log.info("当前{}页", jobPointLog.getPageIndex());

            OkHttpClient client = OkHttpUtil.getInstance();
            MediaType mediaType = MediaType.parse("application/json");
            LinkedHashMap<String, Object> map = new LinkedHashMap<>(3);
            map.put("PageIndex", jobPointLog.getPageIndex());
            map.put("PageRow", jobPointLog.getPageSize());
            RequestBody body = RequestBody.create(mediaType, JSON.toJSONString(map));
            Request request = new Request.Builder()
                    .url(propertiesUtil.getPropertyAsString("STOCK_FBA_URL"))
                    .post(body)
                    .addHeader("Content-Type", "application/json")
                    .build();
            Response response = null;
            String resultStr = null;
            try {
                response = client.newCall(request).execute();
                resultStr = response.body().string();
            } catch (IOException e) {
                throw new RuntimeException("调用wms系统FBA库存接口失败", e);
            } finally {
                if (response != null) {
                    response.close();
                }
            }
            if (StringUtils.isNotBlank(resultStr)) {
                JSONObject jsonObject = JSON.parseObject(resultStr);
                if (jsonObject.get("isSuccess") != null && jsonObject.getBooleanValue("isSuccess")) {
                    FbaStockData fbaStockData = jsonObject.getObject("data", FbaStockData.class);
                    if (fbaStockData != null && fbaStockData.getStocks() != null && fbaStockData.getStocks().size() > 0) {
                        totalPage = fbaStockData.getPageTotal() + 1;//这里是因为接口分页错误
                        for (FbaStockInfo fbaStockInfo : fbaStockData.getStocks()) {
                            /*
                             * 仓库 + 平台SKU维度, 保证FBA库存唯一
                             * 但是同一个bailun_sku 可能当成多个平台SKU在卖
                             */
                            String uniqueKey = fbaStockInfo.getWarehouseCode() + fbaStockInfo.getPlatformSku();
                            DcBaseStockFba dcBaseStockFbaInMap = hashmap.get(uniqueKey);
                            DcBaseStockFba dcBaseStockFba = new DcBaseStockFba();
                            try {
                                BeanUtils.copyProperties(dcBaseStockFba, fbaStockInfo);
                            } catch (Exception e) {
                                throw new RuntimeException("BeanUtils.copyPropertiesFBA库存数据失败");
                            }
                            dcBaseStockFba.setGmtModified(LocalDateTime.now());
                            if (dcBaseStockFbaInMap != null) {
                                dcBaseStockFba.setInWarehouse(dcBaseStockFbaInMap.getInWarehouse() + fbaStockInfo.getInWarehouse());
                                dcBaseStockFba.setUsableStock(dcBaseStockFbaInMap.getUsableStock() + fbaStockInfo.getUsableStock());
                            }
                            hashmap.put(uniqueKey, dcBaseStockFba);
                        }
                    }
                } else {
                    throw new RuntimeException("调用wms系统FBA库存接口返回200但是失败");
                }
            } else {
                throw new RuntimeException("调用wms系统FBA库存接口失败");
            }
            jobPointLog.setPageIndex(jobPointLog.getPageIndex() + 1);
        } while (jobPointLog.getPageIndex() <= totalPage);
        jobPointLog.setPageIndex(1);

        /*
         * 将FBA库存按平台SKU为Key,DcBaseStockFba为值放到了Value中
         */
        HashMap<String, DcBaseStock> bailunSkuWarehouseStock = new HashMap<>();
        try {
            DcBaseStockFbaMapper mapper = SessionUtil.getSession().getMapper(DcBaseStockFbaMapper.class);
            DcBaseStockConfigFbaMapper dcBaseStockConfigFbaMapper = SessionUtil.getSession().getMapper(DcBaseStockConfigFbaMapper.class);
            DcBaseStockConfigFbaLoseMapper dcBaseStockConfigFbaLoseMapper = SessionUtil.getSession().getMapper(DcBaseStockConfigFbaLoseMapper.class);
            DcBaseStockMapper dcBaseStockMapper = SessionUtil.getSession().getMapper(DcBaseStockMapper.class);
            hashmap.forEach(((uniqueKey, dcBaseStockFba) -> {
                int i = mapper.updateByExampleSelective(dcBaseStockFba, DcBaseStockFbaExample.newAndCreateCriteria().andWarehouseCodeEqualTo(dcBaseStockFba.getWarehouseCode()).andPlatformSkuEqualTo(dcBaseStockFba.getPlatformSku()).example());
                if (i == 0) {
                    mapper.insertSelective(dcBaseStockFba);
                }
            }));
            //这些平台SKU都是欧洲共享的
            List<String> listEuropeFbaStockSharePlatformSku = mapper.listEuropeFbaStockSharePlatformSku();
            mapper.insertDcBaseStockFbaShare();
            //将共享库存的SKU 存入dc_base_stock_fba_share, 处理销量时能用到
            for (String fbaStockSharePlatformSku : listEuropeFbaStockSharePlatformSku) {
                DcBaseStockConfigFba dcBaseStockConfigFba = dcBaseStockConfigFbaMapper.selectOneByExample(DcBaseStockConfigFbaExample.newAndCreateCriteria().andPlatformSkuEqualTo(fbaStockSharePlatformSku).example());
                List<DcBaseStockFba> listEuropeFbaStockShare = mapper.listEuropeFbaStockShare(fbaStockSharePlatformSku);
                if (dcBaseStockConfigFba != null) {
                    handleFbaShareStock(listEuropeFbaStockShare, dcBaseStockConfigFba);
                } else {
                    handleFbaShareStock(listEuropeFbaStockShare);
                    DcBaseStockConfigFbaLose dcBaseStockConfigFbaLose = new DcBaseStockConfigFbaLose();
                    dcBaseStockConfigFbaLose.setPlatformSku(fbaStockSharePlatformSku);
                    int i = dcBaseStockConfigFbaLoseMapper.updateByExampleSelective(dcBaseStockConfigFbaLose, DcBaseStockConfigFbaLoseExample.newAndCreateCriteria().andPlatformSkuEqualTo(fbaStockSharePlatformSku).example());
                    if (i == 0) {
                        dcBaseStockConfigFbaLoseMapper.insertSelective(dcBaseStockConfigFbaLose);
                    }
                }
                //listEuropeFbaStockShare这里经过处理, 只有一个仓会有库存
                for (DcBaseStockFba dcBaseStockFba : listEuropeFbaStockShare) {
                    if (StringUtils.isNotBlank(dcBaseStockFba.getBailunSku())) {
                        putInBailunSkuWarehouseMap(bailunSkuWarehouseStock, dcBaseStockFba);
                    }
                }
            }
            /*
             * 将欧洲不共享的FBA库存查出来放入Map
             * 将非欧洲的FBA库存插入
             */
            List<DcBaseStockFba> stockOtherFbaList = mapper.listOtherFbaStock();
            if (stockOtherFbaList != null && stockOtherFbaList.size() > 0) {
                for (DcBaseStockFba dcBaseStockFba : stockOtherFbaList) {
                    putInBailunSkuWarehouseMap(bailunSkuWarehouseStock, dcBaseStockFba);
                }
            }
            bailunSkuWarehouseStock.forEach((bailunSkuWarehouse, dcBaseStock) -> {
                int i = dcBaseStockMapper.updateByExampleSelective(dcBaseStock, DcBaseStockExample.newAndCreateCriteria().andBailunSkuEqualTo(dcBaseStock.getBailunSku()).andWarehouseCodeEqualTo(dcBaseStock.getWarehouseCode()).example());
                if (i == 0) {
                    dcBaseStockMapper.insertSelective(dcBaseStock);
                }
            });
            //TODO 在更新完FBA库存后, 不是这个时间段更新的FBA库存置0, 即可避免FBA共享库存优先级配置修改的BUG
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("MYBATIS操作DB存FBA库存失败", e);
        } finally {
            SessionUtil.closeSession();
        }
    }

    private void putInBailunSkuWarehouseMap(HashMap<String, DcBaseStock> bailunSkuWarehouseStock, DcBaseStockFba dcBaseStockFba) {
        DcBaseStock dcBaseStock = bailunSkuWarehouseStock.get(dcBaseStockFba.getBailunSku() + dcBaseStockFba.getWarehouseCode());
        if (dcBaseStock != null) {
            dcBaseStock.setUsableStock(dcBaseStock.getUsableStock() + dcBaseStockFba.getUsableStock());
            dcBaseStock.setInWarehouse(dcBaseStock.getInWarehouse() + dcBaseStockFba.getInWarehouse());
        } else {
            dcBaseStock = new DcBaseStock();
            dcBaseStock.setBailunSku(dcBaseStockFba.getBailunSku());
            dcBaseStock.setWarehouseCode(dcBaseStockFba.getWarehouseCode());
            dcBaseStock.setUsableStock(dcBaseStockFba.getUsableStock());
            dcBaseStock.setInWarehouse(dcBaseStockFba.getInWarehouse());
        }
        bailunSkuWarehouseStock.put(dcBaseStockFba.getBailunSku() + dcBaseStockFba.getWarehouseCode(), dcBaseStock);
    }

    private void handleFbaShareStock(List<DcBaseStockFba> stockFbaList, DcBaseStockConfigFba dcBaseStockConfigFba) {
        boolean haveConfig = false;
        for (int i = 0; i < stockFbaList.size(); i++) {
            if (i < stockFbaList.size() - 1) {
                if (stockFbaList.get(i).getWarehouseCode().equals(dcBaseStockConfigFba.getWarehouseCode())) {
                    haveConfig = true;
                } else {
                    stockFbaList.get(i).setUsableStock(0);
                    stockFbaList.get(i).setInWarehouse(0);
                }
            } else {
                if (haveConfig) {
                    stockFbaList.get(i).setUsableStock(0);
                    stockFbaList.get(i).setInWarehouse(0);
                }
            }
        }
    }

    private void handleFbaShareStock(List<DcBaseStockFba> stockFbaList) {
        for (int i = 0; i < stockFbaList.size() - 1; i++) {
            stockFbaList.get(i).setUsableStock(0);
            stockFbaList.get(i).setInWarehouse(0);
        }
    }
}
