package com.bailuntec.job;

import com.alibaba.fastjson.JSON;
import com.bailuntec.domain.constant.CommonConstant;
import com.bailuntec.domain.entity.*;
import com.bailuntec.domain.enumerate.CurrencyType;
import com.bailuntec.domain.pojo.PayPalResultInfo;
import com.bailuntec.domain.pojo.ResultData;
import com.bailuntec.domain.entity.DcBaseFinancePaypal;
import com.bailuntec.mapper.DcBaseFinancePaypalMapper;
import com.bailuntec.mapper.JobPointLogMapper;
import com.bailuntec.support.CallBailunSystem;
import com.bailuntec.support.PointJob;
import com.bailuntec.utils.OkHttpUtil;
import com.bailuntec.utils.PropertiesUtil;
import com.bailuntec.utils.SessionUtil;
import com.dangdang.ddframe.job.api.ShardingContext;
import lombok.extern.slf4j.Slf4j;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import org.apache.commons.lang3.StringUtils;
import java.io.IOException;
import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.*;

@Slf4j
public class PayPalSyncJob extends PointJob {
    private OkHttpClient okHttpClient = OkHttpUtil.getInstance();
    private PropertiesUtil propertiesUtil = PropertiesUtil.getInstance("const");

    @Override
    public void executeJob(ShardingContext shardingContext, JobPointLog jobPointLog) {
        LinkedHashMap<String, String> map = new LinkedHashMap<>(4);
        map.put("rows", jobPointLog.getPageSize().toString());
        map.put("btime", DateTimeFormatter.ofPattern(CommonConstant.TIME_FORMAT).format(jobPointLog.getStartTime()));
        map.put("etime", DateTimeFormatter.ofPattern(CommonConstant.TIME_FORMAT).format(jobPointLog.getEndTime()));
        do {
            map.put("page", jobPointLog.getPageIndex().equals(0) ? "1" : jobPointLog.getPageIndex().toString());
            Response response = null;
            String palResultStr = null;
            try {
                Request request = new Request.Builder()
                        .get()
                        .url(OkHttpUtil.attachHttpGetParams(propertiesUtil.getPropertyAsString("PAYPAL_URL"), map))
                        .addHeader("Content-Type", "application/json")
                        .build();
                response = okHttpClient.newCall(request).execute();
                palResultStr = response.body().string();
            } catch (IOException e) {
                throw new RuntimeException(map + "请求流水接口同步百伦接口失败" + response, e);
            } finally {
                if (response != null) {
                    response.close();
                }
            }
            if (StringUtils.isNoneBlank(palResultStr)) {
                PayPalResultInfo palResultRoot = JSON.parseObject(palResultStr, PayPalResultInfo.class);
                if (palResultRoot != null && palResultRoot.getSuccess().booleanValue()) {
                    ResultData<DcBaseFinancePaypal> resultData = palResultRoot.getData();
                    if (jobPointLog.getPageIndex().equals(0)) {
                        jobPointLog.setPageIndex(resultData.getTotalPages() + 1);
                    }
                    if (resultData.getItems() != null && resultData.getItems().size() > 0) {
                        handlePayPalJson(resultData.getItems(), jobPointLog);
                    }
                } else {
                    throw new RuntimeException("调用流水接口同步百伦流水失败, 响应200, 请求参数" + map.toString());
                }
            } else {
                throw new RuntimeException("调用流水接口同步百伦流水失败, 响应为null, 请求参数" + map.toString());
            }
            if (jobPointLog.getPageIndex() % 10 == 0) {
                try {
                    JobPointLogMapper mapper = SessionUtil.getSession().getMapper(JobPointLogMapper.class);
                    mapper.upsertSelective(jobPointLog);
                } catch (Exception e) {
                    e.printStackTrace();
                    throw new RuntimeException("Mybatis操作DB插入任务记录失败", e);
                } finally {
                    SessionUtil.closeSession();
                }
            }
            jobPointLog.setPageIndex(jobPointLog.getPageIndex() - 1);
        } while (0 < jobPointLog.getPageIndex());
        jobPointLog.setPageIndex(0);
        jobPointLog.setStartTime(jobPointLog.getEndTime());
        jobPointLog.setEndTime(jobPointLog.getEndTime().plusDays(jobPointLog.getIntervalTime()).isAfter(LocalDateTime.now()) ? LocalDateTime.now() : jobPointLog.getEndTime().plusDays(jobPointLog.getIntervalTime()));

    }

    private void handlePayPalJson(List<DcBaseFinancePaypal> data, JobPointLog jobPointLog) {
        try{
            for(DcBaseFinancePaypal paypal : data) {
                handleSourceJson(paypal);
                Instant instant = paypal.getPaymentDate().toInstant();
                ZoneId zone = ZoneId.systemDefault();
                LocalDateTime paymentDate = LocalDateTime.ofInstant(instant, zone);
                BigDecimal exchangeRate = CallBailunSystem.getExchangeRate(paypal.getCurrency(), CurrencyType.CNY.value(), paymentDate);
                paypal.setOtherToCnyExchangeRate(exchangeRate);
                DcBaseFinancePaypalMapper mapper = SessionUtil.getSession().getMapper(DcBaseFinancePaypalMapper.class);
                mapper.upsertSelective(paypal);
            }
        }catch (RuntimeException e) {
            throw new RuntimeException("MYBATIS操作流水失败",e);
        }finally {
            SessionUtil.closeSession();
        }
    }

    private void handleSourceJson(DcBaseFinancePaypal paypal) {
        String sourceJson = paypal.getSourceJson();
        if(StringUtils.isNotBlank(sourceJson)) {
            paypal.setBusiness(getValue(sourceJson, "Business\":"));
            paypal.setReceiver(getValue(sourceJson, "Receiver\":"));
            paypal.setReceiverId(getValue(sourceJson, "ReceiverID\":"));
            paypal.setPayer(getValue(sourceJson, "Payer\":"));
            paypal.setPayerId(getValue(sourceJson, "PayerID\":"));
            paypal.setCurrencyId(getValue(sourceJson, "currencyID\":") == null ? 0 : Integer.parseInt(getValue(sourceJson, "currencyID\":")) );
            paypal.setCurrency(CurrencyType.getVal(paypal.getCurrencyId()));
        }
    }

    public String getValue(String sourceJson, String key) {
        String value = sourceJson.substring(sourceJson.indexOf(key) + key.length(), sourceJson.indexOf(",", sourceJson.indexOf(key))).replace("\"", "").replace("}", "");
        if("null".equals(value)) {
            return null;
        }
        return value;
    }

}
