package com.bailuntec.job.service;

import com.alibaba.fastjson.JSON;
import com.bailuntec.api.amazon.AmazonAdApi;
import com.bailuntec.api.amazon.request.AmazonDisplayReportReq;
import com.bailuntec.api.amazon.request.AmazonProductsReportReq;
import com.bailuntec.api.amazon.request.AmazonURI;
import com.bailuntec.api.amazon.response.AmazonAdReportResp;
import com.bailuntec.application.IDcBaseFinanceAmazonAdProductService;
import com.bailuntec.common.BeanUtils;
import com.bailuntec.common.JsonUtilByFsJson;
import com.bailuntec.common.JsonUtilByJackson;
import com.bailuntec.common.StringUtils;
import com.bailuntec.domain.DcBaseCompanyAccount;
import com.bailuntec.domain.DcBaseFinanceAmazonAdProduct;
import com.bailuntec.domain.DcJobConfig;
import com.bailuntec.domain.JobAmazonAdLog;
import com.bailuntec.infrastructure.mapper.DcBaseCompanyAccountMapper;
import com.bailuntec.infrastructure.mapper.DcBaseFinanceAmazonAdProductMapper;
import com.bailuntec.infrastructure.mapper.DcJobConfigMapper;
import com.bailuntec.infrastructure.mapper.JobAmazonAdLogMapper;
import com.bailuntec.infrastructure.util.CallBailunSystemUtil;
import com.bailuntec.infrastructure.util.enumerate.CurrencyType;
import com.bailuntec.job.AmazonAdProduct;
import com.bailuntec.pojo.AmazonAdAuth;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.google.common.reflect.TypeToken;
import feign.FeignException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.net.URI;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.GZIPInputStream;

/**
 * <p>
 *
 * </p>
 *
 * @author robbendev
 * @since 2020/11/12 3:06 下午
 */
@Slf4j
@Service
public class AmazonJobService {
    @Resource
    private DcBaseCompanyAccountMapper dcBaseCompanyAccountMapper;
    @Resource
    private JobAmazonAdLogMapper jobAmazonAdLogMapper;
    @Resource
    private AmazonAdApi amazonAdApi;
    @Resource
    DcJobConfigMapper dcJobConfigMapper;


    private static final String baseAdAmazonReportGenerate = "base-ad-amazon-report-generate";
    private static final String baseAdAmazonReportDownload = "base-ad-amazon-report-download";

    @Transactional(rollbackFor = Exception.class)
    public void generateAmazonAdReport() {
        DcJobConfig dcJobConfig = dcJobConfigMapper.selectByName(baseAdAmazonReportGenerate);

        //需要生成的帐号列表
        List<DcBaseCompanyAccount> dcBaseCompanyAccountList = dcBaseCompanyAccountMapper.selectList(new LambdaQueryWrapper<DcBaseCompanyAccount>()
                .eq(DcBaseCompanyAccount::getPlatformId, DcBaseCompanyAccount.BLT_PLATFORM_ID)
                .ne(DcBaseCompanyAccount::getAmazonAdAuthJson, ""))
                .stream()
                .filter(dcBaseCompanyAccount -> StringUtils.isNotEmpty(dcBaseCompanyAccount.getAmazonAdAuthJson().trim()))
                .collect(Collectors.toList());

        //需要生成报告时间：昨天
        LocalDate localDate = dcJobConfig.getStartTime().minusHours(36).toLocalDate();
        this.generateAmazonAdReport(dcBaseCompanyAccountList, localDate, localDate);

        dcJobConfigMapper.updateById(dcJobConfig.refresh());
    }

    public void generateAmazonAdReport(List<DcBaseCompanyAccount> dcBaseCompanyAccountList, LocalDate start, LocalDate end) {

        //获取时间区间reportDate列表
        List<LocalDate> localDateList = Stream.iterate(start, localDate -> localDate.plusDays(1))
                .limit(ChronoUnit.DAYS.between(start, end) + 1)
                .collect(Collectors.toList());

        //双重循环生成报告
        dcBaseCompanyAccountList
                .forEach(dcBaseCompanyAccount ->
                        localDateList.forEach(reportDate -> {
                            try {
                                this.generateAmazonAdReport(dcBaseCompanyAccount, reportDate);
                            } catch (Exception e) {
                                log.error("生成报告失败 帐号id:{},日期:{}", dcBaseCompanyAccount.getAccountId(), reportDate, e);
                            }
                        }));
    }

    /**
     * <p>
     * 生成指定帐号指定某一天日期的所有类型亚马逊广告报告
     * 包括：
     * </p>
     *
     * <li>品牌展示</li>
     * <li>商品推广</li>
     *
     * @param dcBaseCompanyAccount dcBaseCompanyAccount
     * @param reportDate           reportDate
     */
    private void generateAmazonAdReport(DcBaseCompanyAccount dcBaseCompanyAccount, LocalDate reportDate) {

        /*1.商品展示报告生成*/
        AmazonAdReportResp productAdResp = this.reportAdProduct(dcBaseCompanyAccount, reportDate);
        //处理商品推广报告结果
        this.handleAmazonAdReportResp(dcBaseCompanyAccount, productAdResp);

        /*1.品牌展示报告生成*/
        AmazonAdReportResp displayResp = this.reportDisplay(dcBaseCompanyAccount, reportDate);
        //处理品牌展示报告结果
        this.handleAmazonAdReportResp(dcBaseCompanyAccount, displayResp);

    }

    /**
     * <p>
     * 获取指定账号和指定日期的商品推广生成报告响应
     * 可以看成{@link AmazonAdApi#report(URI, AmazonProductsReportReq, String, String, String, String)} 的代理
     * </p>
     *
     * @param dcBaseCompanyAccount 帐号
     * @param reportDate           指定日期
     * @return 生成商品推广报告响应
     */
    public AmazonAdReportResp reportAdProduct(DcBaseCompanyAccount dcBaseCompanyAccount, LocalDate reportDate) {
        AmazonAdAuth amazonAdAuth = JsonUtilByFsJson.jsonToBean(dcBaseCompanyAccount.getAmazonAdAuthJson(), AmazonAdAuth.class);

        AmazonProductsReportReq req = new AmazonProductsReportReq();
        req.setReportDate(reportDate.format(DateTimeFormatter.ofPattern("yyyyMMdd")));
        req.setMetrics("campaignName,campaignId,adGroupName,adGroupId,impressions,clicks,cost,currency,asin,sku,attributedConversions1d,attributedConversions7d,attributedConversions14d,attributedConversions30d,attributedConversions1dSameSKU,attributedConversions7dSameSKU,attributedConversions14dSameSKU,attributedConversions30dSameSKU,attributedUnitsOrdered1d,attributedUnitsOrdered7d,attributedUnitsOrdered14d,attributedUnitsOrdered30d,attributedSales1d,attributedSales7d,attributedSales14d,attributedSales30d,attributedSales1dSameSKU,attributedSales7dSameSKU,attributedSales14dSameSKU,attributedSales30dSameSKU,attributedUnitsOrdered1dSameSKU,attributedUnitsOrdered7dSameSKU,attributedUnitsOrdered14dSameSKU,attributedUnitsOrdered30dSameSKU");

        String reportResp = amazonAdApi.report(AmazonURI.getAmazonAdSiteUri(dcBaseCompanyAccount.getSiteEn()),
                req,
                "productAds",
                amazonAdAuth.getAccessToken(),
                amazonAdAuth.getClientId(),
                amazonAdAuth.getProfileId());

        log.debug(JsonUtilByFsJson.beanToJson(reportResp));

        //反序列化响应报文
        AmazonAdReportResp resp = JsonUtilByFsJson.jsonToBean(reportResp, AmazonAdReportResp.class);
        resp.setJobAmazonLogType(AmazonAdReportResp.productAds);
        resp.setReportDate(reportDate);
        return resp;
    }


    /**
     * <p>
     * 获取指定账号和指定日期的品牌展示生成报告响应
     * 可以看成{@link AmazonAdApi#reportDisPlay(URI, AmazonDisplayReportReq, String, String, String, String)} 的代理
     * </p>
     *
     * @param dcBaseCompanyAccount 帐号
     * @param reportDate           指定日期
     * @return 生成品牌展示报告响应
     */
    public AmazonAdReportResp reportDisplay(DcBaseCompanyAccount dcBaseCompanyAccount, LocalDate reportDate) {
        AmazonAdAuth amazonAdAuth = JsonUtilByFsJson.jsonToBean(dcBaseCompanyAccount.getAmazonAdAuthJson(), AmazonAdAuth.class);

        AmazonDisplayReportReq req = new AmazonDisplayReportReq();
        req.setReportDate(reportDate.format(DateTimeFormatter.ofPattern("yyyyMMdd")));
        req.setMetrics("campaignName,campaignId,impressions,adGroupId,adGroupName,asin,sku,adId,clicks,cost,currency,attributedConversions1d,attributedConversions7d,attributedConversions14d,attributedConversions30d,attributedConversions1dSameSKU,attributedConversions7dSameSKU,attributedConversions14dSameSKU,attributedConversions30dSameSKU,attributedUnitsOrdered1d,attributedUnitsOrdered7d,attributedUnitsOrdered14d,attributedUnitsOrdered30d,attributedSales1d,attributedSales7d,attributedSales14d,attributedSales30d,attributedSales1dSameSKU,attributedSales7dSameSKU,attributedSales14dSameSKU,attributedSales30dSameSKU");
        req.setTactic("T00020");

        String reportResp = amazonAdApi.reportDisPlay(AmazonURI.getAmazonAdSiteUri(dcBaseCompanyAccount.getSiteEn()),
                req,
                "productAds",
                amazonAdAuth.getAccessToken(),
                amazonAdAuth.getClientId(),
                amazonAdAuth.getProfileId());

        log.debug(JsonUtilByFsJson.beanToJson(reportResp));

        //反序列化响应报文
        AmazonAdReportResp resp = JsonUtilByFsJson.jsonToBean(reportResp, AmazonAdReportResp.class);
        resp.setJobAmazonLogType(AmazonAdReportResp.display);
        resp.setReportDate(reportDate);
        return resp;
    }


    /**
     * <p>
     * 处理亚马逊生成报告响应报文
     * 新增或者更新 {@link JobAmazonAdLog}状态
     * </p>
     *
     * @param resp
     */
    private void handleAmazonAdReportResp(DcBaseCompanyAccount dcBaseCompanyAccount, AmazonAdReportResp resp) {
        //新增或者更新报告状态为未下载
        JobAmazonAdLog jobAmazonAdLog = jobAmazonAdLogMapper.selectOne(new LambdaQueryWrapper<JobAmazonAdLog>()
                .eq(JobAmazonAdLog::getAccountId, dcBaseCompanyAccount.getAccountId())
                .eq(JobAmazonAdLog::getCompanyId, dcBaseCompanyAccount.getCompanyId())
                .eq(JobAmazonAdLog::getReportDate, resp.getReportDate())
                .eq(JobAmazonAdLog::getType, resp.getJobAmazonLogType()));

        if (jobAmazonAdLog == null) {
            jobAmazonAdLog = new JobAmazonAdLog(null,
                    dcBaseCompanyAccount.getAccountId(),
                    resp.getReportId(),
                    resp.getReportDate(),
                    false,
                    LocalDateTime.now(),
                    LocalDateTime.now(),
                    dcBaseCompanyAccount.getSiteEn(),
                    dcBaseCompanyAccount.getCompanyId(),
                    resp.getJobAmazonLogType());
            jobAmazonAdLogMapper.insert(jobAmazonAdLog);
            log.info("新增一条亚马逊报告 帐号id:{},报告日期:{}", jobAmazonAdLog.getAccountId(), jobAmazonAdLog.getReportDate());
        }

        //如果已经生成了报告 是否设置为需要重新下载？//yes
        else {
            jobAmazonAdLog.setReportId(resp.getReportId());
            jobAmazonAdLog.setStatus(false);
            jobAmazonAdLog.setBjModified(LocalDateTime.now());
            jobAmazonAdLogMapper.updateById(jobAmazonAdLog);
            log.info("更新一条报告 帐号id:{},报告日期:{}", jobAmazonAdLog.getAccountId(), jobAmazonAdLog.getReportDate());
        }
    }

    @Resource
    private DcBaseFinanceAmazonAdProductMapper dcBaseFinanceAmazonAdProductMapper;


    public void downloadAmazonAdReport() {
        DcJobConfig dcJobConfig = dcJobConfigMapper.selectByName(baseAdAmazonReportDownload);
        List<JobAmazonAdLog> jobAmazonAdLogList = jobAmazonAdLogMapper.selectList(new LambdaQueryWrapper<JobAmazonAdLog>()
                .eq(JobAmazonAdLog::getStatus, false));

        jobAmazonAdLogList.parallelStream().forEach(jobAmazonAdLog -> {
            try {
                this.downloadAmazonAdReport(jobAmazonAdLog);
            } catch (FeignException ex) {
                log.error("下载报告失败:{}", ex.getMessage(), ex);
            }
        });
        dcJobConfigMapper.updateById(dcJobConfig.refresh());
    }

    @Resource
    IDcBaseFinanceAmazonAdProductService dcBaseFinanceAmazonAdProductService;

    /**
     * <p>根据生成的报告下载报告 并且生成亚马逊广告费用明细</p>
     * <li>根据组合唯一键判断</li>
     * <li> 如果费用明细不存在就新增</li>
     * <li> 如果费用明细存在就更新</li>
     *
     * @param jobAmazonAdLog 生成的报告记录
     */
    private void downloadAmazonAdReport(JobAmazonAdLog jobAmazonAdLog) {
        DcBaseCompanyAccount dcBaseCompanyAccount = dcBaseCompanyAccountMapper.selectOne(new LambdaQueryWrapper<DcBaseCompanyAccount>()
                .eq(DcBaseCompanyAccount::getCompanyId, jobAmazonAdLog.getCompanyId())
                .eq(DcBaseCompanyAccount::getAccountId, jobAmazonAdLog.getAccountId()));

        AmazonAdAuth amazonAdAuth = JSON.parseObject(dcBaseCompanyAccount.getAmazonAdAuthJson(), AmazonAdAuth.class);

        dcBaseFinanceAmazonAdProductMapper.delete(new LambdaQueryWrapper<DcBaseFinanceAmazonAdProduct>()
                .eq(DcBaseFinanceAmazonAdProduct::getCompanyId, jobAmazonAdLog.getCompanyId())
                .eq(DcBaseFinanceAmazonAdProduct::getAccountId, jobAmazonAdLog.getAccountId())
                .eq(DcBaseFinanceAmazonAdProduct::getReportDate, jobAmazonAdLog.getReportDate())
                .eq(DcBaseFinanceAmazonAdProduct::getType, jobAmazonAdLog.getType()));


        byte[] response = amazonAdApi.reportDownload(AmazonURI.getAmazonAdSiteUri(dcBaseCompanyAccount.getSiteEn()),
                jobAmazonAdLog.getReportId(),
                amazonAdAuth.getAccessToken(),
                amazonAdAuth.getClientId(),
                amazonAdAuth.getProfileId());


        try (GZIPInputStream gzipInputStream = new GZIPInputStream(new ByteArrayInputStream(response))) {

            @SuppressWarnings("all")
            Type type = new TypeToken<List<AmazonAdProduct>>() {
            }.getType();

            List<AmazonAdProduct> reportDownloadRespList = JSON.parseObject(gzipInputStream, type);
            log.debug(JsonUtilByJackson.writeValueAsString(reportDownloadRespList));

            for (AmazonAdProduct downloadAdProduct : reportDownloadRespList) {
                if (downloadAdProduct.getCost() != null && BigDecimal.ZERO.compareTo(downloadAdProduct.getCost()) < 0) {

                    LambdaQueryWrapper<DcBaseFinanceAmazonAdProduct> wrapper = new LambdaQueryWrapper<DcBaseFinanceAmazonAdProduct>()
                            .eq(DcBaseFinanceAmazonAdProduct::getAccountId, jobAmazonAdLog.getAccountId())
                            .eq(DcBaseFinanceAmazonAdProduct::getCompanyId, jobAmazonAdLog.getCompanyId())
                            .eq(DcBaseFinanceAmazonAdProduct::getType, jobAmazonAdLog.getType())
                            .eq(DcBaseFinanceAmazonAdProduct::getReportDate, jobAmazonAdLog.getReportDate())

                            .eq(DcBaseFinanceAmazonAdProduct::getCampaignId, downloadAdProduct.getCampaignId());
                    if (StringUtils.isNotEmpty(downloadAdProduct.getAdGroupId())) {
                        wrapper.eq(DcBaseFinanceAmazonAdProduct::getAdGroupId, downloadAdProduct.getAdGroupId());

                    }
                    if (StringUtils.isNotEmpty(downloadAdProduct.getSku())) {
                        wrapper.eq(DcBaseFinanceAmazonAdProduct::getSku, downloadAdProduct.getSku());
                    }

                    DcBaseFinanceAmazonAdProduct dcBaseFinanceAmazonAdProduct = dcBaseFinanceAmazonAdProductMapper.selectOne(wrapper);

                    if (dcBaseFinanceAmazonAdProduct == null) {
                        dcBaseFinanceAmazonAdProduct = new DcBaseFinanceAmazonAdProduct();
                    }

                    BeanUtils.copyProperties(downloadAdProduct, dcBaseFinanceAmazonAdProduct);

                    dcBaseFinanceAmazonAdProduct.setAccountId(jobAmazonAdLog.getAccountId());
                    dcBaseFinanceAmazonAdProduct.setCompanyId(jobAmazonAdLog.getCompanyId());
                    dcBaseFinanceAmazonAdProduct.setReportDate(jobAmazonAdLog.getReportDate());
                    dcBaseFinanceAmazonAdProduct.setType(jobAmazonAdLog.getType());

                    dcBaseFinanceAmazonAdProduct.setExchangeRate(CallBailunSystemUtil.getExchangeRate(dcBaseFinanceAmazonAdProduct.getCurrency(), CurrencyType.CNY.value(), jobAmazonAdLog.getReportDate().atStartOfDay()));
                    dcBaseFinanceAmazonAdProduct.setExchangeRateUsd(CallBailunSystemUtil.getExchangeRate(dcBaseFinanceAmazonAdProduct.getCurrency(), CurrencyType.USD.value(), jobAmazonAdLog.getReportDate().atStartOfDay()));
                    dcBaseFinanceAmazonAdProduct.setBjModifyTime(LocalDateTime.now());
                    try {
                        dcBaseFinanceAmazonAdProductService.saveOrUpdate(dcBaseFinanceAmazonAdProduct, wrapper);
                    } catch (Exception ex) {
                        ex.printStackTrace();
                    }
                    log.info("同步亚马逊广告费");
                }
            }

            jobAmazonAdLog.setStatus(true);
            jobAmazonAdLog.setBjModified(LocalDateTime.now());
            jobAmazonAdLogMapper.updateById(jobAmazonAdLog);

        } catch (IOException e) {
            e.printStackTrace();
        }
    }
}
