package com.bailuntec.job;

import com.bailuntec.application.IDcBaseFinanceEbayService;
import com.bailuntec.common.JsonUtilByFsJson;
import com.bailuntec.common.ListUtil;
import com.bailuntec.domain.DcBaseCompanyAccount;
import com.bailuntec.domain.DcBaseFinanceEbay;
import com.bailuntec.domain.DcJobConfig;
import com.bailuntec.infrastructure.mapper.DcBaseCompanyAccountMapper;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.ebay.sdk.ApiContext;
import com.ebay.sdk.ApiCredential;
import com.ebay.sdk.TimeFilter;
import com.ebay.sdk.call.GetAccountCall;
import com.ebay.soap.eBLBaseComponents.AccountEntryType;
import com.ebay.soap.eBLBaseComponents.AccountHistorySelectionCodeType;
import com.ebay.soap.eBLBaseComponents.DetailLevelCodeType;
import com.ebay.soap.eBLBaseComponents.PaginationType;
import com.google.common.collect.Lists;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.batch.item.ItemReader;

import java.time.ZoneId;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * <p>
 *
 * </p>
 *
 * @author robbendev
 * @since 2021/1/18 5:23 下午
 */
@Slf4j
public class EbayItemReader implements ItemReader<EbayItem> {

    //剩余账户
    private List<DcBaseCompanyAccount> dcBaseCompanyAccountList;
    //剩余广告记录
    private List<AccountEntryType> accountEntryTypeList;
    //当前账户
    private DcBaseCompanyAccount dcBaseCompanyAccount;
    //当前ebay请求接口
    private GetAccountCall getAccountCall;
    //当前页码
    private Integer pageNum;
    //配置
    private DcJobConfig dcJobConfig;

    private final DcBaseCompanyAccountMapper dcBaseCompanyAccountMapper;
    private final IDcBaseFinanceEbayService dcBaseFinanceEbayService;

    @SneakyThrows
    public EbayItemReader(DcBaseCompanyAccountMapper dcBaseCompanyAccountMapper,
                          DcJobConfig dcJobConfig, IDcBaseFinanceEbayService dcBaseFinanceEbayService) {
        this.dcBaseCompanyAccountMapper = dcBaseCompanyAccountMapper;
        this.dcJobConfig = dcJobConfig;
        this.dcBaseFinanceEbayService = dcBaseFinanceEbayService;
        this.init();
    }

    private void init() {
        log.info("初始化reader 开始..");
        dcBaseCompanyAccountList = dcBaseCompanyAccountMapper.queryPageEbay();
        this.pageNum = dcJobConfig.getPageNum();
        this.nextAccount();
        log.info("初始化reader 结束");
    }


    @Override
    public EbayItem read() {
        EbayItem ebayItem = null;
        if (ListUtil.isNotEmpty(accountEntryTypeList)) {
            AccountEntryType accountEntryType = accountEntryTypeList.remove(0);
            ebayItem = new EbayItem(accountEntryType, dcBaseCompanyAccount);
        } else {
            //翻页
            nextPage();
            if (ListUtil.isNotEmpty(accountEntryTypeList)) {
                AccountEntryType accountEntryType = accountEntryTypeList.remove(0);
                ebayItem = new EbayItem(accountEntryType, dcBaseCompanyAccount);
            }
        }
        if (ebayItem != null) {
            log.info("读取一条广告费用记录:{}", JsonUtilByFsJson.beanToJson(ebayItem));
        }
        return ebayItem;
    }


    private void nextPage() {
        this.pageNum++;
        refreshPage();

        //翻页还是空 换账号
        if (ListUtil.isEmpty(accountEntryTypeList) && ListUtil.isNotEmpty(dcBaseCompanyAccountList)) {
            this.nextAccount();
        }
    }

    private void nextAccount() {
        this.pageNum = 1;
        if (ListUtil.isNotEmpty(dcBaseCompanyAccountList)) {
            dcBaseCompanyAccount = dcBaseCompanyAccountList.remove(0);

            log.info("切换账号 切换后当前账号:{},剩余待跑账号数量:{}", dcBaseCompanyAccount.getAccountName(), dcBaseCompanyAccountList.size());
            this.refreshAccountCall();
            this.refreshPage();

            //切换账号后 抓取广告费用前 前先清除已经抓取的数据
            dcBaseFinanceEbayService.remove(new LambdaQueryWrapper<DcBaseFinanceEbay>()
                    .between(DcBaseFinanceEbay::getBjDate, dcJobConfig.getStartTime(), dcJobConfig.getEndTime())
                    .eq(DcBaseFinanceEbay::getAccountId, dcBaseCompanyAccount.getAccountId()));
        }
        if (ListUtil.isEmpty(accountEntryTypeList) && ListUtil.isNotEmpty(dcBaseCompanyAccountList)) {
            this.nextAccount();
        }
    }

    void refreshAccountCall() {
        ApiContext apiContext = new ApiContext();
        ApiCredential cred = apiContext.getApiCredential();
        cred.seteBayToken(dcBaseCompanyAccount.getSoapAuthToken());
        apiContext.setApiServerUrl("https://api.ebay.com/wsapi");

        getAccountCall = new GetAccountCall(apiContext);
        //详情类型 返回所有
        getAccountCall.setDetailLevel(new DetailLevelCodeType[]{DetailLevelCodeType.RETURN_ALL});
        //类型
        getAccountCall.setViewType(AccountHistorySelectionCodeType.BETWEEN_SPECIFIED_DATES);

        //开始时间
        Calendar startCal = Calendar.getInstance();
        startCal.setTime(Date.from(dcJobConfig.getStartTime().atZone(ZoneId.systemDefault()).toInstant()));

        //结束时间
        Calendar endCal = Calendar.getInstance();
        endCal.setTime(Date.from(dcJobConfig.getEndTime().atZone(ZoneId.systemDefault()).toInstant()));

        TimeFilter timeFilter = new TimeFilter(startCal, endCal);
        getAccountCall.setViewPeriod(timeFilter);

        //endCal
        getAccountCall.setInvoiceDate(endCal);
    }

    private void refreshPage() {
        PaginationType paginationType = new PaginationType();
        paginationType.setPageNumber(pageNum);
        getAccountCall.setPagination(paginationType);
        log.info("请求ebay广告服务开始,当前账号:{}，当前页码:{}", dcBaseCompanyAccount.getAccountName(), pageNum);
        try {
            AccountEntryType[] accountEntryTypes = getAccountCall.getAccount();
            if (accountEntryTypes != null && accountEntryTypes.length > 0) {
                accountEntryTypeList = Lists.newArrayList(accountEntryTypes);
            }
        } catch (Exception ex) {
            log.info("请求ebay广告服务异常,当前账号:{}，当前页码:{}", dcBaseCompanyAccount.getAccountName(), pageNum);
            //todo log ebay exception
        } finally {
            log.info("请求ebay广告服务结束,当前账号:{}，当前页码:{}", dcBaseCompanyAccount.getAccountName(), pageNum);
        }
    }
}
