package com.bailuntec.job;

import com.bailuntec.common.ListUtil;
import com.bailuntec.domain.DcBaseCompanyAccount;
import com.bailuntec.domain.DcJobConfig;
import com.bailuntec.infrastructure.mapper.DcBaseCompanyAccountMapper;
import com.bailuntec.infrastructure.mapper.DcJobConfigMapper;
import com.ebay.sdk.ApiContext;
import com.ebay.sdk.ApiCredential;
import com.ebay.sdk.TimeFilter;
import com.ebay.sdk.call.GetAccountCall;
import com.ebay.soap.eBLBaseComponents.AccountEntryType;
import com.ebay.soap.eBLBaseComponents.AccountHistorySelectionCodeType;
import com.ebay.soap.eBLBaseComponents.DetailLevelCodeType;
import com.ebay.soap.eBLBaseComponents.PaginationType;
import com.google.common.collect.Lists;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.batch.item.ItemReader;

import java.time.ZoneId;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * <p>
 *
 * </p>
 *
 * @author robbendev
 * @since 2021/1/18 5:23 下午
 */
@Slf4j
public class EbayItemReader implements ItemReader<EbayItem> {

    //剩余账户
    private List<DcBaseCompanyAccount> dcBaseCompanyAccountList;
    //剩余广告记录
    private List<AccountEntryType> accountEntryTypeList;
    //当前账户
    private DcBaseCompanyAccount dcBaseCompanyAccount;
    //当前ebay请求接口
    private GetAccountCall getAccountCall;
    //当前页码
    private Integer pageNum;
    //配置
    private DcJobConfig dcJobConfig;

    private final DcBaseCompanyAccountMapper dcBaseCompanyAccountMapper;
    private final DcJobConfigMapper dcJobConfigMapper;


    @SneakyThrows
    public EbayItemReader(DcBaseCompanyAccountMapper dcBaseCompanyAccountMapper,
                          DcJobConfigMapper dcJobConfigMapper) {
        this.dcBaseCompanyAccountMapper = dcBaseCompanyAccountMapper;
        this.dcJobConfigMapper = dcJobConfigMapper;
        this.init();
    }

    private void init() throws Exception {
        dcBaseCompanyAccountList = dcBaseCompanyAccountMapper.queryPageEbay();
        dcJobConfig = dcJobConfigMapper.selectByName("base-sync-finance-ebay-ad");
        this.pageNum = dcJobConfig.getPageNum();
        this.nextAccount();
    }


    @Override
    public EbayItem read() throws Exception {
        if (ListUtil.isNotEmpty(accountEntryTypeList)) {
            AccountEntryType accountEntryType = accountEntryTypeList.remove(0);
            log.info("读取一条广告费用记录");
            return new EbayItem(accountEntryType, dcBaseCompanyAccount);
        } else {
            nextPage();
            if (ListUtil.isNotEmpty(accountEntryTypeList)) {
                AccountEntryType accountEntryType = accountEntryTypeList.remove(0);
                log.info("读取一条广告费用记录");
                return new EbayItem(accountEntryType, dcBaseCompanyAccount);

            } else {
                nextAccount();
                if (ListUtil.isNotEmpty(accountEntryTypeList)) {
                    AccountEntryType accountEntryType = accountEntryTypeList.remove(0);
                    log.info("读取一条广告费用记录");
                    return new EbayItem(accountEntryType, dcBaseCompanyAccount);
                } else {
                    return null;
                }
            }
        }
    }


    private void nextPage() throws Exception {
        this.pageNum++;
        refreshPage();
    }

    private void nextAccount() throws Exception {

        this.pageNum = 1;
        if (ListUtil.isNotEmpty(dcBaseCompanyAccountList)) {
            dcBaseCompanyAccount = dcBaseCompanyAccountList.remove(0);
            this.refreshAccountCall();
            this.refreshPage();
        }
    }

    void refreshAccountCall() {
        ApiContext apiContext = new ApiContext();
        ApiCredential cred = apiContext.getApiCredential();
        cred.seteBayToken(dcBaseCompanyAccount.getSoapAuthToken());
        apiContext.setApiServerUrl("https://api.ebay.com/wsapi");

        getAccountCall = new GetAccountCall(apiContext);
        //详情类型 返回所有
        getAccountCall.setDetailLevel(new DetailLevelCodeType[]{DetailLevelCodeType.RETURN_ALL});
        //类型
        getAccountCall.setViewType(AccountHistorySelectionCodeType.BETWEEN_SPECIFIED_DATES);

        //开始时间
        Calendar startCal = Calendar.getInstance();
        startCal.setTime(Date.from(dcJobConfig.getStartTime().atZone(ZoneId.systemDefault()).toInstant()));

        //结束时间
        Calendar endCal = Calendar.getInstance();
        endCal.setTime(Date.from(dcJobConfig.getEndTime().atZone(ZoneId.systemDefault()).toInstant()));

        TimeFilter timeFilter = new TimeFilter(startCal, endCal);
        getAccountCall.setViewPeriod(timeFilter);

        //endCal
        getAccountCall.setInvoiceDate(endCal);
    }

    private void refreshPage() throws Exception {
        PaginationType paginationType = new PaginationType();
        paginationType.setPageNumber(pageNum);
        getAccountCall.setPagination(new PaginationType());
        accountEntryTypeList = Lists.newArrayList(getAccountCall.getAccount());
    }
}
