package com.blt.other.module.cost.service.impl;

import com.bailuntec.common.ListUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.blt.other.common.exception.BizRuntimeException;
import com.blt.other.module.auth.dao.OaUserMapper;
import com.blt.other.module.sys.model.CostReviewer;
import com.blt.other.module.auth.model.OaUser;
import com.blt.other.module.cost.dao.CostCurrentReviewerMapper;
import com.blt.other.module.cost.dao.CostDao;
import com.blt.other.module.cost.dto.request.TransferReviewReq;
import com.blt.other.module.cost.model.CostCurrentReviewer;
import com.blt.other.module.cost.model.CostDomain;
import com.blt.other.module.cost.service.CostLogService;
import com.blt.other.module.cost.service.ICostCurrentReviewerService;
import com.google.common.collect.Lists;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>
 * 费用单当前审核人 服务实现类
 * </p>
 *
 * @author robbendev
 * @since 2020-12-16
 */
@Service
public class CostCurrentReviewerServiceImpl extends ServiceImpl<CostCurrentReviewerMapper, CostCurrentReviewer> implements ICostCurrentReviewerService {

    @Resource
    OaUserMapper oaUserMapper;
    @Resource
    CostLogService costLogService;
    @Resource
    CostDao costDao;


    @Override
    public Boolean canAudit(Integer oaUserId, String costNo) {
        List<CostCurrentReviewer> costCurrentReviewerList = baseMapper.selectList(new LambdaQueryWrapper<CostCurrentReviewer>()
                .eq(CostCurrentReviewer::getCostNo, costNo));
        return ListUtil.isNotEmpty(costCurrentReviewerList) && costCurrentReviewerList.stream()
                .map(CostCurrentReviewer::getOaUserId).anyMatch(id -> id.equals(oaUserId));
    }

    @Override
    public void updateByCostNoAndReviewer(String costNo, List<CostReviewer> costReviewerList) {

        this.remove(new LambdaQueryWrapper<CostCurrentReviewer>()
                .eq(CostCurrentReviewer::getCostNo, costNo));

        List<CostCurrentReviewer> costCurrentReviewerList = costReviewerList.stream()
                .map(costReviewer -> {
                    CostCurrentReviewer costCurrentReviewer = new CostCurrentReviewer();
                    costCurrentReviewer.setCostNo(costNo);
                    costCurrentReviewer.setOaUserId(costReviewer.getReviewerUserId());
                    costCurrentReviewer.setUsername(costReviewer.getReviewerUserName());
                    return costCurrentReviewer;
                })
                .collect(Collectors.toList());

        this.saveBatch(costCurrentReviewerList);
    }

    @Override
    public void updateByCostNoAndOaUserId(String costNo, List<Integer> oaUserid) {

        this.remove(new LambdaQueryWrapper<CostCurrentReviewer>()
                .eq(CostCurrentReviewer::getCostNo, costNo));

        List<CostCurrentReviewer> costCurrentReviewerList =
                oaUserMapper.selectList(new LambdaQueryWrapper<OaUser>()
                        .in(OaUser::getOaUserId, oaUserid)).stream()
                        .map(oaUser -> {
                            CostCurrentReviewer costCurrentReviewer = new CostCurrentReviewer();
                            costCurrentReviewer.setCostNo(costNo);
                            costCurrentReviewer.setOaUserId(oaUser.getOaUserId());
                            costCurrentReviewer.setUsername(oaUser.getUserName());
                            return costCurrentReviewer;
                        })
                        .collect(Collectors.toList());

        this.saveBatch(costCurrentReviewerList);
    }

    @Override
    public void transferReview(TransferReviewReq req) {

        CostDomain costDomain = costDao.selectByCostNo(req.getCostNo());

        if (!Lists.newArrayList(CostDomain.STATUS_HR_CHECK,
                CostDomain.STATUS_DEPARTMENT_CHECK,
                CostDomain.STATUS_FINANCIAL_CHECK,
                CostDomain.STATUS_FINAL_CHECK)
                .contains(costDomain.getCostStatus())) {
            throw new BizRuntimeException("invalid cost status");
        }

        this.updateByCostNoAndOaUserId(req.getCostNo(), req.getOaUserIdList());
        List<OaUser> oaUserList = oaUserMapper.selectList(new LambdaQueryWrapper<OaUser>()
                .in(OaUser::getOaUserId, req.getOaUserIdList()));

        costLogService.save(req.getCostNo(), req.getCurrentUserId(), "转审核，审核人:" + oaUserList.stream().map(OaUser::getUserName).collect(Collectors.joining(",")));
    }
}
