package com.blt.other.module.cost.service.impl.costcheck;

import com.blt.other.common.exception.BizRuntimeException;
import com.blt.other.module.auth.model.CostReviewer;
import com.blt.other.module.cost.dao.CostCompanyDao;
import com.blt.other.module.cost.dao.CostLogDao;
import com.blt.other.module.cost.model.CostDomain;
import com.blt.other.module.cost.model.CostTemplate;
import com.blt.other.module.database.model.CostCompanyDomain;
import com.blt.other.module.database.model.CostLogDomain;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;

/**
 * <p>
 *
 * </p>
 *
 * @author robbendev
 * @since 2020/10/30 2:57 下午
 */
@Slf4j
@Component
public class FinalCheckState extends CostState {

    @Resource
    UnPayState unPayState;
    @Resource
    CostCompanyDao costCompanyDao;
    @Resource
    CostLogDao costLogDao;

    private void autoPass() {
        CostDomain costDomain = costContext.costDomain;

        costDomain.setCostStatus(CostDomain.STATUS_UN_PAY);
        costDomain.setLastModifyDate(LocalDateTime.now());
        costDao.updateById(costDomain);
        costLogService.saveByManage(costDomain.getCostNo(), "最终审核自动通过", CostLogDomain.FINAL_AUTO_PASS);

        //流转状态
        nextState(unPayState);
        //通知财务系统
        costContext.costService.toFinancial(costDomain);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void handle() {
        CostDomain costDomain = costContext.costDomain;
        Integer currentUserId = costContext.currentUserId;
        CostTemplate costTemplate = costTemplateService.queryDetail(costDomain.getCostTemplateId());
        CostCompanyDomain costCompany = costCompanyDao.selectByNo(costDomain.getCompanyNo());


        //check status
        if (!costDomain.getCostStatus().equals(CostDomain.STATUS_FINAL_CHECK)) {
            throw new BizRuntimeException("invalid status");
        }

        //部门审核人和最终审核人是同一个人，最终审核人自动通过。
        CostLogDomain costLogDomain = costLogDao.selectDepartmentCheckLog(costDomain.getCostNo());
        if (costLogDomain != null && costLogDomain.getUpdateUserid().equals(currentUserId)) {
            log.info("费用单:{}部门审核人和最终审核人是同一个人，最终审核人自动通过。", costDomain.getCostNo());
            this.autoPass();
            return;
        }

        //如果不需要审核 并且主体不是工会 直接通过
        if (!costTemplate.shouldFinalCheck(costDomain)) {
            log.info("费用单:{}不需要最终审核审核，并且主体不是工会。最终审核人自动通过", costDomain.getCostNo());
            this.autoPass();
            return;
        }

        //需要自动审核
        if (costTemplate.shouldFinalAutoCheck(costDomain)) {
            //自动审核通过
            if (this.autoCheck(costDomain)) {
                log.info("费用单:{}需要自动审核并且自动审核规则校验通过.最终审核人自动通过", costDomain.getCostNo());
                this.autoPass();
                return;
            }
        }

        if (costContext.currentUserId != null) {
            //人工审核通过
            if (costReviewerMapper.queryOne(costCompany.getId(), CostReviewer.finalReviewer, currentUserId) != null) {
                costDomain.setCostStatus(CostDomain.STATUS_UN_PAY);
                costDomain.setLastModifyDate(LocalDateTime.now());
                costDao.updateById(costDomain);

                log.info("费用单:{}最终审核人工审核通过", costDomain.getCostNo());
                costLogService.save(costDomain.getCostNo(), currentUserId, "最终审核通过", CostLogDomain.FINAL_MANUAL_PASS);

                //流转状态
                nextState(unPayState);
                //通知财务系统
                costContext.costService.toFinancial(costDomain);
            }

            //人工审核没权限
            else {
                throw new BizRuntimeException("current user no authority");
            }
        }
    }

    @Override
    public void refuse(String rejectReason) {
        CostDomain costDomain = costContext.costDomain;
        Integer currentUserId = costContext.currentUserId;

        CostCompanyDomain costCompany = costCompanyDao.selectByNo(costDomain.getCompanyNo());

        //check status
        if (!costDomain.getCostStatus().equals(CostDomain.STATUS_FINAL_CHECK)) {
            throw new BizRuntimeException("invalid status");
        }

        //人工审核通过
        if (costReviewerMapper.queryOne(costCompany.getId(), CostReviewer.finalReviewer, currentUserId) == null) {
            throw new BizRuntimeException("current user no authority");
        }

        costContext.costService.reject(costDomain.getCostNo());

        log.info("费用单:{}最终审核拒绝,理由:{}", costDomain.getCostNo(), rejectReason);
        costLogService.save(costDomain.getCostNo(), currentUserId, "最终审核拒绝,理由:" + rejectReason, CostLogDomain.TYPE_UPDATE);

    }
}
