package com.blt.other.common.rabbitmq.declare;

import com.bailuntec.common.BeanUtils;
import com.blt.other.common.rabbitmq.model.MqQueue;
import com.rabbitmq.client.AMQP;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.AmqpException;
import org.springframework.amqp.core.Queue;
import org.springframework.amqp.rabbit.support.RabbitExceptionTranslator;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

/**
 * @Author: li.yanlin
 * @Description：
 * @Date: Created in  2021-04-28
 * @Modified by:
 */
@Component
@Slf4j
public class AmQueueDeclare extends AbstractDeclare{
    /**
     * 声明队列
     * 向rabbitMQ服务器声明一个队列
     *
     * @param mqQueue
     * @return
     */
    public Queue declareQueue(MqQueue mqQueue) {
        log.info("the parameter queue is : " + mqQueue.toString());


        Queue queue = new Queue(mqQueue.getName());
        BeanUtils.copyProperties(mqQueue, queue);

        log.info("declare queue is : " + queue.toString());

        super.rabbitAdmin.declareQueue(queue);

        log.info("declare queue success");
        return queue;
    }

    /**
     * 清空队列中的消息
     *
     * @param queueName
     * @return 清楚队列中的消息的个数
     */
    public int purgeQueue(String queueName) {
        if (StringUtils.isEmpty(queueName)) {
            throw new AmqpException("队列名不能为空");
        }
        log.info("purge queue is : " + queueName);
        return super.rabbitAdmin.purgeQueue(queueName);
    }

    /**
     * 判断指定的队列是否存在
     * 1. 如果存在则返回该队列
     * 2. 如果不存在则返回null
     *
     * @param queueName
     * @return true 存在， false 不存在
     */
    public boolean isQueueExist(String queueName) {
        if (StringUtils.isEmpty(queueName)) {
            throw new AmqpException("队列名不能为空");
        }

        log.info("isQueueExist queue is : " + queueName);

        String isExist = super.rabbitAdmin.getRabbitTemplate().execute((channel -> {
            try {
                AMQP.Queue.DeclareOk declareOk = channel.queueDeclarePassive(queueName);
                return declareOk.getQueue();
            } catch (Exception e) {
                if (log.isDebugEnabled()) {
                    throw new AmqpException(e);
                }
                return null;
            }
        }));

        log.info("the queue " + queueName + " is exist : " + isExist);
        return StringUtils.isEmpty(isExist) ? Boolean.FALSE : Boolean.TRUE;
    }

    /**
     * 从rabbitMQ服务器中删除指定的队列
     *
     * @param queueName
     * @return
     */
    public boolean deleteQueue(String queueName) {
        log.info("delete queue is :" + queueName);

        if (StringUtils.isEmpty(queueName)) {
            throw new AmqpException("队列名不能为空");
        }

        return super.rabbitAdmin.deleteQueue(queueName);
    }

    /**
     * 从rabbitMQ服务器中删除指定的队列
     *
     * @param queueName 队列名称
     * @param unused    队列是否在使用，如果设置为true则该队列只能在没有被使用的情况下才能删除
     * @param empty     队列是否为空，如果设置为true则该队列只能在该队列没有消息时才会被删除
     */
    public void deleteQueue(String queueName, boolean unused, boolean empty) {
        log.info("delete queue is : { queueName : '" + queueName
                + "' , unused: '" + unused + "' , empty:'" + empty + "'}");

        if (StringUtils.isEmpty(queueName)) {
            throw new AmqpException("队列名不能为空");
        }

        super.rabbitAdmin.deleteQueue(queueName, unused, empty);
    }
}
