package com.blt.other.module.cost.service.impl.costcheck;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.blt.other.common.config.property.WxWebHook;
import com.blt.other.common.exception.BizRuntimeException;
import com.blt.other.database.model.CostCompanyDomain;
import com.blt.other.database.model.CostLogDomain;
import com.blt.other.database.model.UserDomain;
import com.blt.other.module.auth.dao.UserDao;
import com.blt.other.module.cost.dao.CostCurrentReviewerMapper;
import com.blt.other.module.cost.dao.CostLogDao;
import com.blt.other.module.cost.model.ApprovalHistoryDomain;
import com.blt.other.module.cost.model.CostCurrentReviewer;
import com.blt.other.module.cost.model.CostDomain;
import com.blt.other.module.cost.model.CostTemplate;
import com.blt.other.module.cost.service.ApprovalHistoryService;
import com.blt.other.module.sys.model.CostReviewer;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Scope;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.lang.reflect.Array;
import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>
 *
 * </p>
 *
 * @author robbendev
 * @since 2020/10/30 2:57 下午
 */
@Slf4j
@Component
@Scope("prototype")
public class FinalCheckState extends CostState {

    @Resource
    UnPayState unPayState;
    @Resource
    CostLogDao costLogDao;
    @Resource
    CostCurrentReviewerMapper costCurrentReviewerMapper;
    @Resource
    ApprovalHistoryService approvalHistoryService;

    private void autoPass() {
        CostDomain costDomain = costContext.costDomain;

        costDomain.setCostStatus(CostDomain.STATUS_UN_PAY);
        costDomain.setLastModifyDate(LocalDateTime.now());
        costDao.updateById(costDomain);

        List<CostCurrentReviewer> costCurrentReviewerList = costCurrentReviewerMapper.selectByCostNo(costDomain.getCostNo());
        costLogService.saveByManage(costDomain.getCostNo(), "最终审核自动通过,当前审核人:" + costCurrentReviewerList.stream().map(CostCurrentReviewer::getUsername)
                .collect(Collectors.joining(",")), CostLogDomain.DEPARTMENT_AUTO_PASS);
        //流转状态
        nextState(unPayState);
        //通知财务系统
        costContext.costService.toFinancial(costDomain);
    }

    @Resource
    WxWebHook webHook;
    @Resource
    RestTemplate restTemplate;
    @Resource
    UserDao userDao;

    private void sendWxMsg(CostDomain costDomain, Integer currentUserId) {
        UserDomain userDomain = userDao.selectByuserid(currentUserId);
        DateTimeFormatter dtf = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        String msg = String.format("{\n" +
                        "    \"msgtype\": \"markdown\",\n" +
                        "    \"markdown\": {\n" +
                        "        \"content\": \"单号【<font color=\\\"info\\\">%s</font>】通过人工操作最终审核。\\n" +
                        ">金额: <font color=\\\"info\\\">%s</font>\\n" +
                        ">审核人: <font color=\\\"info\\\">%s</font>\\n" +
                        ">审核时间: <font color=\\\"info\\\">%s</font>\"" +
                        "    }\n" +
                        "}", costDomain.getCostNo(),
                "CNY " + costDomain.getAmountRmb(),
                userDomain == null ? currentUserId : userDomain.getUsername(),
                LocalDateTime.now().format(dtf));
        restTemplate.getMessageConverters().set(1, new StringHttpMessageConverter(StandardCharsets.UTF_8));
        restTemplate.postForEntity(webHook.getUrl() + webHook.getFinalReview(), msg, Object.class);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void handle() {
        CostDomain costDomain = costContext.costDomain;
        Integer currentUserId = costContext.currentUserId;
        CostTemplate costTemplate = costTemplateService.queryDetail(costDomain.getCostTemplateId());


        //check status
        if (!costDomain.getCostStatus().equals(CostDomain.STATUS_FINAL_CHECK)) {
            throw new BizRuntimeException("invalid status");
        }

        //部门审核人和最终审核人是同一个人，最终审核人自动通过。
        CostLogDomain costLogDomain = costLogDao.selectDepartmentCheckLog(costDomain.getCostNo());
        List<CostCurrentReviewer> costCurrentReviewerList = costCurrentReviewerMapper.selectByCostNo(costDomain.getCostNo());
        if (costLogDomain != null && costCurrentReviewerList.stream().map(CostCurrentReviewer::getOaUserId).collect(Collectors.toList()).contains(costLogDomain.getUpdateUserid())) {

            log.info("费用单:{}部门审核人和最终审核人是同一个人，最终审核人自动通过。", costDomain.getCostNo());
            this.autoPass();
            return;
        }

        //如果不需要审核 并且主体不是工会 直接通过
        if (!costTemplate.shouldFinalCheck(costDomain)) {
            log.info("费用单:{}不需要最终审核审核，并且主体不是工会。最终审核人自动通过", costDomain.getCostNo());
            this.autoPass();
            return;
        }

        //需要自动审核
        if (costTemplate.shouldFinalAutoCheck(costDomain)) {
            //自动审核通过
            if (this.autoCheck(costDomain)) {
                log.info("费用单:{}需要自动审核并且自动审核规则校验通过.最终审核人自动通过", costDomain.getCostNo());
                this.autoPass();
                return;
            }
        }

        if (costContext.currentUserId != null) {
            //人工审核通过
            if (costCurrentReviewerService.canAudit(currentUserId, costDomain.getCostNo())) {
                ApprovalHistoryDomain approvalHistoryDomain = ApprovalHistoryDomain.builder()
                        .approvalUserId(currentUserId)
                        .approvalTime(LocalDateTime.now())
                        .costStatus(costDomain.getCostStatus())
                        .costNo(costDomain.getCostNo())
                        .isPassed(true)
                        .build();
                costDomain.setCostStatus(CostDomain.STATUS_UN_PAY);
                costDomain.setLastModifyDate(LocalDateTime.now());
                costDao.updateById(costDomain);

                log.info("费用单:{}最终审核人工审核通过", costDomain.getCostNo());
                costLogService.save(costDomain.getCostNo(), currentUserId, "最终审核通过", CostLogDomain.FINAL_MANUAL_PASS);

                approvalHistoryService.save(approvalHistoryDomain);

                sendWxMsg(costDomain, currentUserId);
                //流转状态
                nextState(unPayState);
                //通知财务系统
                costContext.costService.toFinancial(costDomain);
            }

            //人工审核没权限
            else {
                throw new BizRuntimeException("current user no authority");
            }
        }
    }

    @Override
    public void refuse(String rejectReason) {
        CostDomain costDomain = costContext.costDomain;
        Integer currentUserId = costContext.currentUserId;

        //check status
        if (!costDomain.getCostStatus().equals(CostDomain.STATUS_FINAL_CHECK)) {
            throw new BizRuntimeException("invalid status");
        }

        //人工审核通过
        if (!costCurrentReviewerService.canAudit(currentUserId, costDomain.getCostNo())) {
            throw new BizRuntimeException("current user no authority");
        }

        costContext.costService.reject(costDomain.getCostNo());
        ApprovalHistoryDomain approvalHistoryDomain = ApprovalHistoryDomain.builder()
                .approvalUserId(currentUserId)
                .approvalTime(LocalDateTime.now())
                .costStatus(costDomain.getCostStatus())
                .costNo(costDomain.getCostNo())
                .isPassed(false)
                .build();
        approvalHistoryService.save(approvalHistoryDomain);
        log.info("费用单:{}最终审核拒绝,理由:{}", costDomain.getCostNo(), rejectReason);
        costLogService.save(costDomain.getCostNo(), currentUserId, "最终审核拒绝,理由:" + rejectReason, CostLogDomain.TYPE_UPDATE);

    }

    @Override
    public void updateCurrentReviewer() {
        CostDomain costDomain = costContext.costDomain;
        CostCompanyDomain costCompanyDomain = costCompanyDao.selectByNo(costDomain.getCompanyNo());


        List<CostReviewer> costReviewerList = costReviewerMapper.selectList(new LambdaQueryWrapper<CostReviewer>()
                .eq(CostReviewer::getReferId, costCompanyDomain.getId())
                .eq(CostReviewer::getType, CostReviewer.finalReviewer));

        costCurrentReviewerService.updateByCostNoAndReviewer(costDomain.getCostNo(), costReviewerList);
//        costLogService.save(costDomain.getCostNo(), costContext.currentUserId, "最终审核人:" + costReviewerList.stream().map(CostReviewer::getReviewerUserName).collect(Collectors.joining(",")));

    }
}
