package com.blt.other.module.sys.service.impl;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.annotation.ExcelProperty;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.event.AnalysisEventListener;
import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.blt.other.module.auth.dao.OaCompanyMapper;
import com.blt.other.module.auth.dao.OaDepartmentMapper;
import com.blt.other.module.auth.dao.OaUserMapper;
import com.blt.other.module.auth.model.OaUser;
import com.blt.other.module.auth.service.IOaCompanyService;
import com.blt.other.module.cost.dao.SpecDepartmentCheckConfigMapper;
import com.blt.other.module.cost.dto.request.*;
import com.blt.other.module.cost.service.CostCompanyService;
import com.blt.other.module.sys.model.SpecDepartmentCheckConfig;
import com.blt.other.module.sys.service.ISpecDepartmentCheckConfigService;
import io.swagger.annotations.ApiModelProperty;
import lombok.Data;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>
 * 服务实现类
 * </p>
 *
 * @author robbendev
 * @since 2020-12-16
 */
@Service
public class SpecDepartmentCheckConfigServiceImpl extends ServiceImpl<SpecDepartmentCheckConfigMapper, SpecDepartmentCheckConfig> implements ISpecDepartmentCheckConfigService {

    @Resource
    OaUserMapper oaUserMapper;
    @Resource
    private CostCompanyService costCompanyService;
    @Resource
    private IOaCompanyService oaCompanyService;
    @Resource
    private OaCompanyMapper oaCompanyMapper;

    private <T extends SpecDepartmentCheckBaseQuery> LambdaQueryWrapper<SpecDepartmentCheckConfig> buildQueryWrapper(T req) {
        LambdaQueryWrapper<SpecDepartmentCheckConfig> wrapper = new LambdaQueryWrapper<>();
        if (req.getCostCompanyId() != null) {
            wrapper.eq(SpecDepartmentCheckConfig::getCostCompanyId, req.getCostCompanyId());
        }
        if (req.getOaCompanyId() != null) {
            wrapper.eq(SpecDepartmentCheckConfig::getOaCompanyId, req.getOaCompanyId());
        }
        if (req.getReviewerDepartmentId() != null) {
            wrapper.eq(SpecDepartmentCheckConfig::getReviewerDepartmentId, req.getReviewerDepartmentId());
        }
        if (req.getUpdateUserId() != null) {
            wrapper.eq(SpecDepartmentCheckConfig::getUpdateUserId, req.getUpdateUserId());
        }
        if (req.getLastUpdateTimeStart() != null) {
            wrapper.ge(SpecDepartmentCheckConfig::getLastUpdateTime, req.getLastUpdateTimeStart());
        }
        if (req.getLastUpdateTimeStartEnd() != null) {
            wrapper.le(SpecDepartmentCheckConfig::getLastUpdateTime, req.getLastUpdateTimeStartEnd());
        }
        return wrapper;
    }


    @Override
    public Page<SpecDepartmentCheckConfig> queryPage(SpecDepartmentCheckQueryPageReq req) {
        IPage<SpecDepartmentCheckConfig> page = new Page<>(req.getPageNum(), req.getPageSize());
        return (Page<SpecDepartmentCheckConfig>) baseMapper.selectPage(page, this.buildQueryWrapper(req));
    }

    @Override
    public void batchUpdateReviewer(BatchUpdateReviewerReq req) {

        OaUser oaUser = oaUserMapper.selectByOaUserId(req.getUpdateUserId());

        List<SpecDepartmentCheckConfig> specDepartmentCheckConfigList = this.listByIds(req.getIdList());
        specDepartmentCheckConfigList.forEach(specDepartmentCheckConfig -> {
            specDepartmentCheckConfig.setLastUpdateTime(LocalDateTime.now());
            specDepartmentCheckConfig.setUpdateUserId(req.getUpdateUserId());
            specDepartmentCheckConfig.setUpdateUserName(oaUser.getUserName());
        });
    }

    @Resource
    OaDepartmentMapper oaDepartmentMapper;

    @Override
    public void add(SpecDepartmentCheckAddReq req) {
        SpecDepartmentCheckConfig specDepartmentCheckConfig = new SpecDepartmentCheckConfig();
        specDepartmentCheckConfig.setReviewerDepartmentId(req.getReviewerDepartmentId());
        specDepartmentCheckConfig.setReviewerDepartmentName(oaDepartmentMapper.selectByDepartmentId(req.getReviewerDepartmentId()).getName());

        specDepartmentCheckConfig.setOaCompanyId(req.getOaCompanyId());
        specDepartmentCheckConfig.setOaCompany(oaCompanyMapper.selectByCompanyId(req.getOaCompanyId()).getName());

        specDepartmentCheckConfig.setCostCompanyId(req.getCostCompanyId());
        specDepartmentCheckConfig.setCostCompanyName(costCompanyService.getById(req.getCostCompanyId()).getCompanyName());

        specDepartmentCheckConfig.setUpdateUserId(req.getUpdateUserId());
        specDepartmentCheckConfig.setUpdateUserName(oaUserMapper.selectByOaUserId(req.getUpdateUserId()).getUserName());

        specDepartmentCheckConfig.setLastUpdateTime(LocalDateTime.now());
        this.save(specDepartmentCheckConfig);
    }

    @Override
    public void importExcel(SpecDepartmentCheckImportExcelReq req) throws IOException {
        SpecDepartmentCheckConfigExcelItemListener listener = new SpecDepartmentCheckConfigExcelItemListener(
                this,
                costCompanyService,
                oaCompanyService,
                oaDepartmentMapper);
        EasyExcel.read(req.getMultipartFile().getInputStream(), SpecDepartmentCheckConfigImportExcelItem.class, listener).sheet().doRead();
    }

    @Override
    public void exportExcel(HttpServletResponse response, SpecDepartmentCheckExportExcelReq req) throws IOException {
        response.setContentType("application/vnd.ms-excel");
        response.setCharacterEncoding("utf-8");

        String fileName = URLEncoder.encode("模版", "UTF-8");
        response.setHeader("Content-disposition", "attachment;filename=" + fileName + ".xlsx");

        List<SpecDepartmentCheckConfigExportExcelItem> aiList = baseMapper.selectList(this.buildQueryWrapper(req)).stream()
                .map(specDepartmentCheckConfig -> {
                    SpecDepartmentCheckConfigExportExcelItem item = new SpecDepartmentCheckConfigExportExcelItem();
                    item.setCostCompanyName(specDepartmentCheckConfig.getCostCompanyName());
                    item.setOaCompany(specDepartmentCheckConfig.getOaCompany());
                    item.setReviewerDepartmentName(specDepartmentCheckConfig.getReviewerDepartmentName());
                    return item;
                })
                .collect(Collectors.toList());
        EasyExcel.write(response.getOutputStream(), SpecDepartmentCheckConfigExportExcelItem.class).sheet("sheet").doWrite(aiList);
    }

    @Override
    public void modify(SpecDepartmentCheckModifyReq req) {

        SpecDepartmentCheckConfig specDepartmentCheckConfig = this.getById(req.getId());
        if (req.getReviewerDepartmentId() != null) {
            specDepartmentCheckConfig.setReviewerDepartmentId(req.getReviewerDepartmentId());
            specDepartmentCheckConfig.setReviewerDepartmentName(oaDepartmentMapper.selectByDepartmentId(req.getReviewerDepartmentId()).getName());
        }

        if (req.getOaCompanyId() != null) {
            specDepartmentCheckConfig.setOaCompanyId(req.getOaCompanyId());
            specDepartmentCheckConfig.setOaCompany(oaCompanyMapper.selectByCompanyId(req.getOaCompanyId()).getName());
        }

        if (req.getCostCompanyId() != null) {
            specDepartmentCheckConfig.setCostCompanyId(req.getCostCompanyId());
            specDepartmentCheckConfig.setCostCompanyName(costCompanyService.getById(req.getCostCompanyId()).getCompanyName());
        }

        specDepartmentCheckConfig.setUpdateUserId(req.getUpdateUserId());
        specDepartmentCheckConfig.setUpdateUserName(oaUserMapper.selectByOaUserId(req.getUpdateUserId()).getUserName());
        specDepartmentCheckConfig.setLastUpdateTime(LocalDateTime.now());
        this.updateById(specDepartmentCheckConfig);
    }

    @Override
    public void delete(SpecDepartmentCheckDeleteReq req) {
        this.removeById(req.getId());
    }


    @Data
    static class SpecDepartmentCheckConfigExportExcelItem {

        @ExcelProperty("付款财务主体")
        @ApiModelProperty(value = "付款财务主体")
        private String costCompanyName;

        @ExcelProperty("申请人oa公司")
        @ApiModelProperty(value = "申请人oa公司")
        private String oaCompany;

        @ExcelProperty(value = "特殊审核部门")
        @ApiModelProperty(value = "特殊审核部门")
        private String reviewerDepartmentName;

    }

    @Data
    static class SpecDepartmentCheckConfigImportExcelItem {

        @ExcelProperty("付款财务主体")
        @ApiModelProperty(value = "付款财务主体")
        private String costCompanyName;

        @ExcelProperty("申请人oa公司")
        @ApiModelProperty(value = "申请人oa公司")
        private String oaCompany;

        @ExcelProperty(value = "特殊审核部门")
        @ApiModelProperty(value = "特殊审核部门")
        private String reviewerDepartmentName;

    }

    static class SpecDepartmentCheckConfigExcelItemListener extends AnalysisEventListener<SpecDepartmentCheckConfigImportExcelItem> {

        private final ISpecDepartmentCheckConfigService specDepartmentCheckConfigService;
        private final CostCompanyService costCompanyService;
        private final IOaCompanyService oaCompanyService;
        private final OaDepartmentMapper oaDepartmentMapper;


        private static final Logger LOGGER = LoggerFactory.getLogger(SpecDepartmentCheckConfigExcelItemListener.class);

        SpecDepartmentCheckConfigExcelItemListener(ISpecDepartmentCheckConfigService specDepartmentCheckConfigService,
                                                   CostCompanyService costCompanyService,
                                                   IOaCompanyService oaCompanyService,
                                                   OaDepartmentMapper oaDepartmentMapper) {
            this.specDepartmentCheckConfigService = specDepartmentCheckConfigService;
            this.costCompanyService = costCompanyService;
            this.oaCompanyService = oaCompanyService;
            this.oaDepartmentMapper = oaDepartmentMapper;
        }

        private static final int BATCH_COUNT = 2000;
        List<SpecDepartmentCheckConfigImportExcelItem> list = new ArrayList<>();

        @Override
        public void invoke(SpecDepartmentCheckConfigImportExcelItem data, AnalysisContext context) {
            LOGGER.info("解析到一条数据:{}", JSON.toJSONString(data));
            list.add(data);
            if (list.size() >= BATCH_COUNT) {
                syncData();
                list.clear();
            }
        }

        @Override
        public void doAfterAllAnalysed(AnalysisContext context) {
            syncData();
            LOGGER.info("所有数据解析完成！");
        }

        private void syncData() {
            LOGGER.info("{}条数据，开始存储数据库！", list.size());
            List<SpecDepartmentCheckConfig> specDepartmentCheckConfigList = list.stream()
                    .map(specDepartmentCheckConfigImportExcelItem -> {
                        SpecDepartmentCheckConfig specDepartmentCheckConfig = new SpecDepartmentCheckConfig();

                        specDepartmentCheckConfig.setCostCompanyName(specDepartmentCheckConfigImportExcelItem.getCostCompanyName());
                        specDepartmentCheckConfig.setCostCompanyId(costCompanyService.costCompanyMap().get(specDepartmentCheckConfigImportExcelItem.getCostCompanyName()).getId());

                        specDepartmentCheckConfig.setOaCompany(specDepartmentCheckConfig.getOaCompany());
                        specDepartmentCheckConfig.setOaCompanyId(oaCompanyService.companyDict().get(specDepartmentCheckConfigImportExcelItem.getOaCompany()).getOaCompanyId());

                        specDepartmentCheckConfig.setReviewerDepartmentId(oaDepartmentMapper.selectByName(specDepartmentCheckConfigImportExcelItem.getReviewerDepartmentName()).getDepartmentId());
                        specDepartmentCheckConfig.setReviewerDepartmentName(specDepartmentCheckConfigImportExcelItem.getReviewerDepartmentName());

                        specDepartmentCheckConfig.setLastUpdateTime(LocalDateTime.now());
                        return specDepartmentCheckConfig;
                    })
                    .filter(specDepartmentCheckConfig -> specDepartmentCheckConfig.getCostCompanyId() != null)
                    .filter(specDepartmentCheckConfig -> specDepartmentCheckConfig.getOaCompanyId() != null)
                    .filter(specDepartmentCheckConfig -> specDepartmentCheckConfig.getReviewerDepartmentId() != null)
                    .collect(Collectors.toList());


            specDepartmentCheckConfigService.saveBatch(specDepartmentCheckConfigList);
            LOGGER.info("存储数据库成功！");
        }
    }

}
