package com.blt.other.module.purchasing.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.blt.other.common.util.BuyUtils;
import com.blt.other.common.util.CurUtils;
import com.blt.other.common.util.MoneyUtil;
import com.blt.other.common.util.PathUtil;
import com.blt.other.module.auth.dao.UserDao;
import com.blt.other.module.cost.dao.CostCompanyDao;
import com.blt.other.module.cost.dao.CostDao;
import com.blt.other.module.cost.dao.CostPlanDao;
import com.blt.other.module.cost.dao.CostPlanTempDao;
import com.blt.other.module.cost.model.CostDomain;
import com.blt.other.module.cost.service.CostLogService;
import com.blt.other.module.cost.service.CostPlanService;
import com.blt.other.module.cost.service.CostPlanTempService;
import com.blt.other.module.cost.service.CostService;
import com.blt.other.module.cost.utils.CostUtils;
import com.blt.other.module.database.mapper.StatusMapper;
import com.blt.other.module.database.model.*;
import com.blt.other.module.purchasing.dao.BuyDao;
import com.blt.other.module.purchasing.dao.BuyDetailDao;
import com.blt.other.module.purchasing.dto.*;
import com.blt.other.module.purchasing.service.BuyLogService;
import com.blt.other.module.purchasing.service.BuyPlanService;
import com.blt.other.module.purchasing.service.BuyService;
import com.blt.other.module.supplier.service.SupplierService;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import org.apache.poi.hssf.usermodel.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.File;
import java.io.FileOutputStream;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;

@Service
public class BuyServiceImpl implements BuyService {

    private static Logger logger = LoggerFactory.getLogger(BuyServiceImpl.class);

    @Autowired
    private BuyDao buyDao;
    @Autowired
    private StatusMapper statusMapper;
    @Autowired
    private BuyPlanService buyPlanService;
    @Autowired
    private BuyDetailDao buyDetailDao;
    @Autowired
    private BuyLogService buyLogService;
    @Autowired
    private CostService costService;
    @Autowired
    private CostLogService costLogService;
    @Autowired
    private CostPlanService costPlanService;
    @Autowired
    private CostPlanDao costPlanDao;
    @Autowired
    private CostPlanTempDao costPlanTempDao;
    @Autowired
    private CostPlanTempService costPlanTempService;
    @Autowired
    private CostDao costDao;
    @Autowired
    private UserDao userDao;

    @Autowired
    private SupplierService supplierService;
    @Autowired
    private CostCompanyDao costCompanyDao;

    @Value("${url.api.getExchangeRateApi}")
    private String getExchangeRateApi;

    /**
     * 分页获取采购订单列表
     *
     * @param pageSize
     * @param page
     * @return
     */
    @Override
    public Map<String, Object> getAllBuyList(int pageSize, int page) {
        logger.warn("分页查询采购单列表：pagesize " + pageSize + " ，page " + page);
        PageHelper.startPage(page, pageSize);
        Map<String, Object> map = new HashMap<>();

        List<BuyDomain> buyDomains = buyDao.selectAll();
        List<BuyListDto> buyListDto = getBuyListDto(buyDomains);
        PageInfo<BuyDomain> pageInfo = new PageInfo<>(buyDomains);
        map.put("buys", buyListDto);
        map.put("pageInfo", pageInfo);
        return map;
    }

    @Override
    public BuyListDto getBuyByBuyno(String buyno) {
        BuyDomain buyDomain = new BuyDomain();
        buyDomain.setBuyno(buyno);
        List<BuyDomain> buyDomainList = buyDao.selectByBuyno(buyDomain);
        List<BuyListDto> buyListDto = getBuyListDto(buyDomainList);
        if (null != buyListDto && buyListDto.size() >= 1) {
            return buyListDto.get(0);
        }
        return null;
    }

    @Override
    public Integer getUpdate(BuyDomain buyDomain) {
        logger.warn("修改采购单信息" + buyDomain);
        String skucode = buyDomain.getSkucode();
        String skuname = buyDomain.getSkuname();
        if (null != skucode && !skucode.contains("[")) {
            skucode = "[" + skucode + "]";
        }
        if (null != skuname && !skuname.contains("[")) {
            skuname = "[" + skuname + "]";
        }
        Integer update = buyDao.update(buyDomain);
        return update;
    }

    @Override
    public BuyDomain getBuyDomain(BuyDomain buyDomain) {
        return buyDao.selectBuy(buyDomain);
    }

    @Override
    public List<BuyListDto> getDtoList(List<BuyDomain> list) {
        return getBuyListDto(list);
    }

    /**
     * 生成采购单
     *
     * @param buy
     * @param buyPlanDomain
     * @return
     */
    @Override
    public Integer saveBuyFromPlanDetailList(List<BuyPlanDetailDomain> buy, BuyPlanDomain buyPlanDomain) {
        BuyDomain buyDomain = new BuyDomain();
        BuyPlanDto buyPlanByNo = buyPlanService.getBuyPlanByNo(buyPlanDomain.getNo());
        buyDomain.setNo(buyPlanByNo.getNo());
        // 生成唯一、易识别的采购单号
        buyDomain.setBuyno(BuyUtils.createBuyNo(this));
        buyDomain.setBuystatus(0);
        buyDomain.setWarehouseid(buyPlanByNo.getWarehouseid());
        buyDomain.setWarehousename(buyPlanByNo.getWarehousename());
        buyDomain.setSupplierid(buy.get(0).getSupplierid());
        buyDomain.setSuppliername(buy.get(0).getSuppliername());
        buyDomain.setSupplierNo(buy.get(0).getSupplierNo());
        SupplierDomain supplierDomain = supplierService.getSupplierByid(buy.get(0).getSupplierid());
        buyDomain.setSupplierBankname(supplierDomain.getBankname());
        buyDomain.setSupplierCardno(supplierDomain.getCardno());
        buyDomain.setSupplierCardusername(supplierDomain.getCardusername());
        buyDomain.setSkuTypeName(supplierDomain.getSkuTypeName());
        buyDomain.setCreatetime(new Date());
        buyDomain.setLastupdateuserid(buyPlanDomain.getAudituserid());
        buyDomain.setLastupdateusername(buyPlanDomain.getAuditusername());
        buyDomain.setLastupdatetime(new Date());
        buyDomain.setPaystatus(0);
        buyDomain.setSendstatus(0);
        buyDomain.setCreateuserid(buyPlanByNo.getCreateuserid());
        buyDomain.setCreateusername(buyPlanByNo.getCreateusername());
        buyDomain.setNote(buyPlanByNo.getRemark());
        buyDomain.setDepartmentname(buyPlanByNo.getDepartmentname());
        buyDomain.setCompany(buyPlanByNo.getCompany());
        CostCompanyDomain companyDomain = costCompanyDao.selectByName(buyPlanByNo.getCompany());
        buyDomain.setCompanyValue(companyDomain.getValue());
        buyDomain.setBuyType(1);
        if (null != buy) {
            List<String> skucode = new ArrayList<>();
            List<String> skuname = new ArrayList<>();
            BigDecimal amount = new BigDecimal(0);
            Integer count = 0;
            for (BuyPlanDetailDomain detail : buy) {
                skucode.add(detail.getSkucode());
                skuname.add(detail.getSkuname());
                amount = amount.add(detail.getAmount());
                count = count + detail.getCount();
                // 生成 buy_detail 记录
                BuyDetailDomain buyDetailDomain = new BuyDetailDomain();
                BeanUtils.copyProperties(detail, buyDetailDomain);
                buyDetailDomain.setBuyno(buyDomain.getBuyno());
                Integer insert = buyDetailDao.insert(buyDetailDomain);
            }
            buyDomain.setSkucode(skucode.toString()); // 将 skucode 以 List 集合的方式保存
            buyDomain.setSkuname(skuname.toString()); // 将 skuname 以 list 集合的方式保存
            buyDomain.setAmount(amount); //  遍历获取采购总金额
            buyDomain.setPayamount(BigDecimal.ZERO);
            buyDomain.setCount(count); // 遍历获取采购总数量
        }
        Integer insert = buyDao.insert(buyDomain);
        // 生成采购单后，记录日志
        if (null != insert && insert == 1) {
            buyLogService.save(buyDomain.getBuyno(), buyPlanDomain.getAudituserid(), "由" + buyDomain.getNo() + "生成采购单");
        }
        return insert;
    }

    /**
     * 获取最新的一条采购单
     *
     * @return
     */
    @Override
    public BuyDomain getLastBuy() {
        return buyDao.selectLastBuy();
    }

    /**
     * 更改采购单号（避免出纳驳回的采购单重新提交出纳系统出现采购单重复）
     *
     * @param oldBuyno
     * @param newBuyno
     * @return
     */
    @Override
    public Integer updateNewBuyno(String oldBuyno, String newBuyno) {
        return buyDao.updateNewBuyno(oldBuyno, newBuyno);
    }

    private List<BuyListDto> getBuyListDto(List<BuyDomain> buyDomains) {
        List<BuyListDto> buyListDtos = new ArrayList<>();
        if (null != buyDomains && buyDomains.size() >= 1) {
            BuyListDto buyListDto = null;
            for (BuyDomain buyDomain : buyDomains) {
                buyListDto = new BuyListDto();
                BeanUtils.copyProperties(buyDomain, buyListDto);
                String skucode = buyListDto.getSkucode();
                String skuname = buyListDto.getSkuname();
                if (null != skucode) {
                    skucode = skucode.substring(1, skucode.lastIndexOf("]"));
                    skuname = skuname.substring(1, skuname.lastIndexOf("]"));
                }
                if (skucode != null && skucode.contains(",")) {
                    // 含有多种商品
                    String[] split = skucode.split(",");
                    skucode = split[0] + "等共 " + split.length + " 项";
                }
                if (skuname != null && skuname.contains(",")) {
                    // 含有多种商品
                    String[] split = skuname.split(",");
                    skuname = split[0] + "等共 " + split.length + " 项";
                }
                // 获取文件名
                String filePath = buyDomain.getFilepath();
                if (null != filePath && filePath.contains("&")) {
                    buyListDto.setFilename(filePath.substring(filePath.lastIndexOf("&") + 1));
                }
                buyListDto.setSkucode(skucode);
                buyListDto.setSkuname(skuname);
                buyListDto.setBuystatusDto(statusMapper.getStatusValue("buystatus", buyDomain.getBuystatus()));
                buyListDto.setDelstatusDto(statusMapper.getStatusValue("delstatus", buyDomain.getDelstatus()));
                buyListDto.setIsurgencyDto(statusMapper.getStatusValue("isurgency", buyDomain.getIsurgency()));
                buyListDto.setPaystatusDto(statusMapper.getStatusValue("paystatus", buyDomain.getPaystatus()));
                buyListDto.setPaytypeDto(statusMapper.getStatusValue("paytype", buyDomain.getPaytype()));
                buyListDto.setSendstatusDto(statusMapper.getStatusValue("sendstatus", buyDomain.getSendstatus()));

                buyListDtos.add(buyListDto);
            }
        }

        return buyListDtos;
    }

    @Override
    @Transactional
    public String buyLinkLendCost(CostDomain linkLendCost, BuyListDto buyListDto, Integer updateUserId) {
        if (null == linkLendCost.getHadPay()) linkLendCost.setHadPay(new BigDecimal(0));
        if (null == linkLendCost.getCounteract()) linkLendCost.setCounteract(new BigDecimal(0));
        if (null == linkLendCost.getLendBalance()) linkLendCost.setLendBalance(new BigDecimal(0));
        BigDecimal subtract = linkLendCost.getAmount().subtract(linkLendCost.getHadPay()); // 关联借支单还需还差额

        BigDecimal exRate; // 采购单币种（现默认CNY）转借支单币种汇率
        if ("CNY".equals(linkLendCost.getDic())) {
            exRate = new BigDecimal("1");
        } else {
            exRate = CurUtils.getCur("CNY", linkLendCost.getDic(), getExchangeRateApi);
        }
        BigDecimal counteract = buyListDto.getAmount().multiply(exRate).setScale(2, BigDecimal.ROUND_HALF_UP); // 冲销金额，RMB
        BigDecimal numZero = new BigDecimal(0);

        linkLendCost.setHadPay(linkLendCost.getHadPay().add(counteract));

        CostDomain backCost = null; // 冲销时自动生成的已完成借还单

        String result = "采购单【" + buyListDto.getBuyno() + "】冲销借支单【" + linkLendCost.getCostNo() + "】成功，金额为" +
                counteract.toString() + linkLendCost.getDic();
        if (counteract.compareTo(numZero) != 1) {
            result = "采购单金额必须大于0，请驳回";
            return result;
        } else if (counteract.compareTo(subtract) == -1) {
            // 冲销金额<借支单待还金额，借支单改为部分还款状态
            linkLendCost.setLendStatus(2);
            // 生成一张已完成的借还单
            backCost = generateBackCost(buyListDto, linkLendCost, counteract, exRate);


            result += "，借支单剩余待还金额" + subtract.subtract(counteract).toString() + linkLendCost.getDic();
        } else if (counteract.compareTo(subtract) == 0) {
            // 冲销金额=借支单待还金额，借支单改为已结清状态
            linkLendCost.setLendStatus(4);
            // 生成一张已完成的借还单
            backCost = generateBackCost(buyListDto, linkLendCost, counteract, exRate);

            result += "，借支单已结清";
        } else if (counteract.compareTo(subtract) == 1) {
            // 冲销金额>借支单待还金额，不允许关联借支单
            /*result = "采购单的金额大于借支单待还金额，不允许进行冲销，请驳回";
            return result;*/
            // 冲销金额>借支单待还金额，借支单改为待补差额状态，生成一张待提交的付款计划单
            linkLendCost.setLendStatus(3);
            // 生成一张已完成的借还单，金额为借支单剩余待还金额，不包含待补差额
            backCost = generateBackCost(buyListDto, linkLendCost, counteract, exRate);
            backCost.setAmount(subtract);
            backCost.setCounteract(subtract);
            backCost.setPayPlanAmount(subtract.divide(exRate, 4));
            backCost.setPayCounteract(subtract.divide(exRate, 4));

            CostPlanDomain payPlan = new CostPlanDomain();
            payPlan.setCompanyNo(linkLendCost.getCompanyNo());
            payPlan.setCompanyName(linkLendCost.getCompanyName());
            UserDomain user = userDao.selectByuserid(buyListDto.getCreateuserid());
            payPlan.setCreateUserid(user.getUserid());
            payPlan.setCreateUsercode(user.getUsercode());
            payPlan.setCreateUsername(user.getUsername());
            payPlan.setCreateTime(new Date());
            payPlan.setSupCostNo(buyListDto.getLinkLendCostNo());
            payPlan.setLendBalance(new BigDecimal(0));
            payPlan.setPayLendBalance(new BigDecimal(0));
            payPlan.setDic(linkLendCost.getDic()); // 差额付款计划单币种跟借支单币种一致
            payPlan.setPayDic("CNY"); // 付款计划单的支付币种设置为与采购单一样的默认币种CNY
            payPlan.setPayCur(exRate);

            BigDecimal compensate = counteract.subtract(subtract); // 待补差额
            payPlan.setCostPlanNo("S" + costPlanService.createNo()); // 差额计划单生成付款费用单时，费用单号保留 S 打头
            payPlan.setCostPlanStatus(0);
            payPlan.setCostForm(1);
            payPlan.setIsLend(2);
            payPlan.setPlanAmount(compensate);
            payPlan.setPayPlanAmount(compensate.divide(exRate, 4));
            payPlan.setCounteract(new BigDecimal(0));
            payPlan.setPayCounteract(new BigDecimal(0));
            payPlan.setIsTax(0);
            payPlan.setTypeNo("CTN1809140510102");
            payPlan.setTypeName("办公费");
            payPlan.setKindNo("CTK1809141710103");
            payPlan.setKindName("办公用品");
            payPlan.setCostReason("借支差额：" + payPlan.getSupCostNo());
            int insertCostPlan = costPlanDao.insert(payPlan);

            if (insertCostPlan == 1) {
                logger.info("采购单关联借支单时，生成一张补差额的付款单计划：" + payPlan);
                // 生成空子类的子项目
                CostPlanTempDomain temp = new CostPlanTempDomain();
                temp.setTempNo(costPlanTempService.createNo());
                temp.setCostPlanNo(payPlan.getCostPlanNo());
                temp.setCostReason("借支差额：" + payPlan.getSupCostNo());
                temp.setAmount(payPlan.getPlanAmount());
                temp.setDic(payPlan.getDic());
                Integer insertCostPlanTemp = costPlanTempDao.insert(temp);
                if (null != insertCostPlanTemp && insertCostPlanTemp == 1) {
                    logger.info("采购单关联借支单时，生成差额单子项目 " + temp);
                }
                // 借支单的补差额添加金额
                if (null == linkLendCost.getCompensate()) {
                    linkLendCost.setCompensate(new BigDecimal(0));
                }
                linkLendCost.setCompensate(linkLendCost.getCompensate().add(compensate)); // 设置借支单待补差额
            } else {
                logger.info("采购单关联借支单时，生成一张补差额的付款计划单失败：" + payPlan);
                result = "采购单关联借支单时，生成一张补差额的付款计划单失败";
                return result;
            }
            result += "，借支单待补差额" + linkLendCost.getCompensate().setScale(2, BigDecimal.ROUND_HALF_UP).toString() + linkLendCost.getDic();
        }
        costDao.insert(backCost);
        costLogService.save(backCost.getCostNo(), updateUserId, "采购单【" + buyListDto.getBuyno() + "】冲销借支单【" + linkLendCost.getCostNo() + "】时由系统自动生成");

        costDao.update(linkLendCost);
        costLogService.save(linkLendCost.getCostNo(), updateUserId, "财务审核：" + result);
        return result;
    }

    /**
     * 采购单冲销借支单时生成已完成状态的借还单
     *
     * @param buyListDto   采购单
     * @param linkLendCost 关联的借支单
     * @param counteract   采购单冲销金额，即生成的借还单金额
     * @param exRate       汇率：采购单币种-->借支单币种
     * @return
     */
    public CostDomain generateBackCost(BuyListDto buyListDto, CostDomain linkLendCost, BigDecimal counteract, BigDecimal exRate) {
        CostDomain backCost = new CostDomain();
        BeanUtils.copyProperties(linkLendCost, backCost);
        backCost.setCostPlanNo(null);
        backCost.setCostNo(CostUtils.getIdNum(costService));
        backCost.setCostReason("采购单【" + buyListDto.getBuyno() + "】冲销借支单【" + linkLendCost.getCostNo() + "】");
        backCost.setCostRemark("采购单【" + buyListDto.getBuyno() + "】冲销借支单【" + linkLendCost.getCostNo() + "】，财务审核采购单通过后由系统自动生成本借还单");
        backCost.setCreateTime(new Date());
        backCost.setSupCostNo(linkLendCost.getCostNo());
        backCost.setIsLend(2);
        backCost.setHadPay(null);
        backCost.setLendStatus(null);
        backCost.setCompensate(null);
        backCost.setPayDic("CNY"); // 采购单默认CNY
        backCost.setPayCur(exRate);
        backCost.setDetailKey(null);

        backCost.setAmount(counteract);
        backCost.setCounteract(counteract);
        backCost.setPayCounteract(buyListDto.getAmount());
        backCost.setPayPlanAmount(buyListDto.getAmount());
        backCost.setToRmbRate(exRate);
        backCost.setAmountRmb(counteract);
        backCost.setPayUserId(linkLendCost.getCreateUserid());
        backCost.setPayTime(new Date());
        return backCost;
    }

    @Override
    public List<BuyListPrintDto> getBuyPrintList(Integer[] printBuyIds) {
        List<BuyDomain> buyList = buyDao.selectByIds(printBuyIds);
        List<BuyListPrintDto> buyListPrintDtos = new ArrayList<>();
        BuyListPrintDto printDto = null;
        for (BuyDomain buyDomain : buyList) {
            printDto = new BuyListPrintDto();
            BeanUtils.copyProperties(buyDomain, printDto);
            printDto.setTxtAmount(MoneyUtil.getCnNum(printDto.getAmount()));
            buyListPrintDtos.add(printDto);
        }
        return buyListPrintDtos;
    }

    @Override
    public String exportExcel(Integer[] buyIds) {
        HSSFWorkbook workbook = new HSSFWorkbook();
        HSSFSheet sheet = workbook.createSheet("采购单");
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        List<BuyExportDto> exportList = buyDao.selectExportByIds(buyIds);
        if (null == exportList || exportList.size() < 1) {
            return null;
        }
        // 设置表头
        String[] headers = {"采购单号", "SKU大类", "仓库", "采购单状态", "发货状态", "关联借支单号", "采购主体", "供应商", "SKU标题", "数量", "单价", "付款金额", "币种", "创建人名称", "部门", "下单时间", "付款日期"};
        HSSFRow row0 = sheet.createRow(0);
        for (int i = 0; i < headers.length; i++) {
            HSSFCell cell = row0.createCell(i);
            HSSFRichTextString text = new HSSFRichTextString(headers[i]);
            cell.setCellValue(text);
        }
        int rows = 1;
        for (BuyExportDto buyExportDto : exportList) {
            HSSFRow row = sheet.createRow(rows);
            row.createCell(0).setCellValue(buyExportDto.getBuyno());
            row.createCell(1).setCellValue(buyExportDto.getSkuTypeName());
            row.createCell(2).setCellValue(buyExportDto.getWarehousename());
            row.createCell(3).setCellValue(buyExportDto.getBuyStatus());
            row.createCell(4).setCellValue(buyExportDto.getSendStatus());
            row.createCell(5).setCellValue(buyExportDto.getLinkLendCostNo());
            row.createCell(6).setCellValue(buyExportDto.getCompany());
            row.createCell(7).setCellValue(buyExportDto.getSuppliername());
            row.createCell(8).setCellValue(buyExportDto.getSkuname());
            row.createCell(9).setCellValue(buyExportDto.getCount());
            row.createCell(10).setCellValue(buyExportDto.getPrice().toString());
            row.createCell(11).setCellValue(buyExportDto.getAmount().toString());
            row.createCell(12).setCellValue(buyExportDto.getCurrency());
            row.createCell(13).setCellValue(buyExportDto.getCreateusername());
            row.createCell(14).setCellValue(buyExportDto.getDepartmentname());
            row.createCell(15).setCellValue(sdf.format(buyExportDto.getCreatetime()));
            row.createCell(16).setCellValue(buyExportDto.getPayTime() != null ? sdf.format(buyExportDto.getPayTime()) : "");
            rows++;
        }
        // 保存到本地，并且返回路径
        String filePath = PathUtil.getBasePath() + PathUtil.getPath("export/temp/") + System.currentTimeMillis() + "&buy.xls";
        File tempFile = new File(filePath);
        // 创建路径
        if (!tempFile.getParentFile().exists()) {
            tempFile.getParentFile().mkdirs();
        }
        FileOutputStream fileOut = null;
        try {
            fileOut = new FileOutputStream(tempFile);
            workbook.write(fileOut);
            fileOut.flush();
            fileOut.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return filePath;
    }

    @Override
    public String judgeLinkLendCost(CostDomain linkLendCost, BuyListDto buyListDto) {
        if (null == linkLendCost.getHadPay()) linkLendCost.setHadPay(new BigDecimal(0));
        if (null == linkLendCost.getCounteract()) linkLendCost.setCounteract(new BigDecimal(0));
        if (null == linkLendCost.getLendBalance()) linkLendCost.setLendBalance(new BigDecimal(0));
        BigDecimal subtract = linkLendCost.getAmount().subtract(linkLendCost.getHadPay()); // 关联借支单还需还差额
        BigDecimal exRate = null; // 采购单币种（现默认CNY）转借支单币种汇率
        if ("CNY".equals(linkLendCost.getDic())) {
            exRate = new BigDecimal("1");
        } else {
            exRate = CurUtils.getCur("CNY", linkLendCost.getDic(), getExchangeRateApi);
        }
        BigDecimal counteract = buyListDto.getAmount().multiply(exRate).setScale(2, BigDecimal.ROUND_HALF_UP); // 冲销金额

        String result = null;
        if (counteract.compareTo(BigDecimal.ZERO) != 1) {
            result = "采购单金额必须大于0，请修改";
        } /*else if(counteract.compareTo(subtract) == 1){
            // 冲销金额>借支单待还金额，不允许关联借支单
            result = "采购单的金额大于借支单待还金额，不允许进行冲销";
        }*/
        return result;
    }

    @Override
    @Transactional
    public void setInstallment(String buyNo, BigDecimal payment) {
        BuyDomain buyDomain = buyDao.findByBuyNo(buyNo);
        String installmentInfo = buyDomain.getInstallmentInfo();
        List<InstallmentDto> installmentList = null;
        int periods = 1; // 期数
        if (installmentInfo != null) {
            installmentList = JSONObject.parseArray(installmentInfo, InstallmentDto.class);
            periods = installmentList.size() + 1;
        } else {
            installmentList = new ArrayList<>();
        }
        // 生成子采购单
        String subBuyNo = buyNo + "X" + periods;
        BuyDomain subBuy = new BuyDomain();
        BeanUtils.copyProperties(buyDomain, subBuy);
        subBuy.setBuyno(subBuyNo);
        subBuy.setBuyType(2); // 子采购单
        subBuy.setInstallmentInfo(null);
        subBuy.setId(null);
        subBuy.setNo(null);
        subBuy.setAmount(payment);
        subBuy.setPayamount(BigDecimal.ZERO);
        subBuy.setBuystatus(2); //待财务审核
        subBuy.setNote(subBuy.getNote() == null ? "【分期支付】" : "【分期支付】" + subBuy.getNote());
        buyDao.insert(subBuy);
        // 生成子采购单明细
        List<BuyDetailDomain> subBuyDetails = buyDetailDao.selectByBuyno(buyNo);
        int detailNum = subBuyDetails.size();
        if (detailNum > 1) {
            BigDecimal oneSubBuyDetailAmount = payment.divide(new BigDecimal(detailNum)).setScale(2); // 单个SKU总金额
            BigDecimal remainAmount = payment.subtract(oneSubBuyDetailAmount.multiply(new BigDecimal(detailNum - 1))); // 支付额分摊到最后一个SKU的金额
            for (BuyDetailDomain subBuyDetail : subBuyDetails) {
                subBuyDetail.setId(null);
                subBuyDetail.setBuyno(subBuyNo);
                subBuyDetail.setAmount(oneSubBuyDetailAmount);
            }
            subBuyDetails.get(detailNum - 1).setAmount(remainAmount);
        } else {
            subBuyDetails.get(0).setId(null);
            subBuyDetails.get(0).setBuyno(subBuyNo);
            subBuyDetails.get(0).setAmount(payment);
        }
        for (BuyDetailDomain subBuyDetail : subBuyDetails) {
            buyDetailDao.insert(subBuyDetail);
        }
        // 更新分期主采购单
        InstallmentDto oneInstallment = new InstallmentDto(payment, 0);
        installmentList.add(oneInstallment);
        buyDomain.setInstallmentInfo(JSON.toJSONString(installmentList));
        buyDomain.setBuyType(3); // 类型设置为分期主采购单
        buyDao.update(buyDomain);
    }

    @Override
    public void updateInstallment(String subBuyNo, BigDecimal payment) {
        String[] buyNoInfo = subBuyNo.split("X");
        int periods = Integer.parseInt(buyNoInfo[1]);
        BuyDomain buyDomain = buyDao.findByBuyNo(buyNoInfo[0]);
        List<InstallmentDto> installmentList = JSONObject.parseArray(buyDomain.getInstallmentInfo(), InstallmentDto.class);
        installmentList.get(periods - 1).setPayStatus(1); // 对应分期状态更改为已支付
        BigDecimal hadPayAmount = (buyDomain.getPayamount() == null ? BigDecimal.ZERO : buyDomain.getPayamount()).add(payment);
        if (hadPayAmount.compareTo(buyDomain.getAmount()) == 0) {
            buyDomain.setBuystatus(4);
        }
        buyDomain.setPayamount(hadPayAmount);
        buyDomain.setInstallmentInfo(JSON.toJSONString(installmentList));
        buyDao.update(buyDomain);
    }

    @Override
    public void deleteInstallment(String subBuyNo) {
        String[] buyNoInfo = subBuyNo.split("X");
        int periods = Integer.parseInt(buyNoInfo[1]);
        BuyDomain buyDomain = buyDao.findByBuyNo(buyNoInfo[0]);
        List<InstallmentDto> installmentList = JSONObject.parseArray(buyDomain.getInstallmentInfo(), InstallmentDto.class);
        if (installmentList.size() == 1) { // 废弃分期的首期子采购单，分期主采购单类型更改为普通采购单
            buyDomain.setInstallmentInfo(null);
            buyDomain.setBuyType(1); // 普通采购单
            buyDao.updateInstallment(buyDomain);
        } else {
            installmentList.remove(periods - 1);
            buyDomain.setInstallmentInfo(JSON.toJSONString(installmentList));
            buyDao.update(buyDomain);
        }
        buyDao.deleteBuy(subBuyNo);
        buyDetailDao.deleteBuyDetail(subBuyNo);
    }
}
