package com.blt.other.module.cost.service.impl.costcheck;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.blt.other.common.exception.BizRuntimeException;
import com.blt.other.database.model.CostCompanyDomain;
import com.blt.other.database.model.CostLogDomain;
import com.blt.other.module.auth.service.IOaUserService;
import com.blt.other.module.cost.dao.CostCompanyDao;
import com.blt.other.module.cost.model.CostDomain;
import com.blt.other.module.cost.model.CostTemplate;
import com.blt.other.module.sys.model.CostReviewer;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>
 *
 * </p>
 *
 * @author robbendev
 * @since 2020/10/30 2:57 下午
 */
@Component
public class FinancialCheckState extends CostState {


    @Resource
    FinalCheckState finalCheckState;
    @Resource
    CostCompanyDao costCompanyDao;

    @Override
    void nextState(CostState costState) {
        super.nextState(costState);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void handle() {
        CostDomain costDomain = costContext.costDomain;
        Integer currentUserId = costContext.currentUserId;
        CostTemplate costTemplate = costTemplateService.queryDetail(costDomain.getCostTemplateId());

        //check status
        if (!costDomain.getCostStatus().equals(CostDomain.STATUS_FINANCIAL_CHECK)) {
            throw new BizRuntimeException("invalid status");
        }

        //需要自动审核
        if (costTemplate.shouldFinancialAutoCheck(costDomain)) {
            //自动审核通过
            if (this.autoCheck(costDomain)) {
                costDomain.setCostStatus(CostDomain.STATUS_FINAL_CHECK);
                costDomain.setLastModifyDate(LocalDateTime.now());
                costDao.updateById(costDomain);
                costLogService.saveByManage(costDomain.getCostNo(), "财务自动审核通过", CostLogDomain.FINANCIAL_AUTO_PASS);

                //流转状态
                nextState(finalCheckState);
                //发布到总线尝试下个环节的自动审核
                costSubscribe.subscribe(costContext);
                return;
            }

        }

        if (costContext.currentUserId != null) {
            //人工审核通过
            if (costCurrentReviewerService.canAudit(currentUserId, costDomain.getCostNo())) {
                costDomain.setCostStatus(CostDomain.STATUS_FINAL_CHECK);
                costDomain.setLastModifyDate(LocalDateTime.now());
                costDao.updateById(costDomain);

                costLogService.save(costDomain.getCostNo(), currentUserId, "财务审核通过", CostLogDomain.FINANCIAL_MANUAL_PASS);

                //流转状态
                nextState(finalCheckState);
                //发布到总线尝试下个环节的自动审核
                costSubscribe.subscribe(costContext);
            }
            //人工审核没权限
            else {
                throw new BizRuntimeException("current user no authority");
            }
        }
    }

    @Override
    public void refuse(String rejectReason) {
        CostDomain costDomain = costContext.costDomain;
        Integer currentUserId = costContext.currentUserId;

        CostCompanyDomain costCompany = costCompanyDao.selectByNo(costDomain.getCompanyNo());

        //check status
        if (!costDomain.getCostStatus().equals(CostDomain.STATUS_FINANCIAL_CHECK)) {
            throw new BizRuntimeException("invalid status");
        }

        //人工审核通过
        if (!costCurrentReviewerService.canAudit(currentUserId, costDomain.getCostNo())) {
            throw new BizRuntimeException("current user no authority");
        }

        costContext.costService.reject(costDomain.getCostNo());
        costLogService.save(costDomain.getCostNo(), currentUserId, "财务审核拒绝,理由:" + rejectReason, CostLogDomain.TYPE_UPDATE);

    }

    @Override
    public void updateCurrentReviewer() {
        CostDomain costDomain = costContext.costDomain;
        CostCompanyDomain costCompanyDomain = costCompanyDao.selectByNo(costDomain.getCompanyNo());


        List<CostReviewer> costReviewerList = costReviewerMapper.selectList(new LambdaQueryWrapper<CostReviewer>()
                .eq(CostReviewer::getReferId, costCompanyDomain.getId())
                .eq(CostReviewer::getType, CostReviewer.financialReviewer));

        costCurrentReviewerService.updateByCostNoAndReviewer(costDomain.getCostNo(), costReviewerList);
        costLogService.save(costDomain.getCostNo(), costContext.currentUserId, "财务审核人:" + costReviewerList.stream().map(CostReviewer::getReviewerUserName).collect(Collectors.joining(",")));


    }
}
