package com.blt.other.module.cost.service.impl.costplan;

import com.bailuntec.cost.api.dto.CostPlanTempDto;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.blt.other.common.util.CurUtils;
import com.blt.other.common.util.PathUtil;
import com.blt.other.module.cost.dao.CostPlanDao;
import com.blt.other.module.cost.dao.CostPlanTempDao;
import com.blt.other.module.cost.dto.AddItemReq;
import com.blt.other.module.cost.dto.AddItemResp;
import com.blt.other.module.cost.dto.GetCostTempByPlanNoResp;
import com.blt.other.module.cost.dto.ResetItemReq;
import com.blt.other.module.cost.service.CostPlanTempService;
import com.blt.other.module.cost.service.CostTypeKindService;
import com.blt.other.module.cost.utils.CostFileUtil;
import com.blt.other.module.database.model.CostPlanDomain;
import com.blt.other.module.database.model.CostPlanTempDomain;
import com.blt.other.module.database.model.CostTypeKindDomain;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.stream.Collectors;

@Service
public class CostPlanTempServiceImpl implements CostPlanTempService {

    @Autowired
    private CostPlanTempDao costPlanTempDao;
    @Autowired
    CostTypeKindService costTypeKindService;
    @Autowired
    private CostPlanDao costPlanDao;

    @Override
    public String createNo() {

        Random random = new Random();
        SimpleDateFormat sdf = new SimpleDateFormat("yyMMddhhmmss");
        String no = "CPT" + sdf.format(new Date()) + random.nextInt(9);
        CostPlanTempDomain costPlanTempDomain = costPlanTempDao.selectByNo(no);
        while (null != costPlanTempDomain && null != costPlanTempDomain.getCostPlanNo()) {
            no = "CPT" + sdf.format(new Date()) + random.nextInt(9);
            costPlanTempDomain = costPlanTempDao.selectByNo(no);
        }
        return no;
    }

    /**
     * 检查同一个付款计划中是否存在大类相同、币种不同的情况
     *
     * @param costPlanTempDomain 费用计划
     */
    private void checkDic(CostPlanTempDomain costPlanTempDomain) {
        List<CostPlanTempDomain> checkDict = costPlanTempDao.selectByPlanNoAndTypeNo(costPlanTempDomain.getCostPlanNo(),
                costPlanTempDomain.getTypeNo());
        if (checkDict.stream().collect(Collectors.groupingBy(CostPlanTempDomain::getTypeNo)).keySet().size() > 1) {
            throw new RuntimeException("已存在相同大类，请保持币种一致！");
        }
    }


    @Transactional(rollbackFor = Exception.class)
    @Override
    public AddItemResp doSave(AddItemReq req) {

        CostTypeKindDomain kindByKindNo = costTypeKindService.getKindByKindNo(req.getKindNo());
        req.setTypeNo(kindByKindNo.getTypeNo());


        CostPlanTempDomain costPlanTempDomain = new CostPlanTempDomain();
        BeanUtils.copyProperties(req, costPlanTempDomain);
        costPlanTempDomain.setTempNo(this.createNo());
        this.checkDic(costPlanTempDomain);


        //新增付款子项目
        costPlanTempDao.insert(costPlanTempDomain);

        // 添加付款子项目成功，计划单改变总额
        CostPlanDomain costPlanDomain = costPlanDao.selectByNo(costPlanTempDomain.getCostPlanNo());
        if (costPlanDomain != null) {
            costPlanDomain.setPlanAmount(costPlanDomain.getPlanAmount().add(costPlanTempDomain.getAmount()));
            costPlanDao.update(costPlanDomain, new LambdaQueryWrapper<CostPlanDomain>()
                    .eq(CostPlanDomain::getCostPlanNo, costPlanDomain.getCostPlanNo()));
        }

        AddItemResp resp = new AddItemResp();
        resp.setMsg("success");
        resp.setPlaAmount(CurUtils.getCur(costPlanTempDomain.getDic(), "CNY"));
        resp.setTmp(costPlanTempDomain);
        resp.setSuccess(true);
        return resp;
    }


    @Override
    public GetCostTempByPlanNoResp getTempListByPlanNo(String costPlanNo) {
        List<CostPlanTempDomain> list = costPlanTempDao.selectListByPlanNo(costPlanNo);

        List<CostPlanTempDto> temps = list.stream().map(CostPlanTempDomain::castToDto).collect(Collectors.toList());

        BigDecimal planAmount = temps.stream()
                .map(tempDto -> {
                    BigDecimal cur = CurUtils.getCur(tempDto.getDic(), "CNY");
                    return tempDto.getAmount().multiply(cur);
                })
                .reduce(BigDecimal.ZERO, BigDecimal::add);

        GetCostTempByPlanNoResp resp = new GetCostTempByPlanNoResp();
        resp.setPlanAmount(planAmount);
        resp.setSuccess(true);
        resp.setTemps(temps);
        return resp;
    }

    @Transactional
    @Override
    public BigDecimal deleteItemByPlanNoAndTempNo(String costPlanNo, String tempNo) {

        CostPlanTempDomain costPlanTempDomain = costPlanTempDao.selectByPlanNoAndTempNo(costPlanNo, tempNo);
        costPlanTempDao.deletedByPlanNoAndTempNo(costPlanNo, tempNo);

        // 删除付款子项目成功，计划单改变总额
        CostPlanDomain costPlanDomain = costPlanDao.selectByNo(costPlanNo);
        if (costPlanDomain != null) {

            costPlanDomain.setPlanAmount(costPlanDomain.getPlanAmount().subtract(costPlanTempDomain.getAmount()));
            costPlanDao.update(costPlanDomain, new LambdaQueryWrapper<CostPlanDomain>()
                    .eq(CostPlanDomain::getCostPlanNo, costPlanDomain.getCostPlanNo()));
            return costPlanDomain.getPlanAmount();
        }
        return BigDecimal.ZERO;
    }


    @Override
    public CostPlanTempDomain getByTempNo(@NonNull String tempNo) {
        CostPlanTempDomain costPlanTempDomain = costPlanTempDao.selectByNo(tempNo);
        if (costPlanTempDomain == null) {
            throw new RuntimeException("not found");
        }
        return costPlanTempDomain;
    }


    @Transactional(rollbackFor = Exception.class)
    @Override
    public CostPlanTempDomain resetItem(ResetItemReq req) {

        CostTypeKindDomain kindByKindNo = costTypeKindService.getKindByKindNo(req.getKindNo());
        req.setTypeNo(kindByKindNo.getTypeNo());

        CostPlanTempDomain costPlanTempDomain = new CostPlanTempDomain();
        com.bailuntec.common.BeanUtils.copyProperties(req, costPlanTempDomain);

        if (req.getDelecteFile()) {
            costPlanTempDao.deletedFilePath(req.getTempNo());
        }
        costPlanTempDao.update(costPlanTempDomain);

        this.chalkResetDec(costPlanTempDomain.getCostPlanNo());

        if (req.getFileSelect2() != null) {
            String filePath = PathUtil.getBasePath() + PathUtil.getPath("cost/" + costPlanTempDomain.getTempNo() + "/");
            String path = CostFileUtil.upload(req.getFileSelect2(), filePath);
            costPlanTempDomain.setFilePath(path);
        }


        // 同步更新费用计划金额
        CostPlanDomain plan = costPlanDao.selectByNo(req.getCostPlanNo());
        if (plan != null) {
            List<CostPlanTempDomain> tempListByPlanNo = costPlanTempDao.selectListByPlanNo(req.getCostPlanNo());
            BigDecimal newAmount = tempListByPlanNo.stream().map(CostPlanTempDomain::getAmount).reduce(BigDecimal.ZERO, BigDecimal::add);

            plan.setPlanAmount(newAmount);
            if (plan.getCostForm() == 1 && (null != plan.getIsLend() && 2 == plan.getIsLend())) {
                plan.setPayPlanAmount(newAmount.divide(plan.getPayCur(), 4));
            }
            costPlanDao.update(plan, new LambdaQueryWrapper<CostPlanDomain>()
                    .eq(CostPlanDomain::getCostPlanNo, plan.getCostPlanNo()));
        }
        return costPlanTempDomain;
    }


    /**
     * 修改时判断费用单计划是否存在不相同货币的付款小类
     *
     * @param costPlanNo 费用单编号
     */
    private void chalkResetDec(String costPlanNo) {

        List<CostPlanTempDomain> costPlanTempDomainList = costPlanTempDao.selectListByPlanNo(costPlanNo);
        Map<String, List<CostPlanTempDomain>> map = costPlanTempDomainList.stream().collect(Collectors.groupingBy(CostPlanTempDomain::getDic));
        if (map.keySet().size() > 1) {
            throw new RuntimeException("请统一货币单位");
        }
    }
}
