package com.blt.other.module.cost.service.impl.costtemplate;

import com.bailuntec.common.ListUtil;
import com.bailuntec.common.StringUtils;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.blt.other.common.exception.BizRuntimeException;
import com.blt.other.module.auth.dao.UserDao;
import com.blt.other.module.cost.dao.CostCompanyDao;
import com.blt.other.module.cost.dao.CostTemplateColMapper;
import com.blt.other.module.cost.dto.request.QueryListReq;
import com.blt.other.module.cost.model.CostTemplate;
import com.blt.other.module.cost.model.CostTemplateCol;
import com.blt.other.module.cost.service.ICostTemplateBaseColService;
import com.blt.other.module.cost.service.ICostTemplateColService;
import com.blt.other.module.cost.service.ICostTemplateService;
import com.google.common.collect.Lists;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * <p>
 * 费用单模版 服务实现类
 * </p>
 *
 * @author robbendev
 * @since 2020-10-14
 */
@Service
public class CostTemplateColServiceImpl extends ServiceImpl<CostTemplateColMapper, CostTemplateCol> implements ICostTemplateColService {
    @Resource
    ICostTemplateService costTemplateService;
    @Resource
    ICostTemplateBaseColService costTemplateBaseColService;
    @Resource
    CostCompanyDao costCompanyDao;
    @Resource
    UserDao userDao;

    @Override
    public void delete(Integer id) {
        CostTemplateCol costTemplateCol = this.getById(id);
        if (costTemplateCol.getIsDefault()) {
            throw new BizRuntimeException("unable to remove default costTemplateCol");
        }
        this.removeById(id);

        List<CostTemplateCol> costTemplateColList = baseMapper.selectByTemplateId(costTemplateCol.getCostTemplateId());
        CostTemplate costTemplate = costTemplateService.getById(costTemplateCol.getCostTemplateId());
        costTemplate.setSetAutoCheck(ListUtil.isNotEmpty(costTemplateColList.stream()
                .filter(item -> item.getAutoRuleType() > 0)
                .collect(Collectors.toList())));
        costTemplateService.updateById(costTemplate);
    }


    @Override
    public void modify(CostTemplateCol costTemplateCol) {
        if (costTemplateCol.getIsDefault()) {
            throw new BizRuntimeException("unable to modify default costTemplateCol");
        }
        costTemplateCol.setCompanyScopeStr(this.getCompanyScopeStr(costTemplateCol.getCompanyScope()));
        costTemplateCol.setUpdateUserName(userDao.selectByuserid(costTemplateCol.getUpdateUserId()).getUsername());

        this.updateById(costTemplateCol);
        if (costTemplateCol.getCostTemplateBaseCol() != null) {
            costTemplateBaseColService.updateById(costTemplateCol.getCostTemplateBaseCol());
        }

    }

    @Override
    public void add(CostTemplateCol costTemplateCol) {
        costTemplateCol.setCompanyScopeStr(this.getCompanyScopeStr(costTemplateCol.getCompanyScope()));
        costTemplateCol.setCreateUserName(userDao.selectByuserid(costTemplateCol.getCreateUserId()).getUsername());
        costTemplateCol.setUpdateUserName(userDao.selectByuserid(costTemplateCol.getUpdateUserId()).getUsername());

        this.save(costTemplateCol);

        CostTemplate costTemplate = costTemplateService.getById(costTemplateCol.getCostTemplateId());
        costTemplate.setSetAutoCheck(true);
        costTemplateService.updateById(costTemplate);


    }

    /**
     * 获取公司主体范围中文
     *
     * @param companyScope 公司主体范围no 逗号分割
     * @return 取公司主体范围中文 逗号分割
     */
    private String getCompanyScopeStr(String companyScope) {
        return Lists.newArrayList(companyScope.split(","))
                .stream()
                .map(companyNo -> costCompanyDao.selectByNo(companyNo).getCompanyName())
                .collect(Collectors.joining(","));

    }

    @Override
    public void setCostTemplateCol(List<CostTemplate> costTemplateList) {
        List<Integer> costTemplateIds = costTemplateList.stream().map(CostTemplate::getId).collect(Collectors.toList());
        if (ListUtil.isNotEmpty(costTemplateIds)) {
            List<CostTemplateCol> costTemplateColList = this.list(new LambdaQueryWrapper<CostTemplateCol>()
                    .in(CostTemplateCol::getCostTemplateId, costTemplateIds));
            costTemplateBaseColService.setCostTemplateBaseCol(costTemplateColList);

            Map<Integer, List<CostTemplateCol>> map = costTemplateColList.stream().collect(Collectors.groupingBy(CostTemplateCol::getCostTemplateId));
            costTemplateList.forEach(costTemplate -> {
                costTemplate.setCostTemplateColList(map.get(costTemplate.getId()));
            });
        }
    }

    @Override
    public void setCostTemplateCol(CostTemplate costTemplate) {
        List<CostTemplateCol> costTemplateColList = this.list(new LambdaQueryWrapper<CostTemplateCol>()
                .eq(CostTemplateCol::getCostTemplateId, costTemplate.getId()));
        costTemplateBaseColService.setCostTemplateBaseCol(costTemplateColList);

        costTemplate.setCostTemplateColList(costTemplateColList);
    }

    @Override
    public List<CostTemplateCol> queryList(QueryListReq req) {

        return costTemplateService.queryDetail(req.getCostTemplateId()).getCostTemplateColList()
                .stream()
                //非默认
                .filter(costTemplateCol -> !costTemplateCol.getIsDefault())
                //字段类目
                .filter(costTemplateCol -> StringUtils.isEmpty(req.getColCategory()) ||
                        Optional.ofNullable(costTemplateCol.getCostTemplateBaseCol().getColCategory()).map(str -> str.contains(req.getColCategory())).orElse(false))
                //字段提示
                .filter(costTemplateCol -> StringUtils.isEmpty(req.getPrompt()) ||
                        Optional.ofNullable(costTemplateCol.getCostTemplateBaseCol().getPrompt()).map(str -> str.contains(req.getPrompt())).orElse(false))
                //字段名称
                .filter(costTemplateCol -> StringUtils.isEmpty(req.getColName()) ||
                        Optional.ofNullable(costTemplateCol.getCostTemplateBaseCol().getColName()).map(str -> str.contains(req.getColName())).orElse(false))
                //字段类型
                .filter(costTemplateCol -> req.getColType() == null ||
                        Optional.ofNullable(costTemplateCol.getCostTemplateBaseCol().getColType()).map(str -> str.equals(req.getColType())).orElse(false))
                //是否必填
                .filter(costTemplateCol -> req.getIsRequired() == null ||
                        req.getIsRequired().equals(costTemplateCol.getIsRequired()))
                //创建人
                .filter(costTemplateCol -> req.getCreateUserId() == null ||
                        req.getCreateUserId().equals(costTemplateCol.getCreateUserId()))
                //更新人
                .filter(costTemplateCol -> req.getUpdateUserId() == null ||
                        req.getUpdateUserId().equals(costTemplateCol.getUpdateUserId()))
                //审批规则是否配置
                .filter(costTemplateCol -> req.getIsAutoRule() == null ||
                        req.getIsAutoRule().equals(costTemplateCol.getAutoRuleType() > 0))
                //审核匹配符号类型
                .filter(costTemplateCol -> req.getAutoRuleType() == null ||
                        req.getAutoRuleType().equals(costTemplateCol.getAutoRuleType()))
                //创建时间
                .filter(costTemplateCol -> req.getCreateTimeStart() == null ||
                        costTemplateCol.getCreateTime().compareTo(req.getCreateTimeStart()) >= 0)
                .filter(costTemplateCol -> req.getCreateTimeEnd() == null ||
                        costTemplateCol.getCreateTime().compareTo(req.getCreateTimeEnd()) <= 0)
                //更新时间
                .filter(costTemplateCol -> req.getUpdateTimeStart() == null ||
                        costTemplateCol.getLastUpdateTime().compareTo(req.getUpdateTimeStart()) >= 0)
                .filter(costTemplateCol -> req.getCreateTimeEnd() == null ||
                        costTemplateCol.getLastUpdateTime().compareTo(req.getCreateTimeEnd()) <= 0)
                .collect(Collectors.toList());

    }

    @Override
    public CostTemplateCol queryById(Integer id) {
        CostTemplateCol costTemplateCol = this.getById(id);
        costTemplateBaseColService.setCostTemplateBaseCol(costTemplateCol);
        return costTemplateCol;
    }
}
