package com.blt.other.module.cost.service.impl.costcheck;

import com.bailuntec.common.StringUtils;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.blt.other.common.exception.BizRuntimeException;
import com.blt.other.database.model.CostLogDomain;
import com.blt.other.module.auth.dao.OaDepartmentMapper;
import com.blt.other.module.auth.dao.OaUserMapper;
import com.blt.other.module.auth.model.OaDepartment;
import com.blt.other.module.auth.model.OaUser;
import com.blt.other.module.cost.dao.CostCurrentReviewerMapper;
import com.blt.other.module.cost.dao.SpecDepartmentCheckConfigMapper;
import com.blt.other.module.cost.model.ApprovalHistoryDomain;
import com.blt.other.module.cost.model.CostCurrentReviewer;
import com.blt.other.module.cost.model.CostDomain;
import com.blt.other.module.cost.model.CostTemplate;
import com.blt.other.module.cost.service.ApprovalHistoryService;
import com.blt.other.module.sys.dao.DepartmentReviewerMapper;
import com.blt.other.module.sys.model.CostReviewer;
import com.blt.other.module.sys.model.DepartmentReviewer;
import com.blt.other.module.sys.model.SpecDepartmentCheckConfig;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>
 *
 * </p>
 *
 * @author robbendev
 * @since 2020/10/30 2:57 下午
 */
@Slf4j
@Component
@Scope("prototype")
public class DepartmentCheckState extends CostState {

    @Resource
    HrCheckState hrCheckState;
    @Resource
    OaDepartmentMapper oaDepartmentMapper;
    @Resource
    OaUserMapper oaUserMapper;
    @Resource
    SpecDepartmentCheckConfigMapper specDepartmentCheckConfigMapper;
    @Resource
    CostCurrentReviewerMapper costCurrentReviewerMapper;

    @Resource
    ApprovalHistoryService approvalHistoryService;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void handle(String reason) {
        CostDomain costDomain = costContext.costDomain;
        Integer currentUserId = costContext.currentUserId;
        CostTemplate costTemplate = costTemplateService.queryDetail(costDomain.getCostTemplateId());

        //check status
        if (!costDomain.getCostStatus().equals(CostDomain.STATUS_DEPARTMENT_CHECK)) {
            throw new BizRuntimeException("invalid status");
        }

        OaUser oaUser = oaUserMapper.selectByOaUserId(costDomain.getCreateUserid());
        OaDepartment oaDepartment = oaDepartmentMapper.selectByDepartmentId(oaUser.getDepartmentId());

        //1、匹配特殊部门
        SpecDepartmentCheckConfig specDepartmentCheckConfig = specDepartmentCheckConfigMapper.selectOne(new LambdaQueryWrapper<SpecDepartmentCheckConfig>()
                .eq(SpecDepartmentCheckConfig::getCostCompanyName, costDomain.getCompanyName())
                .eq(SpecDepartmentCheckConfig::getOaCompanyId, oaUser.getCompanyId()));

        if (specDepartmentCheckConfig != null) {
            Integer reviewerDepartmentId = specDepartmentCheckConfig.getReviewerDepartmentId();
            oaDepartment = oaDepartmentMapper.selectByDepartmentId(reviewerDepartmentId);
            costLogService.save(costDomain.getCostNo(), 0, "匹配特殊部门:" + oaDepartment.getFullName());
            this.updateCurrentReviewer(reviewerDepartmentId);
        }

        //如果不需要审核 直接通过
        DepartmentReviewer departmentReviewer = this.getCurrentDepartmentReviewer(oaDepartment.getDepartmentId());
        //收款单也不需要审核
        if (costDomain.getCostForm() == 2 || (!costTemplate.shouldDepartmentCheck(costDomain, departmentReviewer.getAutoReviewAmount()))) {

            costDomain.setCostStatus(CostDomain.STATUS_HR_CHECK);
            costDomain.setLastModifyDate(LocalDateTime.now());
            costDao.updateById(costDomain);


            List<CostCurrentReviewer> costCurrentReviewerList = costCurrentReviewerMapper.selectByCostNo(costDomain.getCostNo());
            costLogService.saveByManage(costDomain.getCostNo(), "部门审核自动通过,当前审核人:" + costCurrentReviewerList.stream().map(CostCurrentReviewer::getUsername)
                    .collect(Collectors.joining(",")), CostLogDomain.DEPARTMENT_AUTO_PASS);

            //流转状态
            nextState(hrCheckState);
            costSubscribe.subscribe(costContext);
            return;
        }


        //需要自动审核
        if (costTemplate.shouldDepartmentAutoCheck(costDomain)) {
            //自动审核通过
            if (this.autoCheck(costDomain)) {
                costDomain.setCostStatus(CostDomain.STATUS_HR_CHECK);
                costDomain.setLastModifyDate(LocalDateTime.now());
                costDao.updateById(costDomain);
                List<CostCurrentReviewer> costCurrentReviewerList = costCurrentReviewerMapper.selectByCostNo(costDomain.getCostNo());
                costLogService.saveByManage(costDomain.getCostNo(), "部门审核自动通过,当前审核人:" + costCurrentReviewerList.stream().map(CostCurrentReviewer::getUsername)
                        .collect(Collectors.joining(",")), CostLogDomain.DEPARTMENT_AUTO_PASS);

                //流转状态
                nextState(hrCheckState);
                costSubscribe.subscribe(costContext);
                return;
            }
        }

        //人工审核
        if (costContext.currentUserId != null) {
            //人工审核通过
            if (costCurrentReviewerService.canAudit(currentUserId, costDomain.getCostNo())) {
                ApprovalHistoryDomain approvalHistoryDomain = ApprovalHistoryDomain.builder()
                        .approvalUserId(currentUserId)
                        .approvalTime(LocalDateTime.now())
                        .costStatus(costDomain.getCostStatus())
                        .costNo(costDomain.getCostNo())
                        .isPassed(true)
                        .build();
                costDomain.setCostStatus(CostDomain.STATUS_HR_CHECK);
                costDomain.setLastModifyDate(LocalDateTime.now());
                costDao.updateById(costDomain);
                costLogService.save(costDomain.getCostNo(), currentUserId, "部门审核通过" + (StringUtils.isNotBlank(reason) ? (":" + reason) : ""), CostLogDomain.DEPARTMENT_MANUAL_PASS);
                //审批历史
                approvalHistoryService.save(approvalHistoryDomain);
                nextState(hrCheckState);
                //发布到总线尝试下个环节的自动审核
                costSubscribe.subscribe(costContext);
            }
            //人工审核没权限
            else {
                throw new BizRuntimeException("current user no authority");
            }
        }
    }


    @Override
    public void refuse(String rejectReason) {
        log.info("部门审核拒绝 currentUserId:{},costNo:{}", costContext.currentUserId, costContext.costDomain.getCostNo());
        CostDomain costDomain = costContext.costDomain;
        Integer currentUserId = costContext.currentUserId;

        //check status
        if (!costDomain.getCostStatus().equals(CostDomain.STATUS_DEPARTMENT_CHECK)) {
            throw new BizRuntimeException("invalid status");
        }

        //人工审核没权限
        if (!costCurrentReviewerService.canAudit(currentUserId, costDomain.getCostNo())) {
            throw new BizRuntimeException("current user no authority");
        }

        costLogService.save(costDomain.getCostNo(), currentUserId, "部门审核拒绝,理由:" + rejectReason, CostLogDomain.TYPE_UPDATE);
        ApprovalHistoryDomain approvalHistoryDomain = ApprovalHistoryDomain.builder()
                .approvalUserId(currentUserId)
                .approvalTime(LocalDateTime.now())
                .costStatus(costDomain.getCostStatus())
                .costNo(costDomain.getCostNo())
                .isPassed(false)
                .build();
        approvalHistoryService.save(approvalHistoryDomain);
        costContext.costService.reject(costDomain.getCostNo());
    }

    @Resource
    DepartmentReviewerMapper departmentReviewerMapper;

    @Override
    public void updateCurrentReviewer() {
        CostDomain costDomain = costContext.costDomain;

        OaUser costUser = oaUserMapper.selectByOaUserId(costDomain.getCreateUserid());
        this.updateCurrentReviewer(costUser.getDepartmentId());

    }

    private DepartmentReviewer getCurrentDepartmentReviewer(Integer departmentId) {

        DepartmentChain departmentChain = this.getDepartmentChain(new DepartmentChain(departmentId));

        DepartmentReviewer departmentReviewer;
        DepartmentReviewer primaryDepartmentReviewe = null;
        DepartmentReviewer secondDepartmentReviewer = null;
        primaryDepartmentReviewe = departmentReviewerMapper.selectByPrimaryDepartmentId(departmentChain.getDepartmentId());
        if (departmentChain.getNext() != null) {
            secondDepartmentReviewer = departmentReviewerMapper.selectBySecondDepartmentId(departmentChain.next.getDepartmentId());
        }
        if (secondDepartmentReviewer != null) {
            departmentReviewer = secondDepartmentReviewer;
        } else {
            departmentReviewer = primaryDepartmentReviewe;
        }
        if (departmentReviewer == null) {
            throw new BizRuntimeException("请联系管理员添加部门审核人配置");
        }
        return departmentReviewer;
    }

    /**
     * @param last 最后一个chain
     * @return chain
     */
    private DepartmentChain getDepartmentChain(DepartmentChain last) {
        OaDepartment oaDepartment = oaDepartmentMapper.selectByDepartmentId(last.getDepartmentId());
        if (oaDepartment.getParentId() > 0) {
            DepartmentChain pre = new DepartmentChain(oaDepartment.getParentId());
            last.setPre(pre);
            pre.setNext(last);
            return this.getDepartmentChain(pre);
        } else {
            return last;
        }
    }

    @Data
    static class DepartmentChain {
        private DepartmentChain pre;
        private DepartmentChain next;

        private Integer departmentId;

        public DepartmentChain(Integer departmentId) {
            this.departmentId = departmentId;
        }
    }

    private void updateCurrentReviewer(Integer departmentId) {
        CostDomain costDomain = costContext.costDomain;
        DepartmentReviewer departmentReviewer = this.getCurrentDepartmentReviewer(departmentId);

        if (departmentReviewer.getSecondDepartmentId() > 0) {
            costLogService.save(costDomain.getCostNo(), 0, "匹配二级部门配置:" + departmentReviewer.getSecondDepartmentName());
        }

        List<CostReviewer> costReviewerList = costReviewerMapper.selectList(new LambdaQueryWrapper<CostReviewer>()
                .eq(CostReviewer::getReferId, departmentReviewer.getId())
                .eq(CostReviewer::getType, CostReviewer.departmentReviewer));

        String departmentReviewerNames = costReviewerList.stream().map(CostReviewer::getReviewerUserName).collect(Collectors.joining(","));
        log.info("更新当前审核人为部门审核人:{}", departmentReviewerNames);

        costCurrentReviewerService.updateByCostNoAndReviewer(costDomain.getCostNo(), costReviewerList);
//        costLogService.save(costDomain.getCostNo(), costContext.currentUserId, "部门审核人:" + departmentReviewerNames);
    }
}
