package com.bailuntec.ana.application.order.impl;

import com.bailuntec.ana.application.address.IOrderAddressService;
import com.bailuntec.ana.application.order.IOrderDetailService;
import com.bailuntec.ana.application.order.IOrderService;
import com.bailuntec.ana.application.order.IOrderSpecificationService;
import com.bailuntec.ana.application.purchase.IPurchaseTaskService;
import com.bailuntec.ana.domain.address.OrderAddress;
import com.bailuntec.ana.domain.order.Order;
import com.bailuntec.ana.domain.order.OrderDetail;
import com.bailuntec.ana.infrastructure.common.constant.PurchaseTaskConstant;
import com.bailuntec.ana.infrastructure.common.exception.EntityNotFoundException;
import com.bailuntec.ana.infrastructure.common.util.ListUtil;
import com.bailuntec.ana.infrastructure.common.util.SpringContextUtil;
import com.bailuntec.ana.infrastructure.feign.PamsApi;
import com.bailuntec.ana.infrastructure.mapper.order.OrderMapper;
import com.bailuntec.ana.vo.order.req.OrderQueryPageRequest;
import com.bailuntec.ana.vo.order.req.RefundOrderRequest;
import com.bailuntec.ana.vo.order.resp.Accounts;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;

/**
 * <p>
 * 服务实现类
 * </p>
 *
 * @author robbendev
 * @since 2020-07-28
 */
@Service
public class OrderServiceImpl extends ServiceImpl<OrderMapper, Order> implements IOrderService {

    @Resource
    private IOrderDetailService orderDetailService;
    @Resource
    private IOrderSpecificationService orderSpecificationService;
    @Resource
    private IOrderAddressService orderAddressService;
    @Resource
    private IPurchaseTaskService purchaseTaskService;
    @Resource
    private PamsApi pamsApi;

    @Override
    public void publishOrder(Order order) {
        this.save(order);
        order.getOrderDetailList().forEach(orderDetail -> {
            orderDetail.setOrderId(orderDetail.getId());
            orderDetailService.save(orderDetail);

            orderDetail.getOrderSpecificationList().forEach(orderSpecification -> {
                orderSpecification.setOrderDetailId(orderDetail.getId());
            });
            orderSpecificationService.saveBatch(orderDetail.getOrderSpecificationList());

        });

        OrderAddress orderAddress = order.getOrderAddress();
        orderAddress.setOrderId(order.getId());
        orderAddressService.save(orderAddress);

        IOrderService orderService = SpringContextUtil.getBean(OrderServiceImpl.class);

        //异步从pams获取刊登帐号信息
        orderService.syncOrderPdsAccount(order.getId());
        //异步从刊登系统获取对应的采购sku和采购成本
        orderService.syncOrderDetails(order.getId());
    }


    @Override
    public void orderSalesRefund(RefundOrderRequest request) {
        Order order = baseMapper.selectByOrderNoAndPdsAccountId(request.getOriginalOrderNo(), request.getPdsAccountId());
        order.setSalesRefundAmount(request.getSalesRefundAmount());
        //todo 调用自动采购退款
        baseMapper.updateById(order);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void purchase(String id) {
        Order order = this.queryDetail(id);

        //只采购未采购的sku
        List<OrderDetail> orderDetailList = orderDetailService.selectByOrderIdAndPurchaseStatus(id, OrderDetail.DELIVERY_STATUS_UN_DELIVERY);
        order.setOrderDetailList(orderDetailList);

        purchaseTaskService.addTask(order, PurchaseTaskConstant.ebayPurchaseT);

        this.updateById(order);
    }


    @Override
    public Order queryDetail(String id) {
        Order order = this.getById(id);
        if (order == null) {
            throw new EntityNotFoundException();
        }

        orderDetailService.setOrderDetail(order);
        orderAddressService.setOrderAddress(order);
        return order;
    }

    @Override
    public Page<Order> queryPage(OrderQueryPageRequest request) {

        Page<Order> page = new Page<>(request.getPageNum(), request.getPageSize());
        orderDetailService.setOrderDetail(page.getRecords());
        orderAddressService.setOrderAddress(page.getRecords());
        page = baseMapper.queryPage(page, request);
        return page;
    }

    @Async
    @Override
    public void syncOrderPdsAccount(String orderId) {
        Order order = this.getById(orderId);
        List<Accounts> accountsList = pamsApi.getAccount(order.getPdsAccountId()).getData();
        if (ListUtil.isNotEmpty(accountsList)) {
            Accounts accounts = accountsList.get(0);
            //刊登帐号
            order.setPdsAccount(accounts.getAccount());
            //站点
            order.setSite(accounts.getSite());
        }

    }

    @Async
    @Override
    public void syncOrderDetails(String orderId) {
        //todo 从刊登系统获取对应的采购sku和采购成本
    }


}
