package com.bailuntec.ana.infrastructure.common.aspect;

import com.bailuntec.ana.infrastructure.common.annotation.DictParam;
import com.baomidou.mybatisplus.annotation.TableName;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.bailuntec.ana.infrastructure.common.base.BaseResult;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.lang.reflect.Field;
import java.util.List;

/**
 * 字典翻译注解切面
 */
@Slf4j
@Aspect
@Component
@SuppressWarnings("all")
public class DictTransferAspect {

    @Resource
    StringRedisTemplate stringRedisTemplate;

    @Pointcut("@annotation(apiOperation)")
    public void doDataDictClass(ApiOperation apiOperation) {
    }


    @Around("@annotation(apiOperation)")
    public Object transfer(final ProceedingJoinPoint point, ApiOperation apiOperation) throws Throwable {
        Object result = point.proceed();
        try {
            if (result instanceof BaseResult) {
                this.dictTransfer(((BaseResult) result).getData());
            } else {
                this.dictTransfer(result);
            }
        } catch (Exception e) {
            log.warn("字典翻译失败");
        }
        return result;
    }


    private void dictTransfer(Object result) throws NoSuchFieldException, IllegalAccessException {
        if (result instanceof List) {
            for (Object object : (List) result) {
                this.dictTransferObject(object);
            }
        } else if (result instanceof IPage) {
            for (Object object : ((IPage) result).getRecords()) {
                this.dictTransferObject(object);
            }
        } else {
            this.dictTransferObject(result);
        }
    }

    private void dictTransferObject(Object object) throws IllegalAccessException, NoSuchFieldException {
        Field[] fields = object.getClass().getDeclaredFields();

        for (Field field : fields) {
            //如果是注解需要翻译的字段
            if (field.isAnnotationPresent(DictParam.class)) {
                DictParam dictParam = field.getAnnotation(DictParam.class);

                //设置字段可以访问
                field.setAccessible(true);
                //获取字段值
                String key = String.valueOf(field.get(object));
                //从缓存中翻译字典值
                String value = (String) stringRedisTemplate.opsForHash().get(dictParam.dictCode(), key);
                //获取字典值翻译后存放的字段
                Field valueField = object.getClass().getDeclaredField(dictParam.dictValueParam());
                //设置字段可以访问
                valueField.setAccessible(true);
                //设置翻译后的字典值
                valueField.set(object, value);
            }

            if (field.getType().isAnnotationPresent(ApiModel.class) || field.getType().isAnnotationPresent(TableName.class)) {
                field.setAccessible(true);
                Object subObject = field.get(object);
                if (subObject != null) {
                    this.dictTransfer(subObject);
                }
            }
        }
    }

}
