package com.bailuntec.ana.application.order.impl;

import com.bailuntec.ana.application.order.IOrderService;
import com.bailuntec.ana.application.purchase.IPurchaseTaskService;
import com.bailuntec.ana.domain.address.OrderAddress;
import com.bailuntec.ana.domain.order.Order;
import com.bailuntec.ana.domain.order.OrderDetail;
import com.bailuntec.ana.domain.order.OrderSpecification;
import com.bailuntec.ana.infrastructure.common.constant.PurchaseTaskConstant;
import com.bailuntec.ana.infrastructure.common.exception.BizRuntimeException;
import com.bailuntec.ana.infrastructure.common.exception.EntityNotFoundException;
import com.bailuntec.ana.infrastructure.common.exception.ErrorCode;
import com.bailuntec.ana.infrastructure.common.util.ListUtil;
import com.bailuntec.ana.infrastructure.mapper.address.OrderAddressMapper;
import com.bailuntec.ana.infrastructure.mapper.order.OrderDetailMapper;
import com.bailuntec.ana.infrastructure.mapper.order.OrderMapper;
import com.bailuntec.ana.infrastructure.mapper.order.OrderSpecificationMapper;
import com.bailuntec.ana.vo.order.req.RefundOrderRequest;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * <p>
 * 服务实现类
 * </p>
 *
 * @author robbendev
 * @since 2020-07-28
 */
@Service
public class OrderServiceImpl extends ServiceImpl<OrderMapper, Order> implements IOrderService {

    @Resource
    private OrderDetailMapper orderDetailMapper;
    @Resource
    private OrderSpecificationMapper orderSpecificationMapper;
    @Resource
    private OrderAddressMapper orderAddressMapper;

    @Resource
    private IPurchaseTaskService purchaseTaskService;

    @Override
    public void publishOrder(Order order) {
        this.save(order);
        order.getOrderDetailList().forEach(orderPlatformSku -> {
            orderPlatformSku.setOrderId(order.getId());
            orderDetailMapper.insert(orderPlatformSku);

            orderPlatformSku.getOrderSpecificationList().forEach(orderSpecification -> {
                orderSpecification.setOrderDetailId(orderPlatformSku.getId());
                orderSpecificationMapper.insert(orderSpecification);
            });
        });

        OrderAddress orderAddress = order.getOrderAddress();
        orderAddress.setOrderId(order.getId());
        orderAddressMapper.insert(orderAddress);

        //todo 从刊登系统获取对应的采购sku和采购成本

    }

    @Override
    public void orderSalesRefund(RefundOrderRequest request) {
        Order order = baseMapper.selectByOrderNoAndPdsAccountId(request.getOriginalOrderNo(), request.getPdsAccountId());
        order.setSalesRefundAmount(request.getSalesRefundAmount());
        //todo 调用自动采购退款
        baseMapper.updateById(order);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void purchase(String id) {
        Order order = this.queryDetail(id);

        if (order == null) {
            throw new EntityNotFoundException();
        }

        List<OrderDetail> orderDetailList = orderDetailMapper.selectByOrderIdAndPurchaseStatus(id, OrderDetail.DELIVERY_STATUS_UN_DELIVERY);
        if (ListUtil.isEmpty(orderDetailList)) {
            throw new BizRuntimeException(ErrorCode.OR_00001);
        }
        List<String> orderDetailIdList = orderDetailList.stream().map(OrderDetail::getId).collect(Collectors.toList());

        purchaseTaskService.addTask(orderDetailIdList, id, PurchaseTaskConstant.ebayPurchaseT);

        this.updateById(order);
    }

    @Override
    public void syncPurchaseStatus() {

    }

    @Override
    public Order queryDetail(String id) {
        Order order = this.getById(id);
        if (order == null) {
            throw new EntityNotFoundException();
        }
        List<OrderDetail> orderDetailList = orderDetailMapper.selectByOrderId(id);
        List<String> orderDetailIdList = orderDetailList.stream().map(OrderDetail::getId).collect(Collectors.toList());
        if (ListUtil.isNotEmpty(orderDetailIdList)) {
            Map<String, List<OrderSpecification>> orderSpecificationMap = orderSpecificationMapper.selectList(new LambdaQueryWrapper<OrderSpecification>()
                    .in(OrderSpecification::getOrderDetailId, orderDetailIdList))
                    .stream()
                    .collect(Collectors.groupingBy(OrderSpecification::getOrderDetailId));

            orderDetailList.forEach(orderPlatformSku -> orderPlatformSku.setOrderSpecificationList(orderSpecificationMap.get(orderPlatformSku.getId())));
        }
        order.setOrderDetailList(orderDetailList);

        OrderAddress orderAddress = orderAddressMapper.selectByOrderId(id);
        order.setOrderAddress(orderAddress);


        return order;
    }


}
