package com.bailuntec.ana.application.purchase.impl;

import com.bailuntec.ana.application.order.IOrderDetailService;
import com.bailuntec.ana.application.order.IOrderService;
import com.bailuntec.ana.application.purchase.IPurchaseOrderDetailService;
import com.bailuntec.ana.application.purchase.IPurchaseOrderService;
import com.bailuntec.ana.domain.order.Order;
import com.bailuntec.ana.domain.order.OrderDetail;
import com.bailuntec.ana.domain.purchase.PurchaseOrder;
import com.bailuntec.ana.domain.purchase.PurchaseOrderDetail;
import com.bailuntec.ana.domain.purchase.PurchaseTask;
import com.bailuntec.ana.infrastructure.common.exception.BizRuntimeException;
import com.bailuntec.ana.infrastructure.common.exception.ErrorCode;
import com.bailuntec.ana.infrastructure.feign.OmsApi;
import com.bailuntec.ana.infrastructure.mapper.order.OrderDetailMapper;
import com.bailuntec.ana.infrastructure.mapper.purchase.PurchaseOrderDetailMapper;
import com.bailuntec.ana.infrastructure.mapper.purchase.PurchaseOrderMapper;
import com.bailuntec.ana.infrastructure.mapper.purchase.PurchaseTaskDetailMapper;
import com.bailuntec.ana.infrastructure.mapper.purchase.PurchaseTaskMapper;
import com.bailuntec.ana.vo.order.Currency;
import com.bailuntec.ana.vo.order.resp.PublishOrderCallBackRequest;
import com.bailuntec.ana.vo.order.resp.PurchaseInfo;
import com.bailuntec.ana.vo.purchase.CallBackPurchaseOrderDetail;
import com.bailuntec.ana.vo.purchase.PmsCallBackResult;
import com.bailuntec.ana.vo.purchase.PurchaseCallBackRequest;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>
 * 服务实现类
 * </p>
 *
 * @author robbendev
 * @since 2020-08-01
 */
@Service
public class PurchaseOrderServiceImpl extends ServiceImpl<PurchaseOrderMapper, PurchaseOrder> implements IPurchaseOrderService {

    @Resource
    PurchaseTaskDetailMapper purchaseTaskDetailMapper;
    @Resource
    PurchaseTaskMapper purchaseTaskMapper;
    @Resource
    OrderDetailMapper orderDetailMapper;
    @Resource
    IOrderDetailService orderDetailService;
    @Resource
    IOrderService orderService;
    @Resource
    OmsApi omsApi;
    @Resource
    PurchaseOrderDetailMapper purchaseOrderDetailMapper;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void callBackPurchaseOrder(PmsCallBackResult<PurchaseCallBackRequest> request) {
        String purchaseTaskId = request.getId();

        //采购任务对应的订单详情列表
        List<OrderDetail> orderDetailList = orderDetailMapper.selectByTaskId(purchaseTaskId);

        PurchaseTask purchaseTask = purchaseTaskMapper.selectById(purchaseTaskId);

        if (!request.getData().getSuccess()) {
            //采购任务失败
            purchaseTask.setStatus(PurchaseTask.STATUS_FAILED);
            purchaseTask.setMessage(request.getData().getMessage());
            purchaseTaskMapper.updateById(purchaseTask);

            //订单详情采购状态采购失败
            orderDetailList.forEach(orderDetail -> {
                if (!orderDetail.getPurchaseStatus().equals(OrderDetail.PURCHASE_STATUS_PURCHASING)) {
                    throw new BizRuntimeException(ErrorCode.PU_10001);
                }
                orderDetail.setPurchaseStatus(OrderDetail.PURCHASE_STATUS_FAILED);
                orderDetail.setPurchaseMessage(request.getData().getMessage());
                orderDetailMapper.updateById(orderDetail);
            });
        } else {

            //采购成功
            purchaseTask.setStatus(PurchaseTask.STATUS_SUCCESS);
            purchaseTaskMapper.updateById(purchaseTask);

            //订单详情采购状态采购成功
            orderDetailList.forEach(orderDetail -> {
                if (!orderDetail.getPurchaseStatus().equals(OrderDetail.PURCHASE_STATUS_PURCHASING)) {
                    throw new BizRuntimeException(ErrorCode.PU_10001);
                }
                orderDetail.setPurchaseStatus(OrderDetail.PURCHASE_STATUS_SUCCESS);
                orderDetailMapper.updateById(orderDetail);
            });

            //生成采购订单
            request.getData().getProducts().forEach((purchaseOrderNo, callBackPurchaseOrderDetailList) -> {

                BigDecimal totalPrice = callBackPurchaseOrderDetailList.stream().map(CallBackPurchaseOrderDetail::getPrice).map(Currency::getAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
                BigDecimal totalTax = callBackPurchaseOrderDetailList.stream().map(CallBackPurchaseOrderDetail::getTax).map(Currency::getAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
                BigDecimal totalShip = callBackPurchaseOrderDetailList.stream().map(CallBackPurchaseOrderDetail::getShip).map(Currency::getAmount).reduce(BigDecimal.ZERO, BigDecimal::add);

                PurchaseOrder purchaseOrder = new PurchaseOrder();
                purchaseOrder.setOrderNo(purchaseOrderNo);
                purchaseOrder.setPurchaseTaskId(purchaseTaskId);
                purchaseOrder.setTotalTax(new Currency(totalTax));
                purchaseOrder.setTotalPrice(new Currency(totalPrice));
                purchaseOrder.setTotalShipCost(new Currency(totalShip));
                purchaseOrder.setOrderId(purchaseTask.getOrderId());
                baseMapper.insert(purchaseOrder);

                //采购订单详情
                callBackPurchaseOrderDetailList.forEach(callBackPurchaseOrderDetail -> {

                    OrderDetail orderDetail = orderDetailList.stream().filter(item -> item.getPurchaseAsin().equals(callBackPurchaseOrderDetail.getAsin())
                            && item.getPurchaseSkuNo().equals(callBackPurchaseOrderDetail.getSku()))
                            .findAny()
                            .orElseThrow(() -> new BizRuntimeException(ErrorCode.PU_10002));


                    PurchaseOrderDetail purchaseOrderDetail = new PurchaseOrderDetail();
                    purchaseOrderDetail.setAsin(callBackPurchaseOrderDetail.getAsin());
                    purchaseOrderDetail.setMerchantId(callBackPurchaseOrderDetail.getMerchant_id());
                    purchaseOrderDetail.setPurchaseOrderId(purchaseOrder.getId());
                    purchaseOrderDetail.setPrice(callBackPurchaseOrderDetail.getPrice());
                    purchaseOrderDetail.setTax(callBackPurchaseOrderDetail.getTax());
                    purchaseOrderDetail.setShipCost(callBackPurchaseOrderDetail.getShip());
                    purchaseOrderDetail.setSku(callBackPurchaseOrderDetail.getSku());
                    purchaseOrderDetail.setOrderDetailId(orderDetail.getId());
                    purchaseOrderDetailMapper.insert(purchaseOrderDetail);


                    orderDetail.setPurchaseStatus(OrderDetail.PURCHASE_STATUS_SUCCESS);
                    orderDetailMapper.updateById(orderDetail);
                });
            });

            Order order = orderService.queryDetail(purchaseTask.getOrderId());
            boolean isPartialPurchase = order.getOrderDetailList().stream().anyMatch(orderDetail -> !orderDetail.getPurchaseStatus().equals(OrderDetail.PURCHASE_STATUS_SUCCESS));
            order.setPurchaseStatus(isPartialPurchase ? PublishOrderCallBackRequest.PURCHASE_STATUS_PARTIAL_PURCHASE : Order.PURCHASE_STATUS_SUCCESS);
            orderService.updateById(order);

            //回调oms系统
            PublishOrderCallBackRequest callBackRequest = new PublishOrderCallBackRequest();
            callBackRequest.setStatus(isPartialPurchase ? PublishOrderCallBackRequest.PURCHASE_STATUS_PARTIAL_PURCHASE : PublishOrderCallBackRequest.PURCHASE_STATUS_SUCCESS);
            callBackRequest.setPdsAccountId(order.getPdsAccountId());

            List<PurchaseOrder> purchaseOrderList = this.listByOrderId(order.getId());
            callBackRequest.setPurchaseInfos(purchaseOrderList.stream().map(purchaseOrder -> {
                PurchaseInfo purchaseInfo = new PurchaseInfo();
                purchaseInfo.setPurchaseOrderNo(purchaseOrder.getOrderNo());
                purchaseInfo.setTax(purchaseOrder.getTotalTax());
                purchaseInfo.setShippingCost(purchaseOrder.getTotalShipCost());
                purchaseInfo.setPlatSkus(purchaseOrder.getPurchaseOrderDetailList().stream().map(purchaseOrderDetail -> purchaseOrderDetail.getOrderDetail()
                        .getSkuNo()).collect(Collectors.toList()));

                return purchaseInfo;
            })
                    .collect(Collectors.toList()));

            omsApi.updateOrderStatusForAna(callBackRequest);
        }
    }

    @Resource
    IPurchaseOrderDetailService purchaseOrderDetailService;

    @Override
    public List<PurchaseOrder> listByOrderId(String orderId) {
        List<PurchaseOrder> purchaseOrderList = baseMapper.selectByOrderId(orderId);
        purchaseOrderDetailService.setPurchaseOrderDetail(purchaseOrderList);
        return purchaseOrderList;
    }

}

