package com.bailuntec.ana.infrastructure.common.exception;


import com.bailuntec.ana.infrastructure.common.base.BaseResult;
import com.bailuntec.ana.infrastructure.common.util.JsonUtilByJackson;
import com.bailuntec.ana.infrastructure.common.wrapper.RequestBakRequestWrapper;
import com.bailuntec.ana.infrastructure.service.mail.MailService;
import lombok.Builder;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.stream.Collectors;

/**
 * 统一异常处理器
 *
 * @author robbendev
 */
@ControllerAdvice
@ResponseBody
@Slf4j
public class GlobalExceptionHandler {

    @Value("${spring.profiles.active}")
    private String profile;

    @Resource
    MailService mailService;

    /**
     * 自定义异常
     */
    @ExceptionHandler(BizRuntimeException.class)
    public BaseResult<String> handleBizRuntimeException(HttpServletRequest request,
                                                        BizRuntimeException e) {
        BaseResult<String> result = new BaseResult<>();
        result.setCode(e.getCode());
        result.setMessage(e.getMessage());

        log.error(e.getMessage());
        return result;
    }

    @ExceptionHandler(ConstraintViolationException.class)
    public BaseResult<String> handleConstraintViolationException(HttpServletRequest request,
                                                                 ConstraintViolationException e) {
        BaseResult<String> result = new BaseResult<>();
        result.setCode("INVALID_PARAMETER");
        result.setMessage(e.getConstraintViolations().stream().map(ConstraintViolation::getMessage).collect(Collectors.joining(",")));
        log.error(e.getMessage());
        return result;
    }


    /**
     * <p>
     * spring 参数校验异常
     * </p>
     *
     * @param request http request
     * @param e       ex
     * @return result
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public BaseResult<String> handleMethodArgumentNotValidException(HttpServletRequest request,
                                                                    MethodArgumentNotValidException e) {
        BaseResult<String> result = new BaseResult<>();
        result.setCode("INVALID_PARAMETER");
        result.setMessage(e.getBindingResult().getAllErrors().stream().map(ObjectError::getDefaultMessage).collect(Collectors.joining(",")));

        log.error(e.getMessage());
        return result;
    }


    /**
     * 自定义异常
     */
    @ExceptionHandler(DuplicateKeyException.class)
    public BaseResult<String> handleDuplicateKeyException(HttpServletRequest request,
                                                          DuplicateKeyException e) {
        BaseResult<String> result = new BaseResult<>();
        result.setCode("500");
        result.setMessage("数据库唯一冲突，请勿重复提交");
        log.error(e.getMessage());
        return result;
    }

    /**
     * 未捕获异常
     */
    @ExceptionHandler(Exception.class)
    public BaseResult<String> handleException(HttpServletRequest request,
                                              Exception ex) {
        BaseResult<String> result = new BaseResult<>();

        result.setCode("500");
        result.setMessage(ex.getMessage());

        RequestBakRequestWrapper bakRequestWrapper = (RequestBakRequestWrapper) request;

        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        ex.printStackTrace(pw);

        ErrorLog errorLog = ErrorLog.builder()
                .uri(bakRequestWrapper.getRequestURI())
                .param(JsonUtilByJackson.writeValueAsString(bakRequestWrapper.getParameterMap()))
                .payload(bakRequestWrapper.getCachedContent().toString())
                .errorMsg(ex.getMessage())
                .StackTrace(sw.toString())
                .build();


        //输出日志
        log.error("******异常开始*******");
        log.info("uri:{}", errorLog.getUri());
        log.info("param:{}", errorLog.getParam());
        log.info("payload:{}", errorLog.getPayload());
        log.error(ex.getMessage(), ex);
        log.error("******异常结束********");

        //错误邮件内容

        //发送邮件
        if (profile.equals("prod"))

            mailService.sendSimpleMail("robbendev@qq.com", profile + "异常", JsonUtilByJackson.writeValueAsString(errorLog));

        return result;
    }


    @Data
    @Builder
    static class ErrorLog {

        private String uri;
        private String traceId;
        private String param;
        private String payload;
        private String errorMsg;
        private String StackTrace;

    }
}
