package com.bailuntec.job;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bailuntec.domain.entity.DcBaseStock;
import com.bailuntec.domain.entity.DcBaseStockConfigFba;
import com.bailuntec.domain.entity.DcBaseStockFba;
import com.bailuntec.domain.entity.JobPointLog;
import com.bailuntec.domain.example.DcBaseStockConfigFbaExample;
import com.bailuntec.domain.example.DcBaseStockExample;
import com.bailuntec.domain.example.DcBaseStockFbaExample;
import com.bailuntec.domain.pojo.FbaStockData;
import com.bailuntec.domain.pojo.FbaStockInfo;
import com.bailuntec.mapper.DcBaseStockConfigFbaMapper;
import com.bailuntec.mapper.DcBaseStockFbaMapper;
import com.bailuntec.mapper.DcBaseStockMapper;
import com.bailuntec.support.PointJob;
import com.bailuntec.utils.OkHttpUtil;
import com.bailuntec.utils.PropertiesUtil;
import com.bailuntec.utils.SessionUtil;
import com.dangdang.ddframe.job.api.ShardingContext;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;

@Slf4j
public class FbaStockJob extends PointJob {
    private PropertiesUtil propertiesUtil = PropertiesUtil.getInstance("const");

    /**
     * 1. 这里是将FBA库存分页查出
     * 放到一个哈希表
     * 然后不做处理直接保存到dc_base_stock_fba留底
     * 2. 要把共享FBA库存找出来,
     * 按优先级配置只保留一个
     * 按bailun_sku + warehouse维度 插入dc_base_stock
     *
     * @param shardingContext
     * @param jobPointLog
     */
    @Override
    public void executeJob(ShardingContext shardingContext, JobPointLog jobPointLog) {
        Integer totalPage = null;
        HashMap<String, DcBaseStockFba> hashmap = new HashMap<>();
        do {
            OkHttpClient client = OkHttpUtil.getInstance();
            MediaType mediaType = MediaType.parse("application/json");
            LinkedHashMap<String, Object> map = new LinkedHashMap<>(3);
            map.put("PageIndex", jobPointLog.getPageIndex());
            map.put("PageRow", jobPointLog.getPageSize());
            RequestBody body = RequestBody.create(mediaType, JSON.toJSONString(map));
            Request request = new Request.Builder()
                    .url(propertiesUtil.getPropertyAsString("STOCK_FBA_URL"))
                    .post(body)
                    .addHeader("Content-Type", "application/json")
                    .build();
            Response response = null;
            String resultStr = null;
            try {
                response = client.newCall(request).execute();
                resultStr = response.body().string();
            } catch (IOException e) {
                throw new RuntimeException("调用wms系统FBA库存接口失败", e);
            } finally {
                if (response != null) {
                    response.close();
                }
            }
            if (StringUtils.isNotBlank(resultStr)) {
                JSONObject jsonObject = JSON.parseObject(resultStr);
                if (jsonObject.get("isSuccess") != null && jsonObject.getBooleanValue("isSuccess")) {
                    FbaStockData fbaStockData = jsonObject.getObject("data", FbaStockData.class);
                    if (fbaStockData != null && fbaStockData.getStocks() != null && fbaStockData.getStocks().size() > 0) {
                        totalPage = fbaStockData.getPageTotal() + 1;//这里是因为接口分页错误
                        for (FbaStockInfo fbaStockInfo : fbaStockData.getStocks()) {
                            /*
                             * 仓库 + 平台SKU维度, 保证FBA库存唯一
                             * 但是同一个bailun_sku 可能当成多个平台SKU在卖
                             */
                            String uniqueKey = fbaStockInfo.getWarehouseCode() + fbaStockInfo.getPlatformSku();
                            DcBaseStockFba dcBaseStockFbaInMap = hashmap.get(uniqueKey);
                            DcBaseStockFba dcBaseStockFba = new DcBaseStockFba();
                            try {
                                BeanUtils.copyProperties(dcBaseStockFba, fbaStockInfo);
                            } catch (Exception e) {
                                throw new RuntimeException("BeanUtils.copyPropertiesFBA库存数据失败");
                            }
                            dcBaseStockFba.setGmtModified(LocalDateTime.now());
                            if (dcBaseStockFbaInMap != null) {
                                dcBaseStockFba.setInWarehouse(dcBaseStockFbaInMap.getInWarehouse() + fbaStockInfo.getInWarehouse());
                                dcBaseStockFba.setUsableStock(dcBaseStockFbaInMap.getUsableStock() + fbaStockInfo.getUsableStock());
                            }
                            hashmap.put(uniqueKey, dcBaseStockFba);
                        }
                    }
                } else {
                    throw new RuntimeException("调用wms系统FBA库存接口返回200但是失败");
                }
            } else {
                throw new RuntimeException("调用wms系统FBA库存接口失败");
            }
            jobPointLog.setPageIndex(jobPointLog.getPageIndex() + 1);
        } while (jobPointLog.getPageIndex() <= totalPage);
        jobPointLog.setPageIndex(1);

        HashMap<String, List<DcBaseStockFba>> platSkuAndListMap = new HashMap<>();
        try {
            DcBaseStockFbaMapper mapper = SessionUtil.getSession().getMapper(DcBaseStockFbaMapper.class);
            hashmap.forEach(((uniqueKey, dcBaseStockFba) -> {
                List<DcBaseStockFba> dcBaseStockFbas = platSkuAndListMap.get(dcBaseStockFba.getPlatformSku());
                if (dcBaseStockFbas == null || dcBaseStockFbas.isEmpty()) {
                    dcBaseStockFbas = new ArrayList<>();
                }
                dcBaseStockFbas.add(dcBaseStockFba);
                platSkuAndListMap.put(dcBaseStockFba.getPlatformSku(), dcBaseStockFbas);
                int i = mapper.updateByExampleSelective(dcBaseStockFba, DcBaseStockFbaExample.newAndCreateCriteria().andWarehouseCodeEqualTo(dcBaseStockFba.getWarehouseCode()).andPlatformSkuEqualTo(dcBaseStockFba.getPlatformSku()).example());
                if (i == 0) {
                    mapper.insertSelective(dcBaseStockFba);
                }
            }));

        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("MYBATIS操作DB存FBA库存失败", e);
        } finally {
            SessionUtil.closeSession();
        }
        /*
         * 在FBA库存插入完成后,
         * 要把共享FBA库存找出来,
         * 只保留一个
         * 按bailun_sku + warehouse维度 插入dc_base_stock
         * @param shardingContexts
         */
        HashMap<String, DcBaseStock> bailunSkuStockMap = new HashMap<>();
        platSkuAndListMap.forEach((platformSKu, stockFbaList) -> {
            //找出同一个平台SKU有多个仓库在卖的情况
            //通过平台SKU去找优先级配置, 找到了就保留优先级配置的库存,其他置0; 如果没有找到, 就随便取一个留着, 其他置为0
            DcBaseStockConfigFbaMapper mapper = SessionUtil.getSession().getMapper(DcBaseStockConfigFbaMapper.class);
            DcBaseStockConfigFba dcBaseStockConfigFba = mapper.selectOneByExample(DcBaseStockConfigFbaExample.newAndCreateCriteria().andPlatformSkuEqualTo(platformSKu).example());
            boolean haveConfig = false;
            for (int i = 0; i < stockFbaList.size(); i++) {
                //如果优先级配置存在, 且优先级的仓库编码 和 列表中某仓库编码一致, 就把其他仓库库存置为0
                //如果优先级配置不存在, 就随便保留一个,  其他仓库库存置为0
                if (dcBaseStockConfigFba != null) {
                    //如果不为空,将stockFbaList每个仓库和优先级的仓库编码匹配,不相等就置为0
                    //如果仓库编码匹配不上, 就让最后一个保留
                    if (i < stockFbaList.size() - 1) {
                        if (stockFbaList.get(i).getWarehouseCode().equals(dcBaseStockConfigFba.getWarehouseCode())) {
                            haveConfig = true;
                        } else {
                            stockFbaList.get(i).setUsableStock(0);
                            stockFbaList.get(i).setInWarehouse(0);
                        }
                    } else {
                        if (haveConfig) {
                            stockFbaList.get(i).setUsableStock(0);
                            stockFbaList.get(i).setInWarehouse(0);
                        }
                    }
                } else {
                    if (i > 0) {
                        stockFbaList.get(i).setUsableStock(0);
                        stockFbaList.get(i).setInWarehouse(0);
                    }
                }
                if (StringUtils.isNotBlank(stockFbaList.get(i).getBailunSku())) {
                    String warehouseBailunSkuKey = stockFbaList.get(i).getWarehouseCode() + stockFbaList.get(i).getBailunSku();
                    DcBaseStock dcBaseStock = bailunSkuStockMap.get(warehouseBailunSkuKey);
                    if (dcBaseStock != null) {
                        dcBaseStock.setInWarehouse(stockFbaList.get(i).getInWarehouse() + dcBaseStock.getInWarehouse());
                        dcBaseStock.setUsableStock(stockFbaList.get(i).getUsableStock() + dcBaseStock.getUsableStock());
                    } else {
                        dcBaseStock = new DcBaseStock();
                        dcBaseStock.setBailunSku(stockFbaList.get(i).getBailunSku());
                        dcBaseStock.setWarehouseCode(stockFbaList.get(i).getWarehouseCode());
                        dcBaseStock.setInWarehouse(stockFbaList.get(i).getInWarehouse());
                        dcBaseStock.setUsableStock(stockFbaList.get(i).getUsableStock());
                    }
                    bailunSkuStockMap.put(warehouseBailunSkuKey, dcBaseStock);
                }
            }
        });

        try {
            DcBaseStockMapper mapper = SessionUtil.getSession().getMapper(DcBaseStockMapper.class);
            bailunSkuStockMap.forEach((warehouseBailunSkuKey, baseStock) -> {
                int i = mapper.updateByExampleSelective(baseStock, DcBaseStockExample.newAndCreateCriteria().andBailunSkuEqualTo(baseStock.getBailunSku()).andWarehouseCodeEqualTo(baseStock.getWarehouseCode()).example());
                if (i == 0) {
                    mapper.insertSelective(baseStock);
                }
            });
        } catch (Exception e) {
            throw new RuntimeException("MYBATIS操作DB存库存失败", e);
        } finally {
            SessionUtil.closeSession();
        }
    }
}
