package com.blt.other.module.cost.service.impl;

import com.bailuntec.cost.api.dto.CostDto;
import com.blt.other.common.util.CurUtils;
import com.blt.other.module.auth.dao.UserDao;
import com.blt.other.module.cost.dao.*;
import com.blt.other.module.cost.service.CostService;
import com.blt.other.module.cost.vo.CostExportVo;
import com.blt.other.module.database.mapper.StatusMapper;
import com.blt.other.module.database.model.CostDetailDomain;
import com.blt.other.module.database.model.CostDomain;
import com.blt.other.module.database.model.CostTypeKindDomain;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.google.common.collect.Lists;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.*;
import java.util.stream.Collectors;

@Service
public class CostServiceImpl implements CostService {

    @Autowired
    private CostDao costDao;

    @Autowired
    private StatusMapper statusMapper;

    @Autowired
    private CostCompanyDao costCompanyDao;

    @Autowired
    private CostTypeDao costTypeDao;

    @Autowired
    private CostTypeKindDao costTypeKindDao;

    @Autowired
    private UserDao userDao;

    @Autowired
    private CostDetailDao costDetailDao;

    /**
     * 保存新费用单
     *
     * @param costDomain
     * @return
     */
    @Override
    public Integer saveNewCost(CostDomain costDomain) {
        if (null != costDomain && null == costDomain.getCostStatus()) {
            costDomain.setCostStatus(0);
        }
        setCostDomain(costDomain);
        // createUsername
        costDomain.setCreateUsername(userDao.select(costDomain.getCreateUsercode()).getUsername());
        // createUserid
        costDomain.setCreateUserid(userDao.select(costDomain.getCreateUsercode()).getUserid());
        // createTime
        costDomain.setCreateTime(new Date());
        Integer insert = costDao.insert(costDomain);
        return insert;
    }

    /**
     * 修改费用单信息
     *
     * @param costDomain
     * @return
     */
    @Override
    public Map<String, Object> upadateCost(CostDomain costDomain) {
        Map<String, Object> result = new HashMap<>();
        // companyName
        setCostDomain(costDomain);
        Integer update = costDao.update(costDomain);
        result.put("msg", "error");
        result.put("result", 0);
        if (null != update && update >= 1) {
            result.put("msg", "success");
            result.put("result", costDao.selectByCostNo(costDomain.getCostNo()));
            result.put("update", update);
        }
        return result;
    }

    private void setCostDomain(CostDomain costDomain) {
        if (null != costDomain.getCompanyNo()) {
            costDomain.setCompanyName(costCompanyDao.selectByNo(costDomain.getCompanyNo()).getCompanyName());
        }
        if (null != costDomain.getKindNo() && !"".equals(costDomain.getKindNo())) {
            CostTypeKindDomain costTypeKindDomain = costTypeKindDao.selectByKindNo(costDomain.getKindNo());
            if (null != costTypeKindDomain) {
                costDomain.setKindName(costTypeKindDomain.getKindName());
                costDomain.setTypeName(costTypeKindDomain.getTypeName());
                costDomain.setTypeNo(costTypeKindDomain.getTypeNo());
            }
        }
    }

    /**
     * 分页获取所有费用单
     *
     * @param pageNum
     * @param pageSize
     * @return
     */
    @Override
    public Map<String, Object> getAllCost(Integer pageNum, Integer pageSize, Integer userId, String projectTypes) {
        Map<String, Object> result = new HashMap<>();

        PageHelper.startPage(pageNum, pageSize);

        List<String> projectTypeList = new ArrayList<>();
        if (!StringUtils.isEmpty(projectTypes)) {
            projectTypeList = Lists.newArrayList(projectTypes.split(",")).stream().filter(str -> !StringUtils.isEmpty(str)).collect(Collectors.toList());
        }

        List<CostDomain> costDomains = costDao.selectAll(userId, projectTypeList);
        PageInfo<CostDomain> pageInfo = new PageInfo<>(costDomains);

        List<CostDto> costDtos = domainToDto(costDomains);

        result.put("costs", costDtos);
        result.put("pageInfo", pageInfo);
        return result;
    }

    /**
     * 根据 costNo 获取 costDto
     *
     * @param costNo
     * @return
     */
    @Override
    public CostDomain getCostByCostNo(String costNo) {
        return costDao.selectByCostNo(costNo);
    }

    /**
     * 获取 status >= costStatus(param) 的费用单
     *
     * @param pageNum
     * @param pageSize
     * @param costStatus
     * @return
     */
    @Override
    public Map<String, Object> getCostByStatus(Integer pageNum, Integer pageSize, Integer costStatus) {
        PageHelper.startPage(pageNum, pageSize);
        List<CostDomain> costDomains = costDao.selectByStatus(costStatus);
        PageInfo<CostDomain> pageInfo = new PageInfo<>(costDomains);

        List<CostDto> costDtos = domainToDto(costDomains);

        Map<String, Object> result = new HashMap<>();
        result.put("pageInfo", pageInfo);
        result.put("costs", costDtos);
        return result;
    }

    /**
     * 获取 status == costStatus(param) 的费用单
     *
     * @param pageNum
     * @param pageSize
     * @param costStatus
     * @return
     */
    @Override
    public Map<String, Object> getByStatu(Integer pageNum, Integer pageSize, Integer costStatus) {
        PageHelper.startPage(pageNum, pageSize);
        List<CostDomain> costDomains = costDao.selectByCostStatus(costStatus);
        PageInfo<CostDomain> pageInfo = new PageInfo<>(costDomains);

        List<CostDto> costDtos = domainToDto(costDomains);

        Map<String, Object> result = new HashMap<>();
        result.put("pageInfo", pageInfo);
        result.put("costs", costDtos);
        return result;
    }

    @Override
    public Map<String, Object> getByLendStatu(Integer pageNum, Integer pageSize, Integer lendStatus) {
        PageHelper.startPage(pageNum, pageSize);
        List<CostDomain> costDomains = costDao.selectByLendStatus(lendStatus);
        PageInfo<CostDomain> pageInfo = new PageInfo<>(costDomains);

        List<CostDto> costDtos = domainToDto(costDomains);

        Map<String, Object> result = new HashMap<>();
        result.put("pageInfo", pageInfo);
        result.put("costs", costDtos);
        return result;
    }

    @Override
    public Map<String, Object> getByCostForm(Integer pageNum, Integer pageSize, Integer costForm) {
        PageHelper.startPage(pageNum, pageSize);
        List<CostDomain> costDomains = costDao.selectByCostForm(costForm);
        PageInfo<CostDomain> pageInfo = new PageInfo<>(costDomains);

        List<CostDto> costDtos = domainToDto(costDomains);

        Map<String, Object> result = new HashMap<>();
        result.put("pageInfo", pageInfo);
        result.put("costs", costDtos);
        return result;
    }

    @Override
    public Map<String, Object> getByCostFormAndStatus(Integer pageNum, Integer pageSize, Integer costForm, Integer costStatus) {
        PageHelper.startPage(pageNum, pageSize);
        List<CostDomain> costDomains = costDao.selectByCostFormAndStatus(costForm, costStatus);
        PageInfo<CostDomain> pageInfo = new PageInfo<>(costDomains);

        List<CostDto> costDtos = domainToDto(costDomains);

        Map<String, Object> result = new HashMap<>();
        result.put("pageInfo", pageInfo);
        result.put("costs", costDtos);
        return result;
    }

    @Override
    public Map<String, Object> getByIsLend(Integer pageNum, Integer pageSize, Integer isLend) {
        PageHelper.startPage(pageNum, pageSize);
        List<CostDomain> costDomains = costDao.selectByLend(isLend);
        PageInfo<CostDomain> pageInfo = new PageInfo<>(costDomains);

        List<CostDto> costDtos = domainToDto(costDomains);

        Map<String, Object> result = new HashMap<>();
        result.put("pageInfo", pageInfo);
        result.put("costs", costDtos);
        return result;
    }

    @Override
    public Map<String, Object> getBySupCostNo(Integer pageNum, Integer pageSize, String supCostNo) {
        PageHelper.startPage(pageNum, pageSize);
        List<CostDomain> costDomains = costDao.selectBySupCostNo(supCostNo);
        PageInfo<CostDomain> pageInfo = new PageInfo<>(costDomains);

        List<CostDto> costDtos = domainToDto(costDomains);

        Map<String, Object> result = new HashMap<>();
        result.put("pageInfo", pageInfo);
        result.put("costs", costDtos);
        return result;
    }

    @Override
    public Map<String, Object> getByKeys(Integer pageNum, Integer pageSize, CostDomain costDomain) {
        PageHelper.startPage(pageNum, pageSize);
        List<CostDomain> costDomainList = costDao.selectByKeys(costDomain);
        PageInfo<CostDomain> pageInfo = new PageInfo<>(costDomainList);

        List<CostDto> costDtos = domainToDto(costDomainList);

        Map<String, Object> result = new HashMap<>();
        result.put("pageInfo", pageInfo);
        result.put("costs", costDtos);
        result.put("size", costDomainList.size());
        return result;
    }

    @Override
    public Map<String, Object> getByKey(Integer pageNum, Integer pageSize, String searchKey, Integer costStatus) {
        PageHelper.startPage(pageNum, pageSize);
        List<CostDomain> costDomainList = costDao.selectByKey(searchKey, costStatus);
        PageInfo<CostDomain> pageInfo = new PageInfo<>(costDomainList);

        List<CostDto> costDtos = domainToDto(costDomainList);

        Map<String, Object> result = new HashMap<>();
        result.put("pageInfo", pageInfo);
        result.put("costs", costDtos);
        result.put("size", costDomainList.size());
        return result;
    }

    @Override
    public Map<String, Object> getByLendAndStatus(Integer pageNum, Integer pageSize, Integer isLend, Integer costStatus) {
        PageHelper.startPage(pageNum, pageSize);
        List<CostDomain> costDomains = costDao.selectByLendAndStatus(isLend, costStatus);
        PageInfo<CostDomain> pageInfo = new PageInfo<>(costDomains);

        List<CostDto> costDtos = domainToDto(costDomains);

        Map<String, Object> result = new HashMap<>();
        result.put("pageInfo", pageInfo);
        result.put("costs", costDtos);
        return result;
    }

    @Override
    public List<CostDto> getByCostPlanNo(String costPlanNo) {
        List<CostDomain> list = costDao.selectByCostPlanNo(costPlanNo);
        List<CostDto> costDtos = domainToDto(list);
        return costDtos;
    }

    @Override
    public void resetCost(String costNo) {
        List<CostDetailDomain> costDetailDomains = costDetailDao.selectListByCostNo(costNo);
        BigDecimal amount = new BigDecimal(0);
        if (null != costDetailDomains && costDetailDomains.size() >= 1) {
            for (CostDetailDomain domain : costDetailDomains) {
                amount = amount.add(domain.getAmount());
            }
            CostDomain costDomain = new CostDomain();
            costDomain.setCostNo(costNo);
            costDomain.setAmount(amount);
            costDomain.setKindName(costDetailDomains.get(0).getKindName() + " 等 " + costDetailDomains.size() + " 项");
            costDao.update(costDomain);
        }
    }

    @Override
    public Integer setStatus(String costNo, int i) {
        CostDomain costDomain = new CostDomain();
        costDomain.setCostNo(costNo);
        costDomain.setCostStatus(i);
        return costDao.update(costDomain);
    }

    @Override
    public Integer updateById(CostDomain costDomain) {
        return costDao.updateById(costDomain);
    }

    @Override
    public CostDomain getByNo(String costNo) {
        return costDao.selectByCostNo(costNo);
    }

    @Override
    public CostDomain getLastCost() {
        return costDao.selectLastCost();
    }

    /**
     * 获取关联借支单   需是未还款、部分还款
     *
     * @return
     */
    @Override
    public List<CostDto> getLinkLendCost(Integer createuserid) {
        List<CostDomain> costDomains = costDao.selectLinkLendCost(createuserid);
        return domainToDto(costDomains);
    }

    @Override
    public Integer getCostStatusByNo(String costNo) {
        return costDao.selectCostStatusByNo(costNo);
    }

    @Override
    public List<CostDto> getCostByRejectStatus() {
        return costDao.getCostByRejectStatus();
    }

    @Override
    public List<String> listCostNo() {
        return costDao.listCostNo();
    }

    @Override
    public List<String> listCostNoAll(CostExportVo costExportVo) {
        return costDao.listCostNoAll(costExportVo);
    }

    @Override
    public Integer updateCashierAnnex(String costNo, String filePath, String downloadUrl) {
        return costDao.updateCashierAnnex(costNo, filePath, downloadUrl);
    }


    @Override
    public CostDomain getCostDomainByNo(String costNo) {
        return costDao.selectByCostNo(costNo);
    }

    @Override
    public CostDto getCostDtoByNo(String costNo) {

        CostDomain costDomain = costDao.selectByCostNo(costNo);
        if (null != costDomain && null != costDomain.getId()) {
            List<CostDomain> costDomainList = new ArrayList<>();
            costDomainList.add(costDomain);
            List<CostDto> costDtos = domainToDto(costDomainList);
            return costDtos.get(0);
        }
        return null;
    }

    /**
     * 根据借支类型获取费用单
     *
     * @param isLend
     * @return
     */
    @Override
    public List<CostDto> getCostByLend(Integer isLend) {
        List<CostDomain> costDomains = costDao.selectByLend(isLend);
        return domainToDto(costDomains);
    }

    /**
     * 转换费用列表 Dt
     *
     * @param costDomains
     * @return
     */
    private List<CostDto> domainToDto(List<CostDomain> costDomains) {
        List<CostDto> costDtos = null;

        if (null != costDomains && costDomains.size() >= 1) {
            costDtos = new ArrayList<>();
            CostDto costDto = null;
            for (CostDomain costDomain : costDomains) {
                costDto = new CostDto();
                BeanUtils.copyProperties(costDomain, costDto);
                costDto.setCostStatusDto(statusMapper.getStatusValue("cost_status", costDomain.getCostStatus()));
                if (null != costDomain.getIsLend() && !"null".equals(costDomain.getIsLend())) {
                    costDto.setIsLendDto(statusMapper.getStatusValue("is_lend", costDomain.getIsLend()));
                }
                if (null != costDomain.getCostForm() && !"null".equals(costDomain.getCostForm())) {
                    costDto.setCostFormDto(statusMapper.getStatusValue("cost_form", costDomain.getCostForm()));
                }
                if (null != costDomain.getLendStatus() && !"null".equals(costDomain.getLendStatus())) {
                    costDto.setLendStatusDto(statusMapper.getStatusValue("lend_status", costDomain.getLendStatus()));
                }
                // 获取文件名
                String filePath = costDomain.getFilePath();
                if (null != filePath && filePath.contains("&")) {
                    costDto.setFileName(filePath.substring(filePath.lastIndexOf("&") + 1));
                }
                // 抵个税
                if (1 == costDto.getCostForm()) {
                    costDto.setIsTaxStr(statusMapper.getStatusValue("is_tax", costDto.getIsTax()));
                }
                // 币种
                costDto.setDicDto(CurUtils.getCurDto(costDto.getDic()));
                costDto.setPayDicDto(CurUtils.getCurDto(costDto.getPayDic()));
//
                DecimalFormat df = new DecimalFormat("#0.00");
                if (null != costDto.getAmount()) {
                    String format = df.format(costDto.getAmount());
                    costDto.setAmountDto(format);
                }
                if (null != costDto.getPayPlanAmount()) {
                    String format = df.format(costDto.getPayPlanAmount());
                    costDto.setPayPlanAmountDto(format);
                }
                costDtos.add(costDto);
            }
            return costDtos;
        }
        return costDtos;
    }
}
