package com.blt.other.module.cost.controller.cost;

import com.bailuntec.cost.api.CostApi;
import com.bailuntec.cost.api.dto.CostDto;
import com.bailuntec.cost.api.dto.LogisticsCostDto;
import com.bailuntec.cost.api.dto.ManageCostDto;
import com.bailuntec.cost.api.dto.WageCostDto;
import com.bailuntec.cost.api.response.CostResult;
import com.blt.other.module.cost.service.CostApiService;
import com.blt.other.module.database.model.CostDomain;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/cost/api")
public class CostApiController implements CostApi {

    @Autowired
    private CostApiService costApiService;

    private static Logger logger = LoggerFactory.getLogger(CostApiController.class);


    @ApiOperation("接受工资单，自动生成付款费用单到财务审核")
    @PostMapping("/pushWageCost")
    public CostResult<Void> pushWageCost(@RequestBody WageCostDto wageCostDto) {
        logger.info("接收推送而来的工资单信息：" + wageCostDto);
        try {
            String costNo = costApiService.pushWageCost(wageCostDto);
            return CostResult.successMsg(costNo);
        } catch (Exception e) {
            logger.error("s生成工资系统付款费用单发生错误：", e);
            return CostResult.successMsg("生成工资系统付款费用单发生错误" + e.getMessage());
        }
    }

    @ApiOperation("接收物流单，自动生成付款费用单到财务审核")
    @PostMapping("/pushLogisticsCost")
    public CostResult<Void> pushLogisticsCost(@RequestBody LogisticsCostDto logisticsCostDto) {
        logger.warn("接收推送而来的物流单信息：" + logisticsCostDto);
        try {
            String costNo = costApiService.generateLogisticsCost(1, logisticsCostDto);
            return CostResult.successMsg(costNo);

        } catch (Exception e) {
            logger.error("生成物流付款费用单发生错误：", e);
            return CostResult.successMsg("生成物流付款费用单发生错误" + e.getMessage());
        }
    }


    @ApiOperation("吴通接收物流单，自动生成收款费用单到财务审核")
    @PostMapping("/pushLogisticsReceipt")
    public CostResult<Void> pushLogisticsReceipt(@RequestBody LogisticsCostDto logisticsCostDto) {
        logger.info("接收推送而来的物流单信息：" + logisticsCostDto);
        try {
            String costNo = costApiService.generateLogisticsCost(2, logisticsCostDto);
            return CostResult.successMsg(costNo);
        } catch (Exception e) {
            logger.error("生成物流收款费用单发生错误：", e);
            return CostResult.error("生成物流收款费用单发生错误" + e.getMessage());
        }
    }

    /**
     * <p>
     * 查询所有的费用单和采购单
     * </p>
     *
     * @param startDateStr   开始时间
     * @param endDateStr     开始时间
     * @param feeSuperType   大类
     * @param feeSubType     小类
     * @param companyValue   公司主体value
     * @param companyName    主体名称
     * @param departmentName 部门
     * @param createUserId   创建用户id
     * @param payUserId      支付用户id
     * @return 单据
     */
    @ApiOperation("查询所有的费用单和采购单")
    @GetMapping("/manageCostList")
    public CostResult<List<ManageCostDto>> manageCostList(@RequestParam(name = "startDate", required = false) String startDateStr,
                                                          @RequestParam(name = "endDate", required = false) String endDateStr,
                                                          @RequestParam(name = "feeSuperType", required = false) String feeSuperType,
                                                          @RequestParam(name = "feeSubType", required = false) String feeSubType,
                                                          @RequestParam(name = "companyValue", required = false) Integer companyValue,
                                                          @RequestParam(name = "companyName", required = false) String companyName,
                                                          @RequestParam(name = "departmentName", required = false) String departmentName,
                                                          @RequestParam(name = "createUserId", required = false) Integer createUserId,
                                                          @RequestParam(name = "payUserId", required = false) Integer payUserId) {
        try {
            if (!StringUtils.isEmpty(departmentName)) {
                departmentName = departmentName.toLowerCase();
            }
            List<ManageCostDto> manageCostDtoList = costApiService.getMangeCostList(startDateStr, endDateStr, feeSuperType, feeSubType, companyValue, companyName, departmentName, createUserId, payUserId);
            return CostResult.success(manageCostDtoList);
        } catch (Exception e) {
            e.printStackTrace();
            return CostResult.error();
        }
    }

    /**
     * <p>
     * 查询所有的物流费用单和采购单
     * </p>
     *
     * @param startDateStr   开始时间
     * @param endDateStr     开始时间
     * @param feeSuperType   大类
     * @param feeSubType     小类
     * @param companyValue   公司主体value
     * @param companyName    主体名称
     * @param departmentName 部门
     * @param createUserId   创建用户id
     * @param payUserId      支付用户id
     * @return 单据
     */
    @ApiOperation("查询所有的物流费用单和采购单")
    @GetMapping("/logisticsCostList")
    public CostResult<List<ManageCostDto>> logisticsCostList(@RequestParam(name = "startDate", required = false) String startDateStr,
                                                             @RequestParam(name = "endDate", required = false) String endDateStr,
                                                             @RequestParam(name = "feeSuperType", required = false) String feeSuperType,
                                                             @RequestParam(name = "feeSubType", required = false) String feeSubType,
                                                             @RequestParam(name = "companyValue", required = false) Integer companyValue,
                                                             @RequestParam(name = "companyName", required = false) String companyName,
                                                             @RequestParam(name = "departmentName", required = false) String departmentName,
                                                             @RequestParam(name = "createUserId", required = false) Integer createUserId,
                                                             @RequestParam(name = "payUserId", required = false) Integer payUserId) {
        try {
            if (!StringUtils.isEmpty(departmentName)) {
                departmentName = departmentName.toLowerCase();
            }
            List<ManageCostDto> manageCostDtoList = costApiService.getLogisticsCostList(startDateStr, endDateStr, feeSuperType, feeSubType, companyValue, companyName, departmentName, createUserId, payUserId);
            return CostResult.success(manageCostDtoList);
        } catch (Exception e) {
            e.printStackTrace();
            return CostResult.error();
        }
    }

    /**
     * <p>
     * 获取资产负债表相关费用单
     * </p>
     *
     * @param startDateStr 开始时间
     * @param endDateStr   结束时间
     * @return 资产负债表相关费用单
     */

    @ApiOperation("获取资产负债表相关费用单")
    @GetMapping("/balanceSheetCost")
    public CostResult<List<CostDto>> balanceSheetCostList(@RequestParam(name = "startDate") String startDateStr,
                                                          @RequestParam(name = "endDate") String endDateStr) {
        logger.warn("获取资产负债表相关费用单");
        try {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
            Date startDate = sdf.parse(startDateStr);
            Date endDate = sdf.parse(endDateStr);
            List<CostDomain> balanceSheetCostList = costApiService.getBalanceSheetCost(startDate, endDate);
            return CostResult.success(balanceSheetCostList.stream().map(CostDomain::castToDto).collect(Collectors.toList()));

        } catch (ParseException e) {
            e.printStackTrace();
            return CostResult.error();
        }
    }

    /**
     * <p>
     * 查询原来传输到数据中心时为待支付状态 变成 已支付状态后的费用单
     * </p>
     *
     * @param costNoList 费用单号
     * @return 费用单
     */
    @ApiOperation("查询原来传输到数据中心时为待支付状态 变成 已支付状态后的费用单")
    @GetMapping("/noPayCost")
    public CostResult<List<CostDto>> getNoPayCost(@RequestParam(name = "costNoList") List<String> costNoList) {
        if (costNoList != null && costNoList.size() > 0) {
            List<CostDomain> costDomainList = costApiService.getNoPayCost(costNoList);
            return CostResult.success(costDomainList.stream().map(CostDomain::castToDto).collect(Collectors.toList()));
        } else {
            return CostResult.error();
        }
    }

    /**
     * 吴通新增
     * 根据费用单号获取费用单大类
     *
     * @param costNo 费用单号
     * @return 费用单大类
     */
    @ApiOperation("根据费用单号获取费用单大类")
    @GetMapping("/type/{costNo}")
    public CostResult<String> getTypeName(@PathVariable("costNo") String costNo) {
        // 获取费用单详细信息
        try {
            CostDomain costDomain = costApiService.getCostDetails(costNo);
            return CostResult.success(costDomain.getTypeName());
        } catch (Exception e) {
            e.printStackTrace();
            return CostResult.error();
        }
    }
}
