package com.blt.other.module.purchasing.service.impl;

import com.blt.other.common.config.property.CostUrlProperties;
import com.blt.other.module.auth.dao.UserDao;
import com.blt.other.module.auth.vo.UserGetByBLUserAcct;
import com.blt.other.common.util.UserFromOAUtil;
import com.blt.other.module.database.model.*;
import com.blt.other.module.entrepot.service.WarehouseService;
import com.blt.other.module.purchasing.dao.BuyPlanDao;
import com.blt.other.module.purchasing.dao.BuyPlanDetailDao;
import com.blt.other.module.purchasing.dao.PlanSkuTempDao;
import com.blt.other.module.purchasing.dto.BuyPlanDto;
import com.blt.other.module.purchasing.service.BuyPlanService;
import com.blt.other.module.purchasing.service.BuyService;
import com.blt.other.module.purchasing.vo.BuyPlanSaveVo;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;

@Service
public class BuyPlanServiceImpl implements BuyPlanService {

    private static Logger logger = LoggerFactory.getLogger(BuyPlanServiceImpl.class);
    @Autowired
    private BuyPlanDao buyPlanDao;
    @Autowired
    private PlanSkuTempDao planSkuTempDao;
    @Autowired
    private UserDao userDao;
    @Autowired
    private BuyPlanDetailDao buyPlanDetailDao;
    @Autowired
    private BuyService buyService;
    @Autowired
    private WarehouseService warehouseService;


    @Override
    public Map<String, Object> getAllBuyPlanList(int pageSize, int page) {
        PageHelper.startPage(page, pageSize);
        List<BuyPlanDomain> list = buyPlanDao.selectAllBuyPlan();
        List<BuyPlanDto> buyPlanDtoList = changeAllBuyPlanDtoList(list);
        PageInfo<BuyPlanDomain> pageInfo = new PageInfo<>(list);

        Map<String, Object> map = new HashMap<>();
        map.put("buyPlanList", buyPlanDtoList);
        map.put("pageInfo", pageInfo);

        return map;
    }

    @Override
    public Map<String, Object> getBuyPlanListBySearchNo(String no, Integer pageNum, Integer pageSize) {
        PageHelper.startPage(pageNum, pageSize);

        List<BuyPlanDomain> buyPlanDomains = buyPlanDao.selectBuyPlanByNo(no);
        List<BuyPlanDto> buyPlanDtoList = changeAllBuyPlanDtoList(buyPlanDomains);
        PageInfo<BuyPlanDomain> pageInfo = new PageInfo<>(buyPlanDomains);

        Map<String, Object> map = new HashMap<>();
        map.put("plans", buyPlanDtoList);
        map.put("pageInfo", pageInfo);
        return map;
    }

    /**
     * 根据 no 获取采购计划信息
     *
     * @param no
     * @return
     */
    @Override
    public BuyPlanDto getBuyPlanByNo(String no) {
        List<BuyPlanDomain> list = buyPlanDao.selectBuyPlanByNo(no);
        return domainToDto(list.get(0));
    }

    /**
     * 保存新创建的采购订单
     *
     * @return
     */
    @Override
    public Integer saveNewBuyPlan(BuyPlanSaveVo buyPlanSaveVo) {
        BuyPlanDomain buyPlanDomain = new BuyPlanDomain();
        // 通过从 cookie 中获取的 code ，从数据库中获取用户姓名
        UserDomain user = null;
        if (null != buyPlanSaveVo.getBLUserAcct()) {
            user = getUserNameByCode(buyPlanSaveVo.getBLUserAcct());
        } else {
            // 如果 BLUserAcct 为 Null ，则通过 usercode 获取用户信息
            user = userDao.select(buyPlanSaveVo.getUsercode());
        }
        // 从 paln_sku_temp 获取所有 sku
        List<PlanSkuTempDomain> planSkuTempDomains = planSkuTempDao.selectListByNo(buyPlanSaveVo.getNo());
        BigDecimal amount = new BigDecimal(0.0000);
        Integer count = 0;
        for (PlanSkuTempDomain planSkuTempDomain : planSkuTempDomains) {
            BigDecimal skucount = new BigDecimal(planSkuTempDomain.getSkucount());
            BigDecimal skuprice = planSkuTempDomain.getSkuprice();
            BigDecimal multiply = skuprice.multiply(skucount);
            amount = amount.add(multiply);

            count += planSkuTempDomain.getSkucount();

        }
        buyPlanDomain.setNo(buyPlanSaveVo.getNo());
        buyPlanDomain.setStatus(0);
        buyPlanDomain.setCompany(buyPlanSaveVo.getCompanyName());
        buyPlanDomain.setDepartmentname(buyPlanSaveVo.getDepartmentName());
        // 接收仓库改为由数据库表提供
        WarehouseDomain warehouseDomain = warehouseService.getByWarehosueid(buyPlanSaveVo.getWarehouseid());
        buyPlanDomain.setWarehousename(warehouseDomain.getWarehousename());
        buyPlanDomain.setWarehouseid(warehouseDomain.getWarehouseid());
        buyPlanDomain.setAmount(amount);
        buyPlanDomain.setCount(count);
        buyPlanDomain.setIsurgency(0);
        buyPlanDomain.setIsdelete(0);
        buyPlanDomain.setRemark(buyPlanSaveVo.getRemark());
        buyPlanDomain.setAuditstatus(1);
        buyPlanDomain.setCreatetime(new Date());
        buyPlanDomain.setCreateusername(user.getUsername());
        buyPlanDomain.setCreateuserid(user.getUserid());
        buyPlanDomain.setAudittime(null);

        // 执行 sql ，insert 到数据库
        // 查询是否已经存在相同的采购计划单
        List<BuyPlanDomain> list = buyPlanDao.selectBuyPlanByNo(buyPlanSaveVo.getNo());
        if (list != null && list.size() >= 1 && buyPlanSaveVo.getNo().equals(list.get(0).getNo())) {
            return null;
        }
        Integer result = result = buyPlanDao.insertNewBuyPlan(buyPlanDomain);
        logger.info(user + "保存采购计划单到表 buy_plan 成功：" + buyPlanSaveVo.getNo());
        return result;
    }

    /**
     * 更改采购计划单信息
     *
     * @param buyPlanDomain
     * @return
     */
    @Override
    public Integer updateBuyPlan(BuyPlanDomain buyPlanDomain) {
        Integer integer = buyPlanDao.updateBuyPlan(buyPlanDomain);
        return integer;
    }

    @Override
    public Integer batchAuditUpdate(List<String> planList, Integer audituserid, String auditusername) {
        if (null == planList || planList.size() < 1) {
            return null;
        }
        int count = 0;
        BuyPlanDomain buyPlanDomain = new BuyPlanDomain();
        for (String planNo : planList) {
            buyPlanDomain.setNo(planNo);
            buyPlanDomain.setStatus(3);
            buyPlanDomain.setAuditusername(auditusername);
            buyPlanDomain.setAudituserid(audituserid);
            buyPlanDomain.setAudittime(new Date());
            Integer integer = buyPlanDao.updateBuyPlan(buyPlanDomain);
            if (null != integer && integer == 1) {
                count += 1;
            }
        }
        return count;
    }

    /**
     * 采购计划确认
     *
     * @param plan
     * @return
     */
    @Transactional(readOnly = false, rollbackFor = Exception.class)
    @Override
    public Integer auditPass(BuyPlanDomain plan) {
        // 根据供应商进行拆单
        List<BuyPlanDetailDomain> buyPlanDetailDomains = buyPlanDetailDao.selectBuyPlanDetailByno(plan.getNo());
        if (null == buyPlanDetailDomains || buyPlanDetailDomains.size() < 1) {
            return 0;
        } else {
            // 生成采购单
            // 将商品列表按照供应商分别保存到 List 集合，并以供应商编号作为 key 保存到 map 集合
            Map<String, List<BuyPlanDetailDomain>> buys = new HashMap<>();
            for (BuyPlanDetailDomain detail : buyPlanDetailDomains) {
                List<BuyPlanDetailDomain> o = buys.get(detail.getSupplierNo());
                if (null != o && o.size() >= 1) {
                    o.add(detail);
                    buys.put(detail.getSupplierNo(), o);
                } else {
                    List<BuyPlanDetailDomain> newList = new ArrayList<>();
                    newList.add(detail);
                    buys.put(detail.getSupplierNo(), newList);
                }
            }
            // 遍历 map 集合，并执行生成采购单流程
            if (null != buys && buys.size() >= 1) {
                Set<String> keys = buys.keySet();
                for (String key : keys) {
                    List<BuyPlanDetailDomain> buy = (List<BuyPlanDetailDomain>) buys.get(key);
                    Integer save = buyService.saveBuyFromPlanDetailList(buy, plan);
                }
            }
            // 更改计划单信息
            plan.setStatus(1); // 确认计划
            plan.setAudittime(new Date());
            plan.setModifytime(new Date());
            plan.setModifyuserid(plan.getAudituserid());
            plan.setModifyusername(plan.getAuditusername());
            Integer integer = buyPlanDao.updateBuyPlan(plan);
            return buys.size();
        }
    }

    /**
     * 实体转换
     *
     * @param domains
     * @return
     */
    @Override
    public List<BuyPlanDto> getDtoList(List<BuyPlanDomain> domains) {
        if (null != domains && domains.size() >= 1) {
            List<BuyPlanDto> dtos = new ArrayList<>();
            for (BuyPlanDomain domain : domains) {
                BuyPlanDto dto = domainToDto(domain);
                dtos.add(dto);
            }
            return dtos;
        }
        return null;
    }

    @Resource
    CostUrlProperties costUrlProperties;
    /**
     * 从 user 表中获取用户信息
     *
     * @return
     */
    private UserDomain getUserNameByCode(String BLUserAcct) {
        // 1、从 OA 系统获取 userid
        UserGetByBLUserAcct userMsg = UserFromOAUtil.getUserMsg(BLUserAcct, costUrlProperties.logingUserApi);
        logger.warn("getUserFromOA：" + userMsg);
        // 2、通过 userid 获取 Userdomain
        Integer userid = userMsg.getUserid();
        UserDomain select = userDao.selectByuserid(userid);
        return select;
    }

    /**
     * 将从数据库中获取的 BuyPlanDomain 转换为前端页面访问的 Dto
     *
     * @param list
     * @return
     */
    public List<BuyPlanDto> changeAllBuyPlanDtoList(List<BuyPlanDomain> list) {
        List<BuyPlanDto> buyPlanDtoList = new ArrayList<>();
        for (BuyPlanDomain buyPlanDomain : list) {
            buyPlanDtoList.add(domainToDto(buyPlanDomain));
        }
        return buyPlanDtoList;
    }

    /**
     * BuyPlanDomain 转化为 BuyPlanDto
     *
     * @param buyPlanDomain
     * @return
     */
    private BuyPlanDto domainToDto(BuyPlanDomain buyPlanDomain) {
        BuyPlanDto buyPlanDto = new BuyPlanDto();
        BeanUtils.copyProperties(buyPlanDomain, buyPlanDto);

        Integer status = buyPlanDomain.getStatus();
        if (status == 0) {
            buyPlanDto.setStatusdto("待确认");
        } else if (status == 1) {
            buyPlanDto.setStatusdto("已确认");
        } else if (status == 3) {
            buyPlanDto.setStatusdto("作废");
        }
//        else if (status == 2){
//            buyPlanDto.setStatusdto("主管审核通过");
//        }else if (status == 4){
//            buyPlanDto.setStatusdto("审核驳回");
//        }

        Integer auditStatus = buyPlanDomain.getAuditstatus();
        if (auditStatus == 0) {
            buyPlanDto.setAuditstatusdto("不审核");
        } else if (auditStatus == 1) {
            buyPlanDto.setAuditstatusdto("待审核");
        } else if (auditStatus == 2) {
            buyPlanDto.setAuditstatusdto("审核通过");
        } else {
            buyPlanDto.setAuditstatusdto("审核驳回");
        }
        return buyPlanDto;
    }
}
