package com.blt.other.module.cost.service.impl.cost;

import com.bailuntec.api.bailuntec.cw.CwApi;
import com.bailuntec.api.bailuntec.cw.request.PostApplyReq;
import com.bailuntec.common.JsonUtilByFsJson;
import com.bailuntec.common.JsonUtilByJackson;
import com.bailuntec.common.ListUtil;
import com.bailuntec.cost.api.dto.CostDto;
import com.bailuntec.cost.api.dto.CostListPrintDto;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.blt.other.common.config.property.CostUrlProperties;
import com.blt.other.common.exception.BizRuntimeException;
import com.blt.other.common.util.CurUtils;
import com.blt.other.common.util.MoneyUtil;
import com.blt.other.common.util.SessionUtils;
import com.blt.other.module.auth.dao.CostReviewerMapper;
import com.blt.other.module.auth.dao.OaDepartmentMapper;
import com.blt.other.module.auth.dao.OaUserMapper;
import com.blt.other.module.auth.model.CostReviewer;
import com.blt.other.module.auth.model.OaDepartment;
import com.blt.other.module.auth.model.OaUser;
import com.blt.other.module.auth.service.UserService;
import com.blt.other.module.cost.dao.*;
import com.blt.other.module.cost.dto.CheckCostListReq;
import com.blt.other.module.cost.dto.CostPageResult;
import com.blt.other.module.cost.dto.UpdateCostResp;
import com.blt.other.module.cost.model.CostDetailDomain;
import com.blt.other.module.cost.model.CostDomain;
import com.blt.other.module.cost.model.CostTemplateBaseCol;
import com.blt.other.module.cost.service.*;
import com.blt.other.module.cost.vo.ApplyCallbackUrlDataDataVo;
import com.blt.other.module.cost.vo.ApplyCallbackUrlVo;
import com.blt.other.module.cost.vo.CashierCallbackUrlDataDataVo;
import com.blt.other.module.cost.vo.CashierCallbackUrlVo;
import com.blt.other.module.database.model.*;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * <p>
 *
 * </p>
 *
 * @author robbendev
 * @since 2020/10/29 11:35 上午
 */
@Slf4j
public abstract class AbstractCostService implements CostService {

    @Autowired
    CwApi cwApi;
    @Autowired
    CostTofinanceService costTofinanceService;
    @Autowired
    CostDao costDao;
    @Autowired
    UserService userService;
    @Autowired
    ICostTemplateService costTemplateService;
    @Autowired
    CostCompanyDao costCompanyDao;
    @Autowired
    CostTypeKindDao costTypeKindDao;
    @Autowired
    CostDetailDao costDetailDao;
    @Autowired
    CostLogService costLogService;
    @Resource
    CostUrlProperties costUrlProperties;
    @Resource
    OaDepartmentMapper oaDepartmentMapper;

    @Override
    public Integer saveNewCost(CostDomain costDomain) {
        throw new BizRuntimeException("deprecated method");
    }

    @Override
    public CostPageResult getAllCost(Integer pageNum,
                                     Integer pageSize,
                                     Integer userId,
                                     String projectTypes) {
        PageHelper.startPage(pageNum, pageSize);
        List<String> projectTypeList = new ArrayList<>();
        if (!StringUtils.isEmpty(projectTypes)) {
            projectTypeList = Lists.newArrayList(projectTypes.split(",")).stream().filter(str -> !StringUtils.isEmpty(str)).collect(Collectors.toList());
        }
        List<CostDomain> costDomains = costDao.selectAll(userId, projectTypeList);
        this.setPrimaryDepartment(costDomains);

        CostPageResult result = new CostPageResult();
        result.setCosts(costDomains.stream().map(CostDomain::castToDto).collect(Collectors.toList()));
        result.setPageInfo(new PageInfo<>(costDomains));
        return result;
    }

    @Override
    public void setPrimaryDepartment(List<CostDomain> costDomains) {
        List<OaUser> oaUserList = oaUserMapper.selectList(
                new LambdaQueryWrapper<OaUser>()
                        .in(OaUser::getOaUserId, costDomains.stream().map(CostDomain::getCreateUserid).collect(Collectors.toList()))
        );
        if (ListUtil.isNotEmpty(oaUserList)) {
            Map<Integer, OaDepartment> departmentMap = oaDepartmentMapper.selectList(
                    new LambdaQueryWrapper<OaDepartment>()
                            .in(OaDepartment::getDepartmentId, oaUserList.stream()
                                    .map(OaUser::getPrimaryDepartmentId)
                                    .filter(Objects::nonNull)
                                    .collect(Collectors.toList())))
                    .stream()
                    .collect(Collectors.toMap(OaDepartment::getDepartmentId, oaDepartment -> oaDepartment));

            Map<Integer, OaUser> oaUserMap = oaUserList.stream()
                    .peek(oaUser -> oaUser.setPrimaryDepartment(departmentMap.get(oaUser.getPrimaryDepartmentId())))
                    .collect(Collectors.toMap(OaUser::getOaUserId, oaUser -> oaUser));

            costDomains.forEach(costDomain -> {
                costDomain.setPrimaryDepartmentId(oaUserMap.get(costDomain.getCreateUserid()).getPrimaryDepartmentId());
                costDomain.setPrimaryDepartmentName(oaUserMap.get(costDomain.getCreateUserid()).getPrimaryDepartment().getName());
            });
        }

    }

    @Resource
    CostReviewerMapper costReviewerMapper;
    @Resource
    OaUserMapper oaUserMapper;

    @Override
    public CostDomain getCostByCostNo(String costNo) {
        CostDomain costDomain = costDao.selectByCostNo(costNo);
        costDomain.setCostTemplate(costTemplateService.queryDetail(costDomain.getCostTemplateId()));

        CostCompanyDomain costCompany = costCompanyDao.selectByNo(costDomain.getCompanyNo());
        Integer currentUserId = SessionUtils.getCurrentUserId();
        OaUser costUser = oaUserMapper.selectByOaUserId(costDomain.getCreateUserid());

        if (currentUserId == null) {
            costDomain.setCanAudit(false);
        } else if (costDomain.getCostStatus().equals(CostDomain.STATUS_DEPARTMENT_CHECK)) {
            costDomain.setCanAudit(costReviewerMapper.queryOne(costUser.getPrimaryDepartmentId(), CostReviewer.departmentReviewer, currentUserId) != null);
        } else if (costDomain.getCostStatus().equals(CostDomain.STATUS_FINANCIAL_CHECK)) {
            costDomain.setCanAudit(costReviewerMapper.queryOne(costCompany.getId(), CostReviewer.financialReviewer, currentUserId) != null);
        } else if (costDomain.getCostStatus().equals(CostDomain.STATUS_FINAL_CHECK)) {
            costDomain.setCanAudit(costReviewerMapper.queryOne(costCompany.getId(), CostReviewer.finalReviewer, currentUserId) != null);
        } else if (costDomain.getCostStatus().equals(CostDomain.STATUS_HR_CHECK)) {
            costDomain.setCanAudit(costReviewerMapper.queryOne(costCompany.getId(), CostReviewer.hrReviewer, currentUserId) != null);
        } else {
            costDomain.setCanAudit(false);
        }
//        //todo
//        costDomain.setCanAudit(true);
        return costDomain;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public UpdateCostResp updateCost(CostDomain costDomain) {
        UpdateCostResp resp = new UpdateCostResp();

        setCostDomain(costDomain);
        costDomain.setLastModifyDate(LocalDateTime.now());
        Integer update = costDao.update(costDomain, new LambdaQueryWrapper<CostDomain>()
                .eq(CostDomain::getCostNo, costDomain.getCostNo()));


        resp.setMsg("success");
        resp.setResult(costDao.selectByCostNo(costDomain.getCostNo()));
        resp.setUpdate(update);
        return resp;

    }

    @Resource
    CostTemplateBaseColMapper costTemplateBaseColMapper;

    private void setCostDomain(CostDomain costDomain) {
        if (null != costDomain.getCompanyNo()) {
            costDomain.setCompanyName(costCompanyDao.selectByNo(costDomain.getCompanyNo()).getCompanyName());
        }
        if (null != costDomain.getKindNo() && !"".equals(costDomain.getKindNo())) {
            CostTypeKindDomain costTypeKindDomain = costTypeKindDao.selectByKindNo(costDomain.getKindNo());
            if (null != costTypeKindDomain) {
                costDomain.setKindName(costTypeKindDomain.getKindName());
                costDomain.setTypeName(costTypeKindDomain.getTypeName());
                costDomain.setTypeNo(costTypeKindDomain.getTypeNo());
            }
        }


        if (ListUtil.isNotEmpty(costDomain.getAttach())) {
            costDomain.getAttach().forEach(costAttach -> {
                CostTemplateBaseCol costTemplateBaseCol = costTemplateBaseColMapper.selectById(costAttach.getCostTemplateBaseColId());
                BeanUtils.copyProperties(costTemplateBaseCol, costAttach, "id");
            });
        }
    }

    @Override
    public List<CostDto> getByCostPlanNo(String costPlanNo) {
        return costDao.selectByCostPlanNo(costPlanNo)
                .stream()
                .map(CostDomain::castToDto)
                .collect(Collectors.toList());
    }

    @Override
    public void resetCost(String costNo) {
        List<CostDetailDomain> costDetailDomains = costDetailDao.selectListByCostNo(costNo);
        BigDecimal amount = new BigDecimal(0);
        if (null != costDetailDomains && costDetailDomains.size() >= 1) {
            for (CostDetailDomain domain : costDetailDomains) {
                amount = amount.add(domain.getAmount());
            }
            CostDomain costDomain = new CostDomain();
            costDomain.setCostNo(costNo);
            costDomain.setAmount(amount);

            costDomain.setKindNo(costDetailDomains.get(0).getKindNo());
            costDomain.setKindName(costDetailDomains.get(0).getKindName());
            costDomain.setLastModifyDate(LocalDateTime.now());
            costDao.update(costDomain, new LambdaQueryWrapper<CostDomain>()
                    .eq(CostDomain::getCostNo, costDomain.getCostNo()));
        }
    }

    @Override
    public Integer setStatus(String costNo, int i) {
        CostDomain costDomain = new CostDomain();
        costDomain.setCostNo(costNo);
        costDomain.setCostStatus(i);
        costDomain.setLastModifyDate(LocalDateTime.now());
        return costDao.update(costDomain, new LambdaQueryWrapper<CostDomain>()
                .eq(CostDomain::getCostNo, costDomain.getCostNo()));
    }

    @Override
    public Integer updateById(CostDomain costDomain) {
        costDomain.setLastModifyDate(LocalDateTime.now());
        return costDao.updateById(costDomain);
    }

    @Override
    public void updateCashierAnnex(String costNo, String filePath, String downloadUrl) {
        CostDomain costDomain = costDao.selectByCostNo(costNo);
        costDomain.setCashierFilePath(filePath);
        costDomain.setCashierDownloadPath(downloadUrl);
        costDomain.setLastModifyDate(LocalDateTime.now());
        costDao.updateById(costDomain);
    }

    @Override
    public List<CostListPrintDto> printList(List<String> printNos) {
        return printNos.stream().map(this::printDto).collect(Collectors.toList());
    }

    @Transactional
    @Override
    public IPage<CostDto> checkCostList(CheckCostListReq req) {
        IPage<CostDomain> page = new Page<>(req.getPageNum(), req.getPageSize());

        return costDao.checkCostList(page, req).convert(CostDomain::castToDto);
    }

    @Resource
    CostApplycallbackService costApplycallbackService;

    @Override
    public void applyCallback(ApplyCallbackUrlVo applyCallbackUrlVo) {
        // 执行保存反馈信息流程
        CostApplycallbackDomain applycallbackDomain = new CostApplycallbackDomain();

        ApplyCallbackUrlDataDataVo applyCallbackUrlDataDataVo = JsonUtilByJackson.readValue(
                applyCallbackUrlVo.getData().getData()
                        .replaceAll("/^['|\"](.*)['|\"]$/", ""),
                ApplyCallbackUrlDataDataVo.class
        );

        BeanUtils.copyProperties(applyCallbackUrlDataDataVo, applycallbackDomain);
        String costNo = applyCallbackUrlVo.getId().toUpperCase();
        if (costNo.contains("-")) {
            costNo = costNo.split("-")[0];
        }
        applycallbackDomain.setCostNo(costNo);
        if (com.bailuntec.common.StringUtils.isNotEmpty(applyCallbackUrlVo.getData().getMessage())) {
            applycallbackDomain.setMessage(applyCallbackUrlVo.getData().getMessage().toUpperCase());
        }
        applycallbackDomain.setApplyno(applycallbackDomain.getApplyno().toUpperCase());
        costApplycallbackService.saveApplycallbackResponse(applycallbackDomain);

        log.info("保存财务审核信息成功：");
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void cashierCallback(CashierCallbackUrlVo cashierCallbackUrlVo) {
        CostCashiercallbackDomain costCashiercallbackDomain = new CostCashiercallbackDomain();
        CashierCallbackUrlDataDataVo cashierCallbackUrlDataDataVo = JsonUtilByFsJson.jsonToBean(
                cashierCallbackUrlVo.getData().getData().replaceAll("/^['|\"](.*)['|\"]$/", ""),
                CashierCallbackUrlDataDataVo.class
        );

        BeanUtils.copyProperties(cashierCallbackUrlVo.getData(), costCashiercallbackDomain);
        BeanUtils.copyProperties(cashierCallbackUrlDataDataVo, costCashiercallbackDomain);
        BeanUtils.copyProperties(cashierCallbackUrlDataDataVo.getPaydetail(), costCashiercallbackDomain);

        String costNo = cashierCallbackUrlVo.getId().toUpperCase();
        if (costNo.contains("-")) {
            String[] split = costNo.split("-");
            costNo = split[0];
        }
        costCashiercallbackDomain.setCostNo(costNo);
        if (com.bailuntec.common.StringUtils.isNotEmpty(costCashiercallbackDomain.getMessage())) {
            costCashiercallbackDomain.setMessage(costCashiercallbackDomain.getMessage().toUpperCase());
        }

        costCashiercallbackDomain.setPayno(costCashiercallbackDomain.getPayno().toUpperCase());

        CostDomain costDomain = costDao.selectByCostNo(costNo);
        //
        if (!costDomain.getCostStatus().equals(CostDomain.STATUS_UN_PAY)) {
            throw new BizRuntimeException("invalid status");
        }
        //驳回
        if (costCashiercallbackDomain.getMessage().contains("被驳回")) {
            this.cashierCallbackReject(costCashiercallbackDomain);
        }
        //通过
        else {
            this.cashierCallbackPass(costCashiercallbackDomain);
        }
    }

    @Override
    public void reject(String costNo) {

        if (costNo.startsWith("S")) {
            //费用计划
            CostDomain costDomain = costDao.selectByCostNo(costNo);
            //费用计划关联借支单
            CostDomain supCost = costDao.selectByCostNo(costDomain.getSupCostNo());
            supCost.setCompensate(supCost.getCompensate().subtract(costDomain.getAmount()));
            costDomain.setLastModifyDate(LocalDateTime.now());
            costDao.updateById(supCost);
        }


        CostDomain costDomain = costDao.selectByCostNo(costNo);
        costDomain.setCostStatus(CostDomain.STATUS_REJECT);
        costDomain.setLastModifyDate(LocalDateTime.now());
        costDao.updateById(costDomain);
    }

    @Resource
    CostCashiercallbackService costCashiercallbackService;

    void cashierCallbackReject(CostCashiercallbackDomain costCashiercallbackDomain) {
        // 保存驳回信息
        costCashiercallbackService.saveCostCashiercallbackResponse(costCashiercallbackDomain);
        // 费用单状态改为审核驳回，添加驳回理由
        // 修复驳回后提交出现费用单已存在的问题
        // 将后缀添加到  detail_key 字段
        CostDomain cost = this.getCostByCostNo(costCashiercallbackDomain.getCostNo());
        String oldDetailKey = cost.getDetailKey();

        if (null != oldDetailKey && oldDetailKey.contains("-")) {
            String[] split = oldDetailKey.split("-");
            if (split.length > 1) {
                int i = Integer.parseInt(split[1]) + 1;
                cost.setDetailKey(split[0] + "-" + i);
            }
        } else {
            cost.setDetailKey(cost.getCostNo() + "-" + 1);
        }
        cost.setRejectReason(costCashiercallbackDomain.getPaynote());
        cost.setCostStatus(3);
        cost.setRejectType(2);
        this.updateById(cost);
        log.info("出纳驳回，费用单状态更改成功" + costCashiercallbackDomain);

        costLogService.save(cost.getCostNo(), costCashiercallbackDomain.getPayuserid(), "被出纳驳回：" + costCashiercallbackDomain.getPaynote());
    }

    void cashierCallbackPass(CostCashiercallbackDomain costCashiercallbackDomain) {
        // 获取xx货币-->CNY的汇率
        BigDecimal toRmbRate = CurUtils.getCur(costCashiercallbackDomain.getCashierunitcode().toUpperCase(), "CNY");
        costCashiercallbackDomain.setToRmbRate(toRmbRate);

        //  保存出纳付款记录
        Integer integer = costCashiercallbackService.saveCostCashiercallbackResponse(costCashiercallbackDomain);
        log.warn("保存出纳付款信息成功：" + integer + "   " + costCashiercallbackDomain);

        //  付款成功，更改费用单信息，否则返回失败
        CostDomain costDomain = this.getCostByCostNo(costCashiercallbackDomain.getCostNo());
        costDomain.setCostStatus(4);

        if (null != costDomain.getIsLend()) {
            if (costDomain.getIsLend() == 1 && BigDecimal.ZERO.compareTo(costDomain.getHadPay()) >= 0) {
                costDomain.setLendStatus(1);
            }
        }
        costDomain.setPayUserId(costCashiercallbackDomain.getPayuserid());
        costDomain.setPayTime(new Date());
        costDomain.setAmountRmb(costCashiercallbackDomain.getCashierpaymoneyrmb());
        costDomain.setToRmbRate(toRmbRate);
        this.updateCost(costDomain);
        costLogService.save(costCashiercallbackDomain.getCostNo(), costCashiercallbackDomain.getPayuserid(), "出纳收/付款成功：" + costCashiercallbackDomain.getPaynote());
    }

    private CostListPrintDto printDto(String costNo) {
        CostListPrintDto print = new CostListPrintDto();
        CostDto cost = this.getCostByCostNo(costNo).castToDto();
        UserDomain user = userService.findByUserid(cost.getCreateUserid());
        if (user == null) {
            print.setDepartment("");
            print.setCreateUsername(cost.getCreateUsername() + "【注销】");
        } else {
            print.setDepartment(user.getDepartmentname1());
            print.setCreateUsername(cost.getCreateUsername());
        }
        print.setId(cost.getId());
        print.setCompanyName(cost.getCompanyName());
        print.setCostNo(costNo);
        print.setCreateTime(cost.getCreateTime());
        print.setBankCompany(cost.getBankCompany());
        print.setBankCardUser(cost.getBankCardUser());
        print.setBankCard(cost.getBankCard());
        print.setCostReason(cost.getCostReason());
        print.setDic(cost.getDicDto());
        print.setAmount(cost.getAmountDto());
        print.setTxtAmount(MoneyUtil.getCnNum(cost.getAmount()));
        if (1 == cost.getCostForm()) {
            // 付款单
            print.setCostFormStr("付");
            print.setBankCompanyStr("收款单位");
            print.setBankCardUserStr("收款户名");
            print.setFeeTypeStr("付款方式");
            print.setBankCardStr("收款账户");
            print.setCostReasonStr("付款理由");
            print.setCostReason(cost.getTypeName() + "/" + cost.getKindName() + ";  " + cost.getCostRemark());
            // 差额单的付款理由需详细说明
            if (null != cost.getIsLend() && 2 == cost.getIsLend()) {
                print.setCostReason("【差额单（关联借支单：" + cost.getSupCostNo() + "）】 " + cost.getTypeName() + "  " + cost.getCostRemark());
            }
            print.setCostAmountStr("付款金额");
        }
        if (2 == cost.getCostForm()) {
            // 收款单
            print.setCostFormStr("收");
            print.setBankCompanyStr("付款单位");
            print.setBankCardUserStr("付款户名");
            print.setFeeTypeStr("收款方式");
            print.setBankCardStr("付款账户");
            print.setCostReasonStr("收款理由");
            print.setCostAmountStr("收款金额");
        }

        if (3 == cost.getCostForm() && 1 == cost.getIsLend()) {
            // 借支单
            print.setCostFormStr("借支");
            print.setBankCompanyStr("收款单位");
            print.setBankCardUserStr("收款户名");
            print.setFeeTypeStr("付款方式");
            print.setBankCardStr("收款账户");
            print.setCostReasonStr("付款理由");
            print.setCostAmountStr("付款金额");
        }

        if (3 == cost.getCostForm() && 2 == cost.getIsLend()) {
            // 借还单
            print.setCostFormStr("借还");
            print.setBankCompanyStr("付款单位");
            print.setBankCardUserStr("付款户名");
            print.setFeeTypeStr("收款方式");
            print.setBankCardStr("付款账户");
            print.setCostReasonStr("收款理由");
            print.setCostAmountStr("收款金额");
            print.setDic(cost.getPayDicDto());
            print.setAmount(cost.getPayPlanAmountDto());
            print.setTxtAmount(MoneyUtil.getCnNum(cost.getPayPlanAmount()));
        }
        return print;
    }

    PostApplyReq buildPostApplyReq(CostDomain cost) {
        CostCompanyDomain companyByCompanyNo = costCompanyDao.selectByNo(cost.getCompanyNo());


        PostApplyReq req = new PostApplyReq();
        // 公司主体(CompanyMainName)不可为空；
        req.setCompanyMainName(cost.getCompanyName());
        req.setTitle(cost.getCostNo());

        // 币种编号(UnitCode)不可为空；
        req.setUnitCode(cost.getDic());
        // 币种名称(UnitName)不可为空；
        req.setUnitName(cost.getDic());

        // 交易对象(TradeObjectID/TradeObjectName)不可为空；
        req.setTradeObjectID(companyByCompanyNo.getValue() + "");
        req.setTradeObjectName(cost.getBankCompany() + "");

        // 交易对象账户(BankNameto/BankCardto/BankCardUserto)资料不完整；
        req.setBankCardto(cost.getBankCard());
        req.setBankCardUserto(cost.getBankCardUser());
        req.setBankNameto(cost.getBankName());

        // newOtherPurchase申请来源(SourceCode/SourceTypeCode)不可为空；
        req.setSourceCode("newCost");
        req.setSourceTypeCode("newCostType");
        // 账期 0 PayDay
        req.setPayDay(0);
        // 预计付款时间 ExpectPayTime
        req.setExpectPayTime(new Date());

        // 申请人(UserAcctID/UserAcctName)不存在；
        req.setUserAcctID("" + cost.getCreateUserid());
        req.setUserAcctName("" + cost.getCreateUsername());


        // 设置 detailKey
        String detailKey = cost.getDetailKey();
        if (null == detailKey || !detailKey.contains("F")) {
            detailKey = cost.getCostNo();
        }
        // 回调必要的key参数（DetailKey）不能为空；
        req.setDetailKey(detailKey);
        req.setTypeName(cost.getTypeName());
        return req;
    }
}
