package com.blt.other.cost.service.impl;

import com.bailuntec.cost.api.dto.CostPlanTempDto;
import com.blt.other.commons.utils.CurUtils;
import com.blt.other.commons.utils.PathUtil;
import com.blt.other.cost.dao.CostPlanDao;
import com.blt.other.cost.dao.CostPlanTempDao;
import com.blt.other.cost.dto.CostPlanTempApiDto;
import com.blt.other.cost.service.CostPlanTempService;
import com.blt.other.cost.service.CostTypeKindService;
import com.blt.other.cost.utils.CostFileUtil;
import com.blt.other.database.model.CostPlanDomain;
import com.blt.other.database.model.CostPlanTempDomain;
import com.blt.other.database.model.CostTypeKindDomain;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.stream.Collectors;

@Service
public class CostPlanTempServiceImpl implements CostPlanTempService {

    @Autowired
    private CostPlanTempDao costPlanTempDao;
    @Autowired
    CostTypeKindService costTypeKindService;
    @Autowired
    private CostPlanDao costPlanDao;

    @Value("${url.api.getExchangeRateApi}")
    private String getExchangeRateApi;

    @Override
    public String createNo() {

        Random random = new Random();
        SimpleDateFormat sdf = new SimpleDateFormat("yyMMddhhmmss");
        String no = "CPT" + sdf.format(new Date()) + random.nextInt(9);
        CostPlanTempDomain costPlanTempDomain = costPlanTempDao.selectByNo(no);
        while (null != costPlanTempDomain && null != costPlanTempDomain.getCostPlanNo()) {
            no = "CPT" + sdf.format(new Date()) + random.nextInt(9);
            costPlanTempDomain = costPlanTempDao.selectByNo(no);
        }
        return no;
    }

    /**
     * 检查同一个付款计划中是否存在大类相同、币种不同的情况
     *
     * @param costPlanTempDomain 费用计划
     */
    private void checkDic(CostPlanTempDomain costPlanTempDomain) {
        List<CostPlanTempDomain> checkDict = costPlanTempDao.selectByPlanNoAndTypeNo(costPlanTempDomain.getCostPlanNo(),
                costPlanTempDomain.getTypeNo());
        if (checkDict.stream().collect(Collectors.groupingBy(CostPlanTempDomain::getTypeNo)).keySet().size() > 1) {
            throw new RuntimeException("已存在相同大类，请保持币种一致！");
        }
    }


    @Override
    public CostPlanTempApiDto.AddItemResp doSave(CostPlanTempApiDto.AddItemReq req) {

        CostPlanTempDomain costPlanTempDomain = new CostPlanTempDomain();
        BeanUtils.copyProperties(req, costPlanTempDomain);
        costPlanTempDomain.setTempNo(this.createNo());
        this.checkDic(costPlanTempDomain);

        /*附件字段*/
        if (req.getFileSelect() != null) {
            // 文件上传的路径
            String filePath = PathUtil.getBasePath() + PathUtil.getPath("cost/" + costPlanTempDomain.getTempNo() + "/");
            // 调用工具类执行保存，并返回 path
            String path = CostFileUtil.upload(req.getFileSelect(), filePath);
            costPlanTempDomain.setFilePath(path);
        }
        //新增付款子项目
        costPlanTempDao.insert(costPlanTempDomain);

        // 添加付款子项目成功，计划单改变总额
        CostPlanDomain costPlanDomain = costPlanDao.selectByNo(costPlanTempDomain.getCostPlanNo());
        if (costPlanDomain != null) {
            costPlanDomain.setPlanAmount(costPlanDomain.getPlanAmount().add(costPlanTempDomain.getAmount()));
            costPlanDao.update(costPlanDomain);
        }

        CostPlanTempApiDto.AddItemResp resp = new CostPlanTempApiDto.AddItemResp();
        resp.setMsg("success");
        resp.setPlaAmount(CurUtils.getCur(costPlanTempDomain.getDic(), "CNY", getExchangeRateApi));
        resp.setTmp(costPlanTempDomain);
        resp.setSuccess(true);
        return resp;
    }


    @Override
    public CostPlanTempApiDto.GetCostTempByPlanNoResp getTempListByPlanNo(String costPlanNo) {
        List<CostPlanTempDomain> list = costPlanTempDao.selectListByPlanNo(costPlanNo);

        List<CostPlanTempDto> temps = list.stream().map(CostPlanTempDomain::castToDto).collect(Collectors.toList());

        BigDecimal planAmount = temps.stream()
                .map(tempDto -> {
                    BigDecimal cur = CurUtils.getCur(tempDto.getDic(), "CNY", getExchangeRateApi);
                    return tempDto.getAmount().multiply(cur);
                })
                .reduce(BigDecimal.ZERO, BigDecimal::add);

        CostPlanTempApiDto.GetCostTempByPlanNoResp resp = new CostPlanTempApiDto.GetCostTempByPlanNoResp();
        resp.setPlanAmount(planAmount);
        resp.setSuccess(true);
        resp.setTemps(temps);
        return resp;
    }

    @Transactional
    @Override
    public BigDecimal deleteItemByPlanNoAndTempNo(String costPlanNo, String tempNo) {

        CostPlanTempDomain costPlanTempDomain = costPlanTempDao.selectByPlanNoAndTempNo(costPlanNo, tempNo);
        costPlanTempDao.deletedByPlanNoAndTempNo(costPlanNo, tempNo);

        // 删除付款子项目成功，计划单改变总额
        CostPlanDomain costPlanDomain = costPlanDao.selectByNo(costPlanNo);
        if (costPlanDomain != null) {

            costPlanDomain.setPlanAmount(costPlanDomain.getPlanAmount().subtract(costPlanTempDomain.getAmount()));
            costPlanDao.update(costPlanDomain);
            return costPlanDomain.getPlanAmount();
        }
        return BigDecimal.ZERO;
    }

    @Override
    @NonNull
    public CostPlanTempDomain getByTempNo(@NonNull String tempNo) {
        CostPlanTempDomain costPlanTempDomain = costPlanTempDao.selectByNo(tempNo);
        if (costPlanTempDomain == null) {
            throw new RuntimeException("not found");
        }
        return costPlanTempDomain;
    }


    @Transactional(rollbackFor = Exception.class)
    @Override
    public CostPlanTempDomain resetItem(CostPlanTempApiDto.ResetItemReq req) {

        CostTypeKindDomain kindByKindNo = costTypeKindService.getKindByKindNo(req.getKindNo());
        req.setTypeNo(kindByKindNo.getTypeNo());

        CostPlanTempDomain costPlanTempDomain = new CostPlanTempDomain();
        com.bailuntec.common.BeanUtils.copyProperties(req, costPlanTempDomain);

        if (req.getDelecteFile()) {
            costPlanTempDao.deletedFilePath(req.getTempNo());
        }

        costPlanTempDao.update(costPlanTempDomain);

        this.chalkResetDec(costPlanTempDomain.getCostPlanNo());

        if (req.getFileSelect2() != null) {
            String filePath = PathUtil.getBasePath() + PathUtil.getPath("cost/" + costPlanTempDomain.getTempNo() + "/");
            String path = CostFileUtil.upload(req.getFileSelect2(), filePath);
            costPlanTempDomain.setFilePath(path);
        }


        // 同步更新费用计划金额
        CostPlanDomain plan = costPlanDao.selectByNo(req.getCostPlanNo());
        if (plan != null) {
            List<CostPlanTempDomain> tempListByPlanNo = costPlanTempDao.selectListByPlanNo(req.getCostPlanNo());
            BigDecimal newAmount = tempListByPlanNo.stream().map(CostPlanTempDomain::getAmount).reduce(BigDecimal.ZERO, BigDecimal::add);

            plan.setPlanAmount(newAmount);
            if (1 == plan.getCostForm() && (null != plan.getIsLend() && 2 == plan.getIsLend())) {
                plan.setPayPlanAmount(newAmount.divide(plan.getPayCur(), 4));
            }
            costPlanDao.update(plan);
        }
        return costPlanTempDomain;
    }


    /**
     * 修改时判断费用单计划是否存在不相同货币的付款小类
     *
     * @param costPlanNo 费用单编号
     */
    private void chalkResetDec(String costPlanNo) {

        List<CostPlanTempDomain> costPlanTempDomainList = costPlanTempDao.selectListByPlanNo(costPlanNo);
        Map<String, List<CostPlanTempDomain>> map = costPlanTempDomainList.stream().collect(Collectors.groupingBy(CostPlanTempDomain::getDic));
        if (map.keySet().size() > 1) {
            throw new RuntimeException("请统一货币单位");
        }
    }
}
