package com.blt.other.module.cost.service.impl.costcheck;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.blt.other.common.exception.BizRuntimeException;
import com.blt.other.database.model.CostLogDomain;
import com.blt.other.module.auth.dao.OaDepartmentMapper;
import com.blt.other.module.auth.dao.OaUserMapper;
import com.blt.other.module.auth.model.OaDepartment;
import com.blt.other.module.auth.model.OaUser;
import com.blt.other.module.cost.model.CostDomain;
import com.blt.other.module.cost.model.CostTemplate;
import com.blt.other.module.sys.dao.DepartmentReviewerMapper;
import com.blt.other.module.sys.model.CostReviewer;
import com.blt.other.module.sys.model.DepartmentReviewer;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

/**
 * <p>
 *
 * </p>
 *
 * @author robbendev
 * @since 2020/10/30 2:57 下午
 */
@Slf4j
@Component
public class DepartmentCheckState extends CostState {

    @Resource
    HrCheckState hrCheckState;
    @Resource
    OaDepartmentMapper oaDepartmentMapper;
    @Resource
    OaUserMapper oaUserMapper;

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void handle() {
        CostDomain costDomain = costContext.costDomain;
        Integer currentUserId = costContext.currentUserId;
        CostTemplate costTemplate = costTemplateService.queryDetail(costDomain.getCostTemplateId());

        //check status
        if (!costDomain.getCostStatus().equals(CostDomain.STATUS_DEPARTMENT_CHECK)) {
            throw new BizRuntimeException("invalid status");
        }

        //如果不需要审核 直接通过
        OaUser oaUser = oaUserMapper.selectByOaUserId(costDomain.getCreateUserid());
        OaDepartment oaDepartment = oaDepartmentMapper.selectByDepartmentId(oaUser.getPrimaryDepartmentId());

        //是否通过部门
        if (!costTemplate.shouldDepartmentCheck(costDomain, oaDepartment.getDepartmentMinimumReviewAmount())) {

            costDomain.setCostStatus(CostDomain.STATUS_HR_CHECK);
            costDomain.setLastModifyDate(LocalDateTime.now());
            costDao.updateById(costDomain);

            costLogService.saveByManage(costDomain.getCostNo(), "部门审核自动通过", CostLogDomain.DEPARTMENT_AUTO_PASS);

            //流转状态
            nextState(hrCheckState);
            costSubscribe.subscribe(costContext);
            return;
        }


        //需要自动审核
        if (costTemplate.shouldDepartmentAutoCheck(costDomain)) {
            //自动审核通过
            if (this.autoCheck(costDomain)) {
                costDomain.setCostStatus(CostDomain.STATUS_HR_CHECK);
                costDomain.setLastModifyDate(LocalDateTime.now());
                costDao.updateById(costDomain);
                costLogService.saveByManage(costDomain.getCostNo(), "部门自动审核通过", CostLogDomain.DEPARTMENT_AUTO_PASS);

                //流转状态
                nextState(hrCheckState);
                costSubscribe.subscribe(costContext);
                return;
            }
        }


        //人工审核
        if (costContext.currentUserId != null) {
            //人工审核通过
            if (costCurrentReviewerService.canAudit(currentUserId, costDomain.getCostNo())) {
                costDomain.setCostStatus(CostDomain.STATUS_HR_CHECK);
                costDomain.setLastModifyDate(LocalDateTime.now());
                costDao.updateById(costDomain);
                costLogService.save(costDomain.getCostNo(), currentUserId, "部门审核通过", CostLogDomain.DEPARTMENT_MANUAL_PASS);

                nextState(hrCheckState);
                //发布到总线尝试下个环节的自动审核
                costSubscribe.subscribe(costContext);
            }
            //人工审核没权限
            else {
                throw new BizRuntimeException("current user no authority");
            }
        }
    }


    @Override
    public void refuse(String rejectReason) {
        log.info("部门审核拒绝 currentUserId:{},costNo:{}", costContext.currentUserId, costContext.costDomain.getCostNo());
        CostDomain costDomain = costContext.costDomain;
        Integer currentUserId = costContext.currentUserId;

        //check status
        if (!costDomain.getCostStatus().equals(CostDomain.STATUS_DEPARTMENT_CHECK)) {
            throw new BizRuntimeException("invalid status");
        }

        //人工审核没权限
        if (!costCurrentReviewerService.canAudit(currentUserId, costDomain.getCostNo())) {
            throw new BizRuntimeException("current user no authority");
        }

        costLogService.save(costDomain.getCostNo(), currentUserId, "部门审核拒绝,理由:" + rejectReason, CostLogDomain.TYPE_UPDATE);
        costContext.costService.reject(costDomain.getCostNo());
    }

    @Resource
    DepartmentReviewerMapper departmentReviewerMapper;

    @Override
    public void updateCurrentReviewer() {

        CostDomain costDomain = costContext.costDomain;
        OaUser costUser = oaUserMapper.selectByOaUserId(costDomain.getCreateUserid());

        DepartmentReviewer departmentReviewer = departmentReviewerMapper.selectByPrimaryDepartmentId(costUser.getPrimaryDepartmentId());

        List<CostReviewer> costReviewerList = costReviewerMapper.selectList(new LambdaQueryWrapper<CostReviewer>()
                .eq(CostReviewer::getReferId, departmentReviewer.getId())
                .eq(CostReviewer::getType, CostReviewer.departmentReviewer));

        String departmentReviewerNames = costReviewerList.stream().map(CostReviewer::getReviewerUserName).collect(Collectors.joining(","));
        log.info("更新当前审核人为部门审核人:{}", departmentReviewerNames);

        costCurrentReviewerService.updateByCostNoAndReviewer(costDomain.getCostNo(), costReviewerList);
        costLogService.save(costDomain.getCostNo(), costContext.currentUserId, "一级部门审核人:" + departmentReviewerNames);
    }
}
