﻿using Bailun.DC.Models.Component.DTO;
using Bailun.DC.Models.Component.Entity;
using Dapper;
using MySql.Data.MySqlClient;
using System;
using System.Linq;

namespace Bailun.DC.Services.Component
{
    /// <summary>
    /// 菜单组件
    /// </summary>
    public class MenuService : BaseService
    {
        /// <summary>
        /// 获取字典组件
        /// </summary>
        /// <param name="id">ID</param>
        /// <param name="code">编码</param>
        /// <returns></returns>
        public MenuDTO Get(int? id = null)
        {
            MenuDTO dto = default(MenuDTO);

            if (id > 0)
            {
                dc_menu entity = default(dc_menu);
                using (var db = DB)
                {
                    DynamicParameters sqlparam = new DynamicParameters();
                    sqlparam.Add("id", id);
                    entity = db.QueryFirstOrDefault<dc_menu>($"select t1.*,t2.name as parent_name from dc_menu t1 left join dc_menu t2 on t1.parentid = t2.id where t1.id = @id", sqlparam);
                }
                if (entity != null)
                {
                    dto = new MenuDTO
                    {
                        Id = entity.id,
                        Icon = entity.icon,
                        IsDevelop = entity.is_develop,
                        IsSys = entity.is_sys,
                        Name = entity.name,
                        ParentId = entity.parentid,
                        ParentName = entity.parent_name,
                        Path = entity.path,
                        Sort = entity.sort
                    };
                }
            }

            return dto;
        }

        /// <summary>
        /// 保存菜单组件
        /// </summary>
        /// <param name="dto">菜单组件对象</param>
        /// <returns>保存结果</returns>
        public ResultDTO Save(MenuDTO dto)
        {
            ResultDTO result = this.BeforeSave(dto);

            if (result.Result)
            {
                dc_menu entity = new dc_menu()
                {
                    id = dto.Id,
                    parentid = dto.ParentId ?? 0,
                    name = dto.Name ?? string.Empty,
                    icon = dto.Icon ?? string.Empty,
                    is_develop = dto.IsDevelop ?? false,
                    is_sys = dto.IsSys ?? false,
                    path = dto.Path ?? string.Empty,
                    sort = dto.Sort ?? 0,
                    delstatus = false
                };
                using (var db = DB)
                {
                    //开启事务
                    MySqlTransaction transaction = db.BeginTransaction();
                    try
                    {
                        int? resultCount = entity.id > 0 ? db.Update(entity) : db.Insert(entity);
                        if (resultCount == 0)
                        {
                            throw new Exception("保存失败");
                        }
                        else
                        {
                            //提交事务
                            transaction.Commit();
                            result.Message = "保存成功";
                        }
                    }
                    catch (Exception ex)
                    {
                        //回滚事务
                        transaction.Rollback();
                        result.Message = ex.Message;
                        result.Result = false;
                    }
                }
            }

            return result;
        }

        /// <summary>
        /// 保存报表组件
        /// </summary>
        /// <param name="id">ID</param>
        /// <param name="is_delete">是否删除</param>
        /// <returns>保存结果</returns>
        public ResultDTO Save(int[] id, bool is_delete)
        {
            ResultDTO result = this.BeforeSave(id);

            if (result.Result)
            {
                string sql = "update dc_menu set delstatus = @delstatus where id in @id";
                using (var db = DB)
                {
                    //开启事务
                    MySqlTransaction transaction = db.BeginTransaction();
                    try
                    {
                        int resultCount = db.Execute(sql, new { id = id, delstatus = is_delete });
                        if (resultCount == 0)
                        {
                            throw new Exception("保存失败");
                        }
                        else
                        {
                            //提交事务
                            transaction.Commit();
                            result.Message = "保存成功";
                        }
                    }
                    catch (Exception ex)
                    {
                        //回滚事务
                        transaction.Rollback();
                        result.Message = ex.Message;
                        result.Result = false;
                    }
                }
            }

            return result;
        }

        /// <summary>
        /// 保存报表组件
        /// </summary>
        /// <param name="dto">报表组件对象</param>
        /// <returns>保存结果</returns>
        private ResultDTO BeforeSave(MenuDTO dto)
        {
            ResultDTO result = new ResultDTO();

            if (dto == null)
            {
                result.Message = "参数有误";
            }
            else if (string.IsNullOrWhiteSpace(dto.Name))
            {
                result.Message = "名称不能为空";
            }
            else
            {
                result.Result = true;
            }

            return result;
        }

        /// <summary>
        /// 保存报表组件
        /// </summary>
        /// <param name="id">ID</param>
        /// <returns>保存结果</returns>
        private ResultDTO BeforeSave(int[] id)
        {
            ResultDTO result = new ResultDTO();
            if (id == null || id.Length == 0 || id.Contains(0))
            {
                result.Message = "ID不能为空";
            }
            else
            {
                result.Result = true;
            }
            return result;
        }

    }
}
