﻿using Bailun.DC.Models.Component.DTO;
using Bailun.DC.Models.Component.Entity;
using Bailun.DC.Models.Component.Enum;
using Dapper;
using MySql.Data.MySqlClient;
using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;

namespace Bailun.DC.Services.Component
{
    /// <summary>
    /// 字典组件
    /// </summary>
    public class DictionaryService : BaseService
    {
        /// <summary>
        /// 获取字典组件
        /// </summary>
        /// <param name="sqlparam">参数</param>
        /// <returns></returns>
        public IEnumerable<dynamic> GetList(DynamicParameters sqlparam = null)
        {
            IEnumerable<dynamic> list = default(IEnumerable<dynamic>);
            dc_component_dictionary entity = default(dc_component_dictionary);
            if (sqlparam != null && sqlparam.ParameterNames.Contains("code"))
            {
                using (var db = DB)
                {
                    entity = db.QueryFirstOrDefault<dc_component_dictionary>("select * from dc_component_dictionary where is_delete = 0 and code = @code", sqlparam);
                }
                if (entity != null)
                {
                    switch (entity.type)
                    {
                        case DictionaryTypeEnum.SQL:
                            MySqlConnection db = default(MySqlConnection);
                            switch (entity.data_db)
                            {
                                case DBEnum.DataWareHouse:
                                    db = DW_DB;
                                    break;
                                case DBEnum.Pro:
                                    db = PRO_DB;
                                    break;
                                case DBEnum.Fund:
                                    db = FUND_DB;
                                    break;
                                default:
                                    db = DB;
                                    break;
                            }
                            //默认参数
                            var paramIndex = entity.value.IndexOf("@");
                            if (paramIndex >= 0)
                            {
                                foreach (Match item in Regex.Matches(entity.value.Substring(paramIndex), @"@\w+[^\w]*"))
                                {
                                    string paramName = item.Value.Substring(1, Regex.Match(item.Value.Substring(1), @"[^\w]*$").Index);
                                    if (!sqlparam.ParameterNames.Contains(paramName))
                                    {
                                        if (entity.value.Substring(0, paramIndex + item.Index).TrimEnd().EndsWith("in", StringComparison.OrdinalIgnoreCase))
                                        {
                                            sqlparam.Add(paramName, new string[] { "" });
                                        }
                                        else
                                        {
                                            sqlparam.Add(paramName, null);
                                        }
                                    }
                                }
                            }
                            //查询
                            using (db)
                            {
                                list = db.Query(entity.value, sqlparam);
                            }
                            break;
                        case DictionaryTypeEnum.CONFIG:
                            list = DeserializeCollection<dynamic>(entity.value);
                            break;
                    }
                }
            }

            return list;
        }

        /// <summary>
        /// 获取字典组件
        /// </summary>
        /// <param name="id">ID</param>
        /// <param name="code">编码</param>
        /// <returns></returns>
        public DictionaryDTO Get(int? id = null, string code = null)
        {
            DictionaryDTO dto = default(DictionaryDTO);

            if (id > 0 || !string.IsNullOrWhiteSpace(code))
            {
                dc_component_dictionary entity = default(dc_component_dictionary);
                using (var db = DB)
                {
                    DynamicParameters sqlparam = new DynamicParameters();
                    string where = string.Empty;
                    if (id > 0)
                    {
                        sqlparam.Add("id", id);
                        where = "id = @id";
                    }
                    else
                    {
                        sqlparam.Add("code", code);
                        where = "code = @code";
                    }
                    entity = db.QueryFirstOrDefault<dc_component_dictionary>($"select * from dc_component_dictionary where {where}", sqlparam);
                }
                if (entity != null)
                {
                    dto = new DictionaryDTO
                    {
                        Id = entity.id,
                        Code = entity.code,
                        Name = entity.name,
                        Category = entity.category,
                        DataDb = entity.data_db,
                        Method = entity.method,
                        Type = entity.type,
                    };
                    switch (dto.Type)
                    {
                        case DictionaryTypeEnum.SQL:
                            dto.Value = entity.value;
                            break;
                        case DictionaryTypeEnum.CONFIG:
                            dto.listOption = DeserializeCollection<OptionDTO>(entity.value);
                            break;
                    }
                }
            }

            return dto;
        }

        /// <summary>
        /// 保存字典组件
        /// </summary>
        /// <param name="dto">字典组件对象</param>
        /// <returns>保存结果</returns>
        public ResultDTO Save(DictionaryDTO dto)
        {
            ResultDTO result = this.BeforeSave(dto);

            if (result.Result)
            {
                dc_component_dictionary entity = new dc_component_dictionary()
                {
                    id = dto.Id,
                    code = dto.Code,
                    name = dto.Name,
                    category = dto.Category,
                    type = dto.Type,
                    method = dto.Method,
                    value = dto.Type == DictionaryTypeEnum.CONFIG ? JsonConvert.SerializeObject(dto.listOption) : dto.Value,
                    data_db = dto.DataDb,
                    gmt_create = DateTime.Now,
                    gmt_modified = DateTime.Now,
                };
                using (var db = DB)
                {
                    //开启事务
                    MySqlTransaction transaction = db.BeginTransaction();
                    try
                    {
                        int? resultCount = entity.id > 0 ? db.Update(entity) : db.Insert(entity);
                        if (resultCount == 0)
                        {
                            throw new Exception("保存失败");
                        }
                        else
                        {
                            //提交事务
                            transaction.Commit();
                            result.Message = "保存成功";
                        }
                    }
                    catch (Exception ex)
                    {
                        //回滚事务
                        transaction.Rollback();
                        result.Message = ex.Message;
                        result.Result = false;
                    }
                }
            }

            return result;
        }

        /// <summary>
        /// 保存报表组件
        /// </summary>
        /// <param name="id">ID</param>
        /// <param name="is_delete">是否删除</param>
        /// <returns>保存结果</returns>
        public ResultDTO Save(int[] id, bool is_delete)
        {
            ResultDTO result = this.BeforeSave(id);

            if (result.Result)
            {
                string sql = "update dc_component_dictionary set is_delete = @is_delete, gmt_modified = @gmt_modified where id in @id";
                using (var db = DB)
                {
                    //开启事务
                    MySqlTransaction transaction = db.BeginTransaction();
                    try
                    {
                        int resultCount = db.Execute(sql, new { id = id, is_delete = is_delete, gmt_modified = DateTime.Now });
                        if (resultCount == 0)
                        {
                            throw new Exception("保存失败");
                        }
                        else
                        {
                            //提交事务
                            transaction.Commit();
                            result.Message = "保存成功";
                        }
                    }
                    catch (Exception ex)
                    {
                        //回滚事务
                        transaction.Rollback();
                        result.Message = ex.Message;
                        result.Result = false;
                    }
                }
            }

            return result;
        }

        /// <summary>
        /// 保存报表组件
        /// </summary>
        /// <param name="dto">报表组件对象</param>
        /// <returns>保存结果</returns>
        private ResultDTO BeforeSave(DictionaryDTO dto)
        {
            ResultDTO result = new ResultDTO();

            if (dto == null)
            {
                result.Message = "参数有误";
            }
            else if (string.IsNullOrWhiteSpace(dto.Code))
            {
                result.Message = "编码不能为空";
            }
            else if (string.IsNullOrWhiteSpace(dto.Name))
            {
                result.Message = "名称不能为空";
            }
            else if (!dto.Category.HasValue)
            {
                result.Message = "分类不能为空";
            }
            else if (!dto.Type.HasValue)
            {
                result.Message = "类型不能为空";
            }
            else if (dto.Type == DictionaryTypeEnum.SQL && string.IsNullOrWhiteSpace(dto.Value))
            {
                result.Message = "SQL不能为空";
            }
            else if (dto.Type == DictionaryTypeEnum.CONFIG && (dto.listOption == null || dto.listOption.Count() == 0))
            {
                result.Message = "选项不能为空";
            }
            else if (DB.QueryFirstOrDefault<int>($"select count(id) from dc_component_dictionary where code = @code {(dto.Id > 0 ? "and id <> @id" : string.Empty)}", new { id = dto.Id, code = dto.Code }) > 0)
            {
                result.Message = "编码重复";
            }
            else
            {
                result.Result = true;
            }

            return result;
        }

        /// <summary>
        /// 保存报表组件
        /// </summary>
        /// <param name="id">ID</param>
        /// <returns>保存结果</returns>
        private ResultDTO BeforeSave(int[] id)
        {
            ResultDTO result = new ResultDTO();
            if (id == null || id.Length == 0 || id.Contains(0))
            {
                result.Message = "ID不能为空";
            }
            else
            {
                result.Result = true;
            }
            return result;
        }

    }
}
