﻿using Bailun.DC.Models;
using Bailun.DC.Models.Common;
using Bailun.DC.Models.Common.Page;
using Bailun.DC.Models.Dtos.Finance;
using Bailun.DC.Models.WebApiModels;
using Bailun.DC.Services.WebApiService;
using Bailun.DC.WebApi.Models;
using Bailun.ServiceFabric.Authorize;
using Bailun.ServiceFabric.Core.Extension;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Mvc;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Reflection;
using System.Text;

namespace Bailun.DC.WebApi.Controllers
{
    /// <summary>
    /// 财务报表
    /// </summary>
    [Route("api/Finance")]
    [ApiController]
    public class FinanceController : BaseController
    {
        private readonly IHostingEnvironment _hostingEnvironment;

        public FinanceController(IHostingEnvironment hostingEnvironment)
        {
            _hostingEnvironment = hostingEnvironment;
        }

        /// <summary>
        /// 获取付现流水报表
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        [HttpPost("getCashFlowStatement")]
        public CommonApiResponseDto<FinanceDto> GetCashFlowStatement(GetCashFlowStatementInput input)
            => new FinanceService().GetCashFlowStatement(input);


        #region 付现流水明细
        /// <summary>
        /// 分页查询付现流水明细
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        [HttpPost("getFinanceFeeDetailsPage"), BailunAuthentication(LoginMode.Enforce)]
        public CommonApiResponseDto<FinanceFeeDetailsPageOutputDto> GetFinanceFeeDetailsPage([FromBody] FinanceFeeDetailsPageInputDto input)
        {
            return new FinanceService().GetFinanceFeeDetailsPage(input);
        }

        /// <summary>
        /// 导出明细
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        [HttpPost("exportFinanceFeeDetails"), BailunAuthentication(LoginMode.Enforce)]
        public ActionResult ExportFinanceFeeDetails([FromBody] FinanceFeeDetailsPageInputDto input)
        {
            input.IsPages = false;
            var dataList = new FinanceService().GetFinanceFeeDetailsPage(input).Data.Items;
            var colNames = new List<string> {"费用单号","公司名称","费用类型", "费用总金额", "人民币总金额","付款理由","大类标题","小类标题","收款/付款人",
                                            "收款单位","会计科目","财务审核时间" ,"付款时间","最后更新时间","部门名称","创建人名称","创建时间"};
            var list = new List<string>();
            foreach (var item in dataList)
            {
                list.Add(item.cost_no + "|" + item.company_name + "|" + item.type_name + "|" + item.amount + "|" + item.amount_rmb + "|" + item.cost_reason + "|" +
                    item.type_name + "|" + item.kind_name + "|" + item.bank_card_user + "|" + item.bank_company + "|" + item.accounting_subject_name + "|" +
                    item.audit_time + "|" + (item.pay_time.HasValue ? item.pay_time.Value.ToString("yyyy-MM-dd HH:mm:ss") : string.Empty) + "|" +
                    (item.last_modify_date.HasValue ? item.last_modify_date.Value.ToString("yyyy-MM-dd HH:mm:ss") : string.Empty) + "|" +
                    item.department_name + "|" + item.create_username + "|" + (item.create_time.HasValue ? item.create_time.Value.ToString("yyyy-MM-dd HH:mm:ss") : string.Empty));
            }
            var guid = Guid.NewGuid().ToString();
            var filepath = _hostingEnvironment.WebRootPath + "\\Files\\Report\\" + DateTime.Now.ToString("yyyy-MM-dd") + "\\";
            ToCSV(list, colNames, guid, filepath);
            var ms = new System.IO.MemoryStream();
            using (var f = new System.IO.FileStream(filepath + guid + ".csv", System.IO.FileMode.Open))
            {
                f.CopyTo(ms);
            }
            ms.Position = 0;
            return File(ms, "text/csv", "付现流水明细.csv");

        }
        #endregion

        /// <summary>
        /// 获取付现流水报表明细
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        [HttpPost("getFinanceDetails")]
        public CommonApiResponseDto<List<FinanceDetailsDto>> GetFinanceDetails(GetCashFlowStatementInput input)
            => new FinanceService().GetFinanceDetails(input);
        /// <summary>
        /// 获取管理成本
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        [HttpPost("getManagementCost")]
        public CommonApiResponseDto<FinanceDto> GetManagementCost(GetCashFlowStatementInput input)
            => new FinanceService().GetManagementCost(input);

        #region 月销售利润
        /// <summary>
        /// 获取月销售利润
        /// </summary>
        /// <param name="month"></param>
        /// <returns></returns>
        [HttpPost("getListMonthSaleProfit")]
        public CommonApiResponseDto<MonthSalesProfitResponse> GetListMonthSaleProfit(string month)
        {
            var result = new CommonApiResponseDto<MonthSalesProfitResponse> { Data = new MonthSalesProfitResponse() };
            if (string.IsNullOrWhiteSpace(month))
            {
                result.Message = "月份不能空";
                result.IsSuccess = false;
                return result;
            }
            var newList = new List<MonthSalesProfitDto>();
            var dataList = new FinanceService().ListMonthSaleProfitNew(month, null);
            var noEditList = dataList.Where(x => x.isedit == 0).ToList();
            result.Data.Col.Add(new MonthSalesProfitColumnsModel { Title = "项目", Key = "Item" });
            foreach (var item in noEditList)
            {
                result.Data.Col.Add(new MonthSalesProfitColumnsModel
                {
                    Title = $"{item.month}月份",
                    Key = Guid.NewGuid().ToString("N"),
                    Children = new List<MonthSalesProfitColumnsChildren>
                    {
                        new MonthSalesProfitColumnsChildren{ Key = item.month + "-0", Title = "月份基础数据" },
                        new MonthSalesProfitColumnsChildren{ Key = item.month + "-1", Title = $"月份调整数据" },
                        new MonthSalesProfitColumnsChildren{ Key = item.month + "-2", Title = $"月份合计数据" }
                    }
                });
                newList.Add(item);
                MonthSaleProfitSupplement(newList, dataList, item);
            }
            result.Data.DataJson = GetMonthSalesProfitDataJson(newList, out List<string> excelList);
            return result;
        }

        /// <summary>
        /// 生成json
        /// </summary>
        /// <param name="dataList"></param>
        /// <param name="excelList"></param>
        /// <returns></returns>
        private string GetMonthSalesProfitDataJson(List<MonthSalesProfitDto> dataList, out List<string> excelList)
        {
            excelList = new List<string>();
            if (dataList.Count <= 0) return "[]";
            var months = dataList.Where(x => x.isedit == 0).GroupBy(x => x.month).ToList();
            StringBuilder jsonStr = new StringBuilder();
            string excelItemStr = string.Empty;
            jsonStr.Append("[");
            List<KeyDisplayNameModel> displayNameList = new List<KeyDisplayNameModel>();
            PropertyInfo[] properties = dataList.First().GetType().GetProperties(BindingFlags.Instance | BindingFlags.Public);
            foreach (var item in properties)
            {
                var displayName = item.GetCustomAttribute<ParentNameAttribute>()?.DisplayName;
                var parentName = item.GetCustomAttribute<ParentNameAttribute>()?.ParentName;
                if (string.IsNullOrWhiteSpace(displayName)) continue;
                displayNameList.Add(new KeyDisplayNameModel { DisplayName = displayName, KeyName = item.Name, ParentName = parentName });
            }
            var firstList = displayNameList.Where(x => x.KeyName == x.ParentName.ToString()).Distinct().ToList();
            var str = string.Empty;
            foreach (var item in displayNameList)
            {
                if (item.KeyName == "id" || item.KeyName == "month") continue;
                var isFirst = firstList.Count(x => x.ParentName.ToString() == item.KeyName) > 0?1:0;
                jsonStr.Append("{\"ItemName\":\"" + item.KeyName + "\",\"Item\":\"" + item.DisplayName + "\",\"IsFirst\":"+ isFirst + ",\"FirstId\":\""+ (int)item.ParentName+ "\",");
                excelItemStr = item.DisplayName + "|";
                foreach (var subItem in dataList)
                {
                    PropertyInfo[] propertiesValue = subItem.GetType().GetProperties(BindingFlags.Instance | BindingFlags.Public);
                    var amount_sales = propertiesValue.Where(x => x.CustomAttributes.GetType() == typeof(ParentNameAttribute) && x.GetCustomAttribute<ParentNameAttribute>().ParentName == ParentNameEnum.amount_sales).ToList();
                    foreach (var proItem in propertiesValue)
                    {
                        if (proItem.Name != item.KeyName) continue;
                        object value = proItem.GetValue(subItem, null);
                        jsonStr.Append("\"" + subItem.month + "-" + subItem.isedit + "\": \"" + value + "\",");
                        excelItemStr += value + "|";
                    }
                }
                excelItemStr = excelItemStr.TrimEnd('|');
                str = jsonStr.ToString().TrimEnd(',');
                jsonStr.Clear();
                jsonStr.Append(str);
                jsonStr.Append("},");
                excelList.Add(excelItemStr);
            }
            str = jsonStr.ToString().TrimEnd(',');
            jsonStr.Clear();
            jsonStr.Append(str);
            jsonStr.Append("]");
            return jsonStr.ToString();
        }

        /// <summary>
        /// 补充月销售利润空数据及合并计算
        /// </summary>
        /// <param name="newList"></param>
        /// <param name="dataList"></param>
        /// <param name="item"></param>
        private void MonthSaleProfitSupplement(List<MonthSalesProfitDto> newList, List<MonthSalesProfitDto> dataList, MonthSalesProfitDto item)
        {
            var editModel = dataList.Where(x => x.month == item.month && x.isedit == 1).FirstOrDefault();
            if (editModel == null)
            {
                newList.Add(new MonthSalesProfitDto { month = item.month, isedit = 1 });
                newList.Add(new MonthSalesProfitDto { month = item.month, isedit = 2 });
                return;
            }
            newList.Add(editModel);
            newList.Add(new MonthSalesProfitDto
            {
                month = item.month,
                amount_sales = editModel.amount_sales + item.amount_sales,
                amount_sales_jingyou = editModel.amount_sales_jingyou + item.amount_sales_jingyou,
                amount_sales_dianzi = editModel.amount_sales_dianzi + item.amount_sales_dianzi,
                amount_sales_jiaju = editModel.amount_sales_jiaju + item.amount_sales_jiaju,
                amount_sales_meirongmj = editModel.amount_sales_meirongmj + item.amount_sales_meirongmj,
                amount_sales_fuzhuang = editModel.amount_sales_fuzhuang + item.amount_sales_fuzhuang,
                amount_sales_other = editModel.amount_sales_other + item.amount_sales_other,
                fee_refund = editModel.fee_refund + item.fee_refund,
                cost = editModel.cost + item.cost,
                cost_jingyou = editModel.cost_jingyou + item.cost_jingyou,
                cost_dianzi = editModel.cost_dianzi + item.cost_dianzi,
                cost_jiaju = editModel.cost_jiaju + item.cost_jiaju,
                cost_meirongmj = editModel.cost_meirongmj + item.cost_meirongmj,
                cost_fuzhuang = editModel.cost_fuzhuang + item.cost_fuzhuang,
                cost_other = editModel.cost_other + item.cost_other,
                fee_platform = editModel.fee_platform + item.fee_platform,
                //fee_platform_and_refund = editModel.fee_platform_and_refund + item.fee_platform_and_refund,
                fee_fba = editModel.fee_fba + item.fee_fba,
                fee_ad = editModel.fee_ad + item.fee_ad,
                fee_logistics_storage = editModel.fee_logistics_storage + item.fee_logistics_storage,
                fee_logistics_first = editModel.fee_logistics_first + item.fee_logistics_first,
                fee_logistics_direct = editModel.fee_logistics_direct + item.fee_logistics_direct,
                fee_logistics_tail = editModel.fee_logistics_tail + item.fee_logistics_tail,
                fee_storage = editModel.fee_storage + item.fee_storage,
                //fee_storage_incidentals = editModel.fee_storage_incidentals + item.fee_storage_incidentals,
                //fee_paycash_sales = editModel.fee_paycash_sales + item.fee_paycash_sales,
                fee_sales_count = editModel.fee_sales_count + item.fee_sales_count,
                profit_sales = editModel.profit_sales + item.profit_sales,
                rate_profit_sales = editModel.rate_profit_sales + item.rate_profit_sales,
                fee_finance = editModel.fee_finance + item.fee_finance,
                managercost_count = editModel.managercost_count + item.managercost_count,
                managecost_bl_gz = editModel.managecost_bl_gz + item.managecost_bl_gz,
                managecost_meijia = editModel.managecost_meijia + item.managecost_meijia,
                managercost_yangshan = editModel.managercost_yangshan + item.managercost_yangshan,
                //managercost_chengpincang = editModel.managercost_chengpincang + item.managercost_chengpincang,
                managercost_xinhuilan = editModel.managercost_xinhuilan + item.managercost_xinhuilan,
                managercost_dizhi = editModel.managercost_dizhi + item.managercost_dizhi,
                profit = editModel.profit + item.profit,
                rate_profit = editModel.rate_profit + item.rate_profit,
                incoming_other = editModel.incoming_other + item.incoming_other,
                incoming_refundtax = editModel.incoming_refundtax + item.incoming_refundtax,
                incoming_gogirl = editModel.incoming_gogirl + item.incoming_gogirl,
                incoming_non_operating = editModel.incoming_non_operating + item.incoming_non_operating,
                pay_non_operating = editModel.pay_non_operating + item.pay_non_operating,
                //impairment_assets = editModel.impairment_assets + item.impairment_assets,
                actual_profit = editModel.actual_profit + item.actual_profit,
                rate_profit_actual = editModel.rate_profit_actual + item.rate_profit_actual,

                managercost_gzcy = editModel.managercost_gzcy + item.managercost_gzcy,
                managercost_gzlk = editModel.managercost_gzlk + item.managercost_gzlk,
                managercost_ys = editModel.managercost_ys + item.managercost_ys,
                managercost_gzdz = editModel.managercost_gzdz + item.managercost_gzdz,
                fee_sales_taxes = editModel.fee_sales_taxes + item.fee_sales_taxes,
                fee_nostorage = editModel.fee_nostorage + item.fee_nostorage,
                fee_nologistics_direct = editModel.fee_nologistics_direct + item.fee_nologistics_direct,
                fee_nologistics_tail = editModel.fee_nologistics_tail + item.fee_nologistics_tail,
                fee_nologistics_first = editModel.fee_nologistics_first + item.fee_nologistics_first,
                fee_ohtre = editModel.fee_ohtre + item.fee_ohtre,
                fee_ebay = editModel.fee_ebay + item.fee_ebay,
                fee_nofba = editModel.fee_nofba + item.fee_nofba,

                isedit = 2
            });
        }

        /// <summary>
        /// 导出月销售利润
        /// </summary>
        /// <param name="month"></param>
        /// <returns></returns>
        [HttpPost("exportListMonthSaleProfit")]
        public ActionResult ExportListMonthSaleProfit(string month)
        {
            var result = new CommonApiResponseDto<bool>();
            if (string.IsNullOrWhiteSpace(month))
            {
                result.Message = "月份不能空";
                result.IsSuccess = false;
                return Ok(result);
            }
            var newList = new List<MonthSalesProfitDto>();
            var dataList = new FinanceService().ListMonthSaleProfitNew(month, null);
            var noEditList = dataList.Where(x => x.isedit == 0).ToList();
            var colNames = new List<string>();
            colNames.Add("项目");
            foreach (var item in noEditList)
            {
                colNames.Add($"{item.month}月份基础数据");
                colNames.Add($"{item.month}月份调整数据");
                colNames.Add($"{item.month}月份合计数据");
                newList.Add(item);
                MonthSaleProfitSupplement(newList, dataList, item);
            }
            GetMonthSalesProfitDataJson(newList, out List<string> excelList);
            var guid = Guid.NewGuid().ToString();
            var filename = "月销售利润报表" + DateTime.Now.ToString("yyyyMMddHHmmss");
            var filepath = _hostingEnvironment.WebRootPath + "\\Files\\Report\\" + DateTime.Now.ToString("yyyy-MM-dd") + "\\";
            ToCSV(excelList, colNames, guid, filepath);
            var ms = new System.IO.MemoryStream();
            using (var f = new System.IO.FileStream(filepath + guid + ".csv", System.IO.FileMode.Open))
            {
                f.CopyTo(ms);
            }
            ms.Position = 0;
            return File(ms, "text/csv", $"{filename}.csv");
        }

        /// <summary>
        /// 查询月销售平台统计数据
        /// </summary>
        /// <param name="month"></param>
        /// <param name="financecategory"></param>
        /// <returns></returns>
        [HttpGet("getPlatformTypeMonthlyStatistics")]
        public CommonApiResponseDto<List<PlatformTypeMonthlyStatisticsDto>> GetPlatformTypeMonthlyStatistics(string month, string financecategory,string isCost)
        {
            var result = new CommonApiResponseDto<List<PlatformTypeMonthlyStatisticsDto>> { Data = new List<PlatformTypeMonthlyStatisticsDto>()};
            result.Data = new FinanceService().GetMonthSalesProfiOrderSummary(month, financecategory,isCost);
            return result;
        }

        /// <summary>
        /// 分页查询月销售明细
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        [HttpPost("getMonthProfitOrderDetailPage")]
        public CommonApiResponseDto<MonthProfitOrderDetailPageOutputDto> GetMonthProfitOrderDetailPage([FromBody]MonthProfitOrderDetailPageInputDto input)
        {
            return new FinanceService().GetMonthProfitOrderDetailPage(input);
        }

        /// <summary>
        /// 导出月销售利润明细
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        [HttpPost("exportMonthProfitOrderDetail")]
        public ActionResult ExportMonthProfitOrderDetail([FromBody] MonthProfitOrderDetailPageInputDto input)
        {
            input.IsPages = false;
            var dataList = new FinanceService().GetMonthProfitOrderDetailPage(input).Data.Items;
            var colNames = new List<string> {"财务分类","平台类型","订单号", "SKU", "销售额（原币）", "销售额RMB", "汇率", "发货量","发货时间","创建时间"};
            var list = new List<string>();
            foreach (var item in dataList)
            {
                list.Add($"{item.financecategoryname}|{item.platform_type}|{item.origin_order_id}|{item.bailun_sku}|{item.amount_sales}|{item.amount_sales_rmb}|{item.seller_order_exchange_rate}|{item.quantity_shipped}|{item.shipping_time.ToString("yyyy-MM-dd HH:mm:ss")}|{item.createtime.ToString("yyyy-MM-dd HH:mm:ss")}");
            }
            var guid = Guid.NewGuid().ToString();
            var filepath = _hostingEnvironment.WebRootPath + "\\Files\\Report\\" + DateTime.Now.ToString("yyyy-MM-dd") + "\\";
            ToCSV(list, colNames, guid, filepath);
            var ms = new System.IO.MemoryStream();
            using (var f = new System.IO.FileStream(filepath + guid + ".csv", System.IO.FileMode.Open))
            {
                f.CopyTo(ms);
            }
            ms.Position = 0;
            return File(ms, "text/csv", "月销售利润表明细.csv");

        }
        /// <summary>
        /// 更新调整数据
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        [HttpPost("updateMonthSalseProfit")]
        public CommonApiResponseDto<string> UpdateMonthSalseProfit(UpdateMonthSalseProfitInput input)
        {
            try
            {
                var user = HttpContext.User;
                //var user = HttpContextHelper.Current?.User;
                var userId = user.GetUid();
                var userName = user.GetUserName();
                var output = new FinanceService().UpdateMonthSalseProfit(input, userId, userName);
                if (output)
                {
                    return new CommonApiResponseDto<string>();
                }
                return new CommonApiResponseDto<string>() { IsSuccess = false, Data = "操作失败！请与管理员联系~" };
            }
            catch (Exception e)
            {
                return new CommonApiResponseDto<string>() { IsSuccess = false, Data = e.Message };
            }
        }
        /// <summary>
        /// 月销售利润表获取管理成本明细
        /// </summary>
        /// <param name="month"></param>
        /// <param name="feeName"></param>
        /// <returns></returns>
        [HttpGet("getMonthProfitFeeManagementCostDetail")]
        public CommonApiResponseDto<List<FinanceDetailsDto>> GetMonthProfitFeeManagementCostDetail(string month,string feeName)
        => new FinanceService().GetMonthProfitFeeManagementCostDetail(month, feeName);
        /// <summary>
        /// 月销售利润平台费用明细
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        [HttpPost("getMonthProfitFeePlatformFeeDetail")]
        public CommonApiResponseDto<PageResult<List<MonthPlatformProfitDto>>> GetMonthProfitFeePlatformFeeDetail(GetMonthProfitPlatformDetailInput input)
        => new FinanceService().GetMonthProfitFeePlatformFeeDetail(input);
        /// <summary>
        /// 获取平台费和仓储费按照平台汇总明细
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        [HttpPost("getGetMonthProfitFeePlatformFeeSummary")]
        public CommonApiResponseDto<List<PlatformTypeMonthlyStatisticsDto>> GetGetMonthProfitFeePlatformFeeSummary(GetMonthProfitPlatformDetailInput input)
        {
            var result = new CommonApiResponseDto<List<PlatformTypeMonthlyStatisticsDto>>();
            result.Data = new FinanceService().GetGetMonthProfitFeePlatformFeeSummary(input);
            return result;
        }
        /// <summary>
        /// 导出月销售利润平台费用明细
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        [HttpPost("exportMonthProfitFeePlatformFeeDetail")]
        public ActionResult ExportMonthProfitFeePlatformFeeDetail(GetMonthProfitPlatformDetailInput input)
        {
           var dataList = new FinanceService().ExportMonthProfitFeePlatformFeeDetail(input);
            var colNames = new List<string> { "平台", "站点", "账单时间", "单号", "币种", "原币金额", "汇率", "RMB金额", "费用字段说明", "财务分类归集说明", "数据中心费用类型", "会计科目编码"
                , "项目编码" };
            var list = new List<string>();
            foreach (var item in dataList)
            {
                list.Add($"{item.PlatForm}|{item.WebSite}|{item.DataTime}|{item.Orderno}|{item.Currency}|{item.AmountVal}|{item.ExchangeRate}|{item.AmountValRmb}|{item.FeeType}|{item.FinanceCategory}|{item.DatacenterCol}|{item.Subjectcode}|{item.ProjectCode}");
            }
            var guid = Guid.NewGuid().ToString();
            var filepath = _hostingEnvironment.WebRootPath + "\\Files\\Report\\" + DateTime.Now.ToString("yyyy-MM-dd") + "\\";
            ToCSV(list, colNames, guid, filepath);
            var ms = new System.IO.MemoryStream();
            using (var f = new System.IO.FileStream(filepath + guid + ".csv", System.IO.FileMode.Open))
            {
                f.CopyTo(ms);
            }
            ms.Position = 0;
            return File(ms, "text/csv", $"{input.Month}_{input.FeeName}月销售利润表明细.csv");
        }
        /// <summary>
        /// 退款明细
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        [HttpPost("getMonthProfitFeeReFundDetail")]
        public CommonApiResponseDto<PageResult<List<MonthReFundDto>>> GetMonthProfitFeeReFundDetail(GetMonthProfitPlatformDetailInput input)
            => new FinanceService().GetMonthProfitFeeReFundDetail(input);
        /// <summary>
        /// 导出退款明细
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        [HttpPost("exportMonthProfitFeeReFundDetail")]
        public ActionResult ExportMonthProfitFeeReFundDetail(GetMonthProfitPlatformDetailInput input)
        {
            var dataList = new FinanceService().ExportMonthProfitFeeReFundDetail(input);
            var colNames = new List<string> { "平台", "订单号", "站点", "卖家账号", "退款金额", "退款USD", "退款时间" };
            var list = new List<string>();
            foreach (var item in dataList)
            {
                list.Add($"{item.PlatformType}|{item.OriginOrderId}|{item.WebSite}|{item.BailunAccount}|{item.RefundRmb}|{item.RefundUsd}|{item.RefundTime}");
            }
            var guid = Guid.NewGuid().ToString();
            var filepath = _hostingEnvironment.WebRootPath + "\\Files\\Report\\" + DateTime.Now.ToString("yyyy-MM-dd") + "\\";
            ToCSV(list, colNames, guid, filepath);
            var ms = new System.IO.MemoryStream();
            using (var f = new System.IO.FileStream(filepath + guid + ".csv", System.IO.FileMode.Open))
            {
                f.CopyTo(ms);
            }
            ms.Position = 0;
            return File(ms, "text/csv", $"{input.Month}_退款明细.csv");
        }

        /// <summary>
        /// 按平台汇总退款信息
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        [HttpPost("getMonthProfitPlatformPlatformSummary")]
        public CommonApiResponseDto<List<MonthReFundSummary>> GetMonthProfitPlatformPlatformSummary(GetMonthProfitPlatformDetailInput input)
            => new FinanceService().GetMonthProfitPlatformPlatformSummary(input);
        /// <summary>
        /// 销售费用合计》销售利润
        /// </summary>
        /// <param name="month"></param>
        /// <returns></returns>
        [HttpPost("getMonthSalesProfitDetail")]
        public object GetMonthSalesProfitDetail(MonthProfitOrderDetailPageInputDto input)
            => new FinanceService().GetMonthSalesProfitDetail(input);
        /// <summary>
        /// 导出销售费用合计》销售利润
        /// </summary>
        /// <param name="input"></param>
        /// <returns></returns>
        [HttpPost("exportMonthSalesProfitDetail")]
        public ActionResult ExportMonthSalesProfitDetail(MonthProfitOrderDetailPageInputDto input)
        {
           var dataList = new FinanceService().GetMonthSalesProfitDetail(input,false);
            var colNames = new List<string> { "财务分类", "平台类型", "订单号", "SKU", "销售利润", "发货量", "发货时间", "创建时间"};
            var list = new List<string>();
            foreach (var item in dataList.Data.Data)
            {
                list.Add($"{item.financecategoryname}|{item.platform_type}|{item.origin_order_id}|{item.bailun_sku}|{item.sale_profit}|{item.quantity_shipped}|{item.shipping_time}|{item.createtime}");
            }
            var guid = Guid.NewGuid().ToString();
            var filepath = _hostingEnvironment.WebRootPath + "\\Files\\Report\\" + DateTime.Now.ToString("yyyy-MM-dd") + "\\";
            ToCSV(list, colNames, guid, filepath);
            var ms = new System.IO.MemoryStream();
            using (var f = new System.IO.FileStream(filepath + guid + ".csv", System.IO.FileMode.Open))
            {
                f.CopyTo(ms);
            }
            ms.Position = 0;
            return File(ms, "text/csv", $"{input.Month}_月销售费用合计》费用利润表明细.csv");
        }
      /// <summary>
      /// 同步月利润销售报告
      /// </summary>
      /// <param name="time"></param>
      /// <returns></returns>
        [HttpGet("syncMonthSalesProfit")]
        public bool SyncMonthSalesProfit(string time)
        {
           return new FinanceService().SyncMonthSalesProfit(time);
            //var sql = "select * from dc_month_sales_profit_orderdetail limit 1";
            //var data = Dapper.SimpleCRUD.Query<dc_month_sales_profit_orderdetail>(sql, null, Common.GlobalConfig.ConnectionString).FirstOrDefault();
            //return data;
        }
        /// <summary>
        /// 同步月销售利润报告销售额明细汇总
        /// </summary>
        /// <param name="date"></param>
        /// <returns></returns>
        [HttpGet("syncMonthSalesProfiOrderDetail")]
        public bool SyncMonthSalesProfiOrderDetail(string date)
            =>new FinanceService().SyncMonthSalesProfiOrderDetail(date);
        [HttpGet("testRedis")]
        public object TestRedis()
        {
            try
            {
                Bailun.DC.Common.RedisHelper.Add("test0819", "test add redis");
                return Bailun.DC.Common.RedisHelper.GetString("test0819");
            }
            catch (Exception e)
            {
                return e;
            }
        }
        #endregion
    }
}
