﻿using Bailun.DC.Models.Component.DTO;
using Bailun.DC.Models.Component.Entity;
using Dapper;
using MySql.Data.MySqlClient;
using Newtonsoft.Json;
using System;
using System.Linq;

namespace Bailun.DC.Services.Component
{
    /// <summary>
    /// 表单组件
    /// </summary>
    public class FormService : BaseService
    {
        /// <summary>
        /// 获取表单组件
        /// </summary>
        /// <param name="code">编码</param>
        /// <returns></returns>
        public FormDTO Get(string code)
        {
            FormDTO dto = default(FormDTO);

            if (!string.IsNullOrWhiteSpace(code))
            {
                dc_component_form entity = default(dc_component_form);
                using (var db = DB)
                {
                    DynamicParameters sqlparam = new DynamicParameters();
                    sqlparam.Add("code", code);
                    entity = db.QueryFirstOrDefault<dc_component_form>("select * from dc_component_form where is_delete = 0 and code = @code", sqlparam);
                }
                if (entity != null)
                {
                    dto = new FormDTO
                    {
                        Id = entity.id,
                        Code = entity.code,
                        Name = entity.name,
                        Width = entity.width,
                        LabelWidth = entity.label_width,
                        IsShowBorder = entity.is_show_border,
                        IsShowSaveButton = entity.is_show_save_button,
                        SaveButtonText = entity.save_button_text,
                        IsShowCancelButton = entity.is_show_cancel_button,
                        CancelButtonText = entity.cancel_button_text,
                        Justify = entity.justify,
                        IsValidate = entity.is_validate,
                        IsDrag = entity.is_drag,
                        SaveScript = entity.save_script,
                        javascript = entity.javascript,
                        ListFormItem = !string.IsNullOrWhiteSpace(entity.form_item) ? DeserializeCollection<dynamic>(entity.form_item) : null,
                        ListControl = !string.IsNullOrWhiteSpace(entity.controls) ? DeserializeCollection<dynamic>(entity.controls) : null,
                        ListJavascriptSrc = !string.IsNullOrWhiteSpace(entity.javascript_src) ? DeserializeCollection<string>(entity.javascript_src) : null,
                    };
                }
            }

            return dto;
        }

        /// <summary>
        /// 保存报表组件
        /// </summary>
        /// <param name="dto">报表组件对象</param>
        /// <returns>保存结果</returns>
        public ResultDTO Save(FormDTO dto)
        {
            ResultDTO result = this.BeforeSave(dto);

            if (result.Result)
            {
                dc_component_form entity = new dc_component_form()
                {
                    id = dto.Id,
                    code = dto.Code,
                    name = dto.Name,
                    width = dto.Width,
                    label_width = dto.LabelWidth,
                    is_show_border = dto.IsShowBorder,
                    is_show_save_button = dto.IsShowSaveButton,
                    save_button_text = dto.SaveButtonText,
                    is_show_cancel_button = dto.IsShowCancelButton,
                    cancel_button_text = dto.CancelButtonText,
                    justify = dto.Justify,
                    is_validate = dto.IsValidate,
                    is_drag = dto.IsDrag,
                    save_script = dto.SaveScript,
                    javascript = dto.javascript,
                    gmt_create = DateTime.Now,
                    gmt_modified = DateTime.Now,
                    form_item = dto.ListFormItem?.Count() > 0 ? JsonConvert.SerializeObject(dto.ListFormItem) : null,
                    controls = dto.ListControl?.Count() > 0 ? JsonConvert.SerializeObject(dto.ListControl) : null,
                    javascript_src = dto.ListJavascriptSrc?.Where(l => !string.IsNullOrWhiteSpace(l)).Count() > 0 ? JsonConvert.SerializeObject(dto.ListJavascriptSrc) : null
                };
                using (var db = DB)
                {
                    //开启事务
                    MySqlTransaction transaction = db.BeginTransaction();
                    try
                    {
                        int? resultCount = entity.id > 0 ? db.Update(entity) : db.Insert(entity);
                        if (resultCount == 0)
                        {
                            throw new Exception("保存失败");
                        }
                        else
                        {
                            //提交事务
                            transaction.Commit();
                            result.Message = "保存成功";
                        }
                    }
                    catch (Exception ex)
                    {
                        //回滚事务
                        transaction.Rollback();
                        result.Message = ex.Message;
                        result.Result = false;
                    }
                }
            }

            return result;
        }

        /// <summary>
        /// 保存报表组件
        /// </summary>
        /// <param name="id">ID</param>
        /// <param name="is_delete">是否删除</param>
        /// <returns>保存结果</returns>
        public ResultDTO Save(int[] id, bool is_delete)
        {
            ResultDTO result = this.BeforeSave(id);

            if (result.Result)
            {
                string sql = "update dc_component_form set is_delete = @is_delete, gmt_modified = @gmt_modified where id in @id";
                using (var db = DB)
                {
                    //开启事务
                    MySqlTransaction transaction = db.BeginTransaction();
                    try
                    {
                        int resultCount = db.Execute(sql, new { id = id, is_delete = is_delete, gmt_modified = DateTime.Now });
                        if (resultCount == 0)
                        {
                            throw new Exception("保存失败");
                        }
                        else
                        {
                            //提交事务
                            transaction.Commit();
                            result.Message = "保存成功";
                        }
                    }
                    catch (Exception ex)
                    {
                        //回滚事务
                        transaction.Rollback();
                        result.Message = ex.Message;
                        result.Result = false;
                    }
                }
            }

            return result;
        }

        /// <summary>
        /// 保存报表组件
        /// </summary>
        /// <param name="dto">报表组件对象</param>
        /// <returns>保存结果</returns>
        private ResultDTO BeforeSave(FormDTO dto)
        {
            ResultDTO result = new ResultDTO();

            if (dto == null)
            {
                result.Message = "参数有误";
            }
            else if (string.IsNullOrWhiteSpace(dto.Code))
            {
                result.Message = "编码不能为空";
            }
            else if (string.IsNullOrWhiteSpace(dto.Name))
            {
                result.Message = "名称不能为空";
            }
            else if (DB.QueryFirstOrDefault<int>($"select count(id) from dc_component_form where code = @code {(dto.Id > 0 ? "and id <> @id" : string.Empty)}", new { id = dto.Id, code = dto.Code }) > 0)
            {
                result.Message = "编码重复";
            }
            else
            {
                result.Result = true;
            }

            return result;
        }

        /// <summary>
        /// 保存报表组件
        /// </summary>
        /// <param name="id">ID</param>
        /// <returns>保存结果</returns>
        private ResultDTO BeforeSave(int[] id)
        {
            ResultDTO result = new ResultDTO();
            if (id == null || id.Length == 0 || id.Contains(0))
            {
                result.Message = "ID不能为空";
            }
            else
            {
                result.Result = true;
            }
            return result;
        }

    }
}
