(function () {
  const defaults = {
    url: false, //Image path; it's best to pass the path in from the entry point
    alt: '', //Image alt
    callback: false, //callback function
    target: false, //A selector or DOM element that should be used as the parent container for the zoomed image.
    on: 'mouseover', // other options: click, toggle
    touch: false, // enables a touch fallback
    magnify: 1, // Based on the magnification of the original container
    mobileShow: true,
    objectFit: 'cover'
  };
  let setOpacity = function (element, v) {
    if (window.ActiveXObject) {
      element.style.filter = 'alpha(opacity=' + v * 100 + ')'; // IE
    } else {
      element.style.opacity = v; // Gecko/Opera
    }
  };
  //Refer to the offset() source in jquery
  let offSet = function (elem) {
    const rect = elem.getBoundingClientRect();
    const win = elem.ownerDocument.defaultView;
    return {
      top: rect.top + win.pageYOffset,
      left: rect.left + win.pageXOffset
    };
  };
  function zoomInit(target, source, img, settings) {
    let targetWidth = target.offsetWidth,
      targetHeight = target.offsetHeight,
      sourceHeight,
      sourceWidth,
      xRatio,
      yRatio,
      offset,
      position = target.getAttribute('position');
    target.style.position = /(absolute|fixed)/.test(position) ? position : 'relative';
    target.style.overflow = 'hidden';
    return {
      init: function () {
        if (source === target) {
          sourceWidth = targetWidth;
          sourceHeight = targetHeight;
        } else {
          sourceWidth = source.offsetWidth;
          sourceHeight = source.offsetHeight;
        }
        img.style.width = img.style.height = '';
        img.classList.add('zoomImg');
        img.style.cssText = 'position: absolute;top: 0;left: 0;opacity: 0; border: none;max-width: none;max-height: none;transition: opacity 200ms;';
        img.style.objectFit = settings.objectFit;
        const imgHeight = sourceHeight * settings.magnify;
        const imgWidth = sourceWidth * settings.magnify;
        img.style.height = imgHeight + 'px';
        img.style.width = imgWidth + 'px';
        target.append(img);
        xRatio = (imgWidth - targetWidth) / sourceWidth;
        yRatio = (imgHeight - targetHeight) / sourceHeight;
        offset = offSet(source);
      },
      move: function (e) {
        let left = e.pageX - offset.left,
          top = e.pageY - offset.top;
        top = Math.max(Math.min(top, sourceHeight), 0);
        left = Math.max(Math.min(left, sourceWidth), 0);
        img.style.left = left * -xRatio + 'px';
        img.style.top = top * -yRatio + 'px';
      }
    };
  }
  window.zoomImg = (_this, options) => {
    const largeImg = _this.querySelector('.zoomImg');
    if (largeImg) return;
    const img = document.createElement('img'),
      elementImg = _this.querySelector('img')
    let touched = false;
    let settings = Object.assign({}, defaults, options),
      //target will display the zoomed image
      target = settings.target || _this;

    // If a url wasn't specified, look for an image element.
    if (!settings.url) {
      if (elementImg) {
        settings.url = elementImg.getAttribute('data-img-zoom') || elementImg.src;
      }
    }
    // If a alt wasn't specified, look for an image element.
    if (!settings.alt) {
      if (elementImg) {
        settings.alt = elementImg.getAttribute('alt') || elementImg.alt || '';
      }
    }

    img.alt = settings.alt;
    img.src = settings.url;
    img.onload = () => {
      //avoid reloading
      let zoom = zoomInit(target, _this, img, settings);
      const start = (e) => {
        zoom.move(e);
        setOpacity(img, 1);
      };
      const stop = () => {
        setOpacity(img, 0);
      };
      if (settings.on === 'mouseover' && settings.mobileShow) {
        zoom.init();
        _this.addEventListener('mouseenter', start);
        _this.addEventListener('mouseleave', stop);
        _this.addEventListener('mousemove', start);
      }
      // Touch fallback
      if (settings.touch) {
        _this.addEventListener('touchstart', function (e) {
          e.preventDefault();
          if (touched) {
            touched = false;
            stop();
          } else {
            touched = true;
            start(e.touches[0] || e.changedTouches[0]);
          }
        });
        _this.addEventListener('touchmove', function (e) {
          e.preventDefault();
          zoom.move(e.touches[0] || e.changedTouches[0]);
        });
        _this.addEventListener('touchend', function (e) {
          e.preventDefault();
          if (touched) {
            touched = false;
            stop();
          }
        });
      }
      if (Object.prototype.toString.call(settings.callback) === '[object Function]') {
        settings.callback.call(img);
      }
      if (!settings.mobileShow && largeImg) {
        _this.removeChild(largeImg);
        return;
      }
    };
  };
})();