﻿using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Abp.Application.Services;
using AutoMapper;
using Bailun.Discuz.Application.Account.Dto;
using Bailun.Discuz.Domain.Account;
using Bailun.Discuz.Domain.Account.IRepositories;
using Bailun.Discuz.Domain.ExternalApi;
using Bailun.ServiceFabric;
using Microsoft.EntityFrameworkCore;

namespace Bailun.Discuz.Application.Account
{
    public class AccountPublishConfigService : ApplicationService, IAccountPublishConfigService
    {
        private readonly ILazadaAccountPublishConfigRepository _lazadaAccountPublishConfigRepository;
        private readonly IMapper _mapper;
        private IPamsApi _pamsApi;

        public AccountPublishConfigService(ILazadaAccountPublishConfigRepository lazadaAccountPublishConfigRepository,
            IMapper mapper,
            IPamsApi pamsApi)
        {
            _lazadaAccountPublishConfigRepository = lazadaAccountPublishConfigRepository;
            _mapper = mapper;
            _pamsApi = pamsApi;
        }

        public async Task DeleteAccountPublishConfig(long configId)
        {
            await _lazadaAccountPublishConfigRepository.DeleteAsync(configId);
        }

        public async Task<IEnumerable<string>> GetAccountNames()
        {
            var result = await _pamsApi.GetPlatformAccount();

            var accounts = _mapper.Map<List<ThinAccountDto>>(result.Data);
            return accounts.Select(m => m.Account).Distinct().ToList();
        }

        public async Task<AccountPublishConfigDto> GetAccountPublishConfig(long configId)
        {
            var config = await _lazadaAccountPublishConfigRepository.Query().AsNoTracking()
                .FirstOrDefaultAsync(m => m.Id == configId);
            var dto = _mapper.Map<AccountPublishConfigDto>(config);
            return dto;
        }

        public async Task<PageResult<List<AccountPublishConfigPageDto>>> GetAccountPublishConfigPages(
            GetAccountPublishConfigPagesInput input)
        {
            var watch = TimeHelper.TimerStart();
            var query = BuildAccountPublishConfigQuery(input);
            var data = await query.Skip((input.PageIndex - 1) * input.PageNumber).Take(input.PageNumber).ToListAsync();
            var list = _mapper.Map<List<AccountPublishConfigPageDto>>(data);
            var total = await query.CountAsync();
            return new PageResult<List<AccountPublishConfigPageDto>>().ToPageResult(input.PageIndex, total, list,
                TimeHelper.TimerEnd(watch));
        }

        public async Task SaveAccountPublishConfig(AccountPublishConfigDto input)
        {
            var config = _mapper.Map<LazadaAccountPublishConfig>(input);
            config.PublishableCategoryJson = input.PublishableCategories.ToJson();
            config.UnpublishableCategoryJson = input.UnpublishableCategories.ToJson();
            await _lazadaAccountPublishConfigRepository.InsertOrUpdateAsync(config);
        }

        private IQueryable<LazadaAccountPublishConfig> BuildAccountPublishConfigQuery(
            GetAccountPublishConfigPagesInput input)
        {
            var query = _lazadaAccountPublishConfigRepository.Query().AsNoTracking();
            if (!input.AccountName.IsNullOrEmpty()) query = query.Where(m => m.AccountName.Contains(input.AccountName));

            if (input.AccountType.HasValue) query = query.Where(m => m.AccountType == input.AccountType.Value);

            return query;
        }
    }
}